package com.ovopark.iohub.sdk.model.proto.internal;

import com.ovopark.iohub.sdk.model.proto.CellFile;
import com.ovopark.iohub.sdk.model.proto.LimitLogger;
import com.ovopark.iohub.sdk.model.proto.OutStore;
import com.ovopark.iohub.sdk.model.proto.Segment;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Model;
import com.ovopark.kernel.shared.vfile.FileIO;
import com.ovopark.kernel.shared.vfile.SimpleFileIO;
import lombok.Data;
import lombok.Getter;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CancellationException;

import static com.ovopark.kernel.shared.Util.*;

/**
 *  FILE PATH , DATA VIA NFS
 */
public class NFSOutStore implements OutStore, LimitLogger.LimitLoggerSetter {

    @Getter
    private String fileName;

    @Getter
    private final String path;

    final private List<SegmentFileDescriptor> segmentFileDescriptorList;

    private boolean committed;

    final private List<SegmentFileWriter> segmentFileWriterList=new ArrayList<>();

    final private FileIO fileIO;

    @Getter
    final private String fileIOUrl;

    @Getter
    final private Map<String,Group> groupMap=new HashMap<>();

    final private RowTransLogConfig rowTransLogConfig;

    final Stat stat=new Stat();

    final RuntimeStat runtimeStat=new RuntimeStatImpl();

    @Getter
    final private Feature feature=new Feature();

    LimitLogger limitLogger;

    public NFSOutStore(String path,int size){
        this(path,size,null);
    }


    public NFSOutStore(String path,int size,RowTransLogConfig rowTransLogConfig) {
        this.path = path;
        File file=new File(this.path);
        if (!file.exists()) {
            file.mkdirs();
        }
        segmentFileDescriptorList=new ArrayList<>(size);
        this.rowTransLogConfig=rowTransLogConfig==null?new RowTransLogConfig():rowTransLogConfig;
        SimpleFileIO.Conf conf=SimpleFileIO.Conf.defaultConf();
        conf.setCompressed(false);
        this.fileIOUrl=this.path+"/"+uniqueFirstPart()+"-file.bf";
        this.fileIO=new SimpleFileIO("nfs",fileIOUrl,conf,false);
    }

    @Override
    public SegmentWriter createSegment(String name) {
        return createSegment(name,1);
    }

    @Override
    public SegmentWriter createSegment(String name,int rowSize) {
        return createSegment(name,rowSize,"default");
    }

    @Override
    public SegmentWriter createSegment(String name, String group) {
        return createSegment(name,1,group);
    }

    @Override
    public SegmentWriter createSegment(String name, int rowSize, String group) {
        check();
        if (isEmpty(group)) {
            throw new IllegalArgumentException("group is empty");
        }
        if (segmentFileWriterList.size()>256) {
            throw new IllegalArgumentException("exceed max segment size 256 ");
        }

        SegmentFileWriter segmentFileWriter = new SegmentFileWriter(name, group, rowSize);
        segmentFileWriterList.add(segmentFileWriter);
        groupMap.putIfAbsent(group,new Group());

        runtimeStat.segmentAdd(group+":"+name);
        runtimeStat.segmentCountAdd(1);

        limitLogger.log("create segment "+group+":"+name);
        return segmentFileWriter;
    }

    @Override
    public synchronized SegmentWriter cloneSegmentExcludeRowData(String name, String target) {
        SegmentFileWriter sf=null;
        for (SegmentFileWriter segmentFileWriter : segmentFileWriterList) {
            if (segmentFileWriter.name.equals(name)) {
                sf=segmentFileWriter;
                break;
            }
        }

        if (sf==null) {
            throw new IllegalArgumentException("cannot find segment: "+name);
        }



        SegmentFileWriter segmentFileWriter = new SegmentFileWriter(target, sf.group, sf.rowCount);
        Segment.Header targetHeader = segmentFileWriter.header();

        sf.header.scanUp2Down(new Segment.Header.Scan() {
            @Override
            public void scan(Segment.Header.Cell cell) {
                String p = cell.path();
                if (p.contains(".")) {
                    targetHeader.upset(p.substring(0,p.lastIndexOf(".")),p.substring(p.lastIndexOf(".")),null);
                }
                else {
                    targetHeader.upset(null,p,null);
                }
                HeaderCellImpl c = (HeaderCellImpl) cell;
                // feature
                HeaderCellImpl targetCell = ((HeaderCellImpl) targetHeader.cell(p));

                targetCell.setWidth(c.getWidth());
                targetCell.setI18Key(c.getI18Key());
                targetCell.setName(c.getName());
                targetCell.setTip(c.getTip());

                targetCell.setVirtual(c.isVirtual());

                Segment.HeaderFeature feature1 = c.feature();
                if (feature1 != null) {
                    Segment.HeaderFeature headerFeature = feature1.clone();
                    targetCell.setHeaderFeature(headerFeature);
                }
            }
        });
        segmentFileWriter.title().title(sf.title().title());
        return segmentFileWriter;
    }

    @Override
    public void commit() {
        committed=true;
    }

    @Override
    public void close() {
        Exception exception=null;
        for (SegmentFileWriter segmentFileWriter : segmentFileWriterList) {
            try {
                segmentFileWriter.close();
            }catch (Exception e){
                //
                exception=e;
            }
        }

        try {
            fileIO.close();
        } catch (Exception e) {
            //
            exception=e;
        }

        if (exception!=null) {
            throw convert2RuntimeException(exception);
        }
    }

    private void check(){
        if (committed) {
            throw new UnsupportedOperationException("all committed???");
        }
    }

    @Override
    public Segment segment(Segment.SD sd) {
        for (SegmentFileDescriptor segmentFileDescriptor : segmentFileDescriptorList) {
            if (segmentFileDescriptor.getName().equals(sd.getName()) && segmentFileDescriptor.getGroup().equals(sd.getGroup())) {
                return new Segment() {
                    @Override
                    public List<Map<String, Object>> rowList() {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public int size() {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public void append(Map<String, Object> row) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public String name() {
                        return segmentFileDescriptor.getName();
                    }

                    @Override
                    public Map<String, Object> meta() {
                        return segmentFileDescriptor.getMeta();
                    }

                    @Override
                    public Title title() {
                        return segmentFileDescriptor.title;
                    }

                    @Override
                    public Header header() {
                        return segmentFileDescriptor.header;
                    }

                    @Override
                    public String group() {
                        return segmentFileDescriptor.getGroup();
                    }

                    @Override
                    public void commit() {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public Feature feature() {
                        return segmentFileDescriptor.getFeature();
                    }

                    @Override
                    public List<Map<String,Object>> capture(int n) {
                        return new ArrayList<>(0);
                    }
                };
            }
        }
        return null;
    }

    @Override
    public List<Segment.SD> sdList() {
        List<Segment.SD> sdList=new ArrayList<>(segmentFileDescriptorList.size());
        for (SegmentFileDescriptor segmentFileDescriptor : segmentFileDescriptorList) {
            Segment.SD sd=new Segment.SD();
            sd.setName(segmentFileDescriptor.getName());
            sd.setMemory(false);
            sd.setUrl(segmentFileDescriptor.getPath());
            sd.setRowCount(segmentFileDescriptor.getRowCount());
            sd.setGroup(segmentFileDescriptor.getGroup());
            sdList.add(sd);
        }
        return sdList;
    }

    @Override
    public CellFile file(byte[] data, String name, String type) {
        Map<String,Object> meta=new HashMap<>();
        meta.put("name",name);
        meta.put("type",type);
        FileIO.FilePutResult filePutResult = fileIO.put(meta, data);
        CellFileImpl cellFile=new CellFileImpl();
        cellFile.setName(name);
        cellFile.setType(type);
        cellFile.setBase64(false);
        cellFile.setUri(filePutResult.key());
        return cellFile;
    }

    @Override
    public LimitLogger limitLogger() {
        return limitLogger;
    }

    @Override
    public Map<String, List<Map<String,Object>>> capture(int n) {
        Map<String, List<Map<String,Object>>> listMap=new LinkedHashMap<>();
        int size = segmentFileWriterList.size();
        for (int i = 0; i < size; i++) {
            SegmentFileWriter segmentFileWriter = segmentFileWriterList.get(i);
            List<Map<String,Object>> list = segmentFileWriter.capture(n);
            listMap.put(segmentFileWriter.name,list);
        }
        return listMap;
    }

    @Override
    public void fileName(String fileName) {
        this.fileName=fileName;
    }

    @Override
    public GroupOperation group(String group) {
        return new GroupOperation() {
            @Override
            public void fileName(String fileName) {
                groupMap.get(group).setFileName(fileName);
            }
        };
    }

    @Override
    public FeatureOperation feature() {
        return new FeatureOperation() {
            @Override
            public void supportRangeMerge(boolean rangeMerge) {
                feature.setRangeMerge(rangeMerge);
            }

            @Override
            public void supportBorder(boolean border) {
                feature.setBorder(border);
            }

            @Override
            public void supportParallelism(boolean parallelism) {
                feature.setParallelism(parallelism);
            }
        };
    }

    @Override
    public Stat stat() {
        return stat;
    }

    @Override
    public RuntimeStat runtimeStat() {
        return runtimeStat;
    }

    @Override
    public void setLimitLogger(LimitLogger limitLogger) {
        this.limitLogger=limitLogger;
    }

    @Data
    static class SegmentFileDescriptor implements Model{

        private String path;

        private String name;

        private Segment.Title title;

        private Segment.Header header;

        private Map<String, Object> meta;

        private String group;

        private int rowCount;

        private Segment.Feature feature;
    }

    class SegmentFileWriter implements SegmentWriter{

        static final int maxCount = 104_8000;

        final private String name;

        final private String group;

        final private RowTransLog rowTransLog;

        final String filePath;

        final private Segment.Header header=new HeaderImpl();

        final private Segment.Title title=new TitleImpl();

        private Map<String, Object> meta;

        private int rowCount;

        private boolean committed;

        private boolean closed;

        final private SegmentStat segmentStat=new SegmentStat();

        final private Segment.Feature segmentFeature=new Segment.Feature();

        public SegmentFileWriter(String name,String group,int size) {
            this.name=name;
            this.group=group;
//            RowTransLogConfig rowTransLogConfig =new RowTransLogConfig();
//            rowTransLogConfig.setWalBufferSizeMb(10);
//            rowTransLogConfig.setWalDiskSizeMb(1024);
//            rowTransLogConfig.setWalIntervalSec(1);
            this.filePath=path+"/"+md5(group+":"+name);
            rowTransLog =new MixedRowTransLogImpl("segment",filePath, rowTransLogConfig,runtimeStat);
        }

        @Override
        public synchronized void append(Map<String, Object> data) {
            if (runtimeStat.cancelled()) {
                throw new CancellationException("task is cancelled");
            }
            rowTransLog.append(data);
            rowCount++;
            runtimeStat.rowCountAdd(1);
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
            runtimeStat.byteSize(rowTransLog.byteSize());
            if (rowCount> maxCount) {
                throw new IllegalArgumentException("exceed max row count: "+maxCount+" / 1048576 ???");
            }
        }

        @Override
        public synchronized void append(List<Map<String, Object>> data) {
            if (runtimeStat.cancelled()) {
                throw new CancellationException("task is cancelled");
            }
            rowTransLog.append(data);
            rowCount+=data.size();
            runtimeStat.rowCountAdd(data.size());
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
            runtimeStat.byteSize(rowTransLog.byteSize());
            if (rowCount>maxCount) {
                throw new IllegalArgumentException("exceed max row count: "+maxCount+" / 1048576 ???");
            }
        }

        @Override
        public synchronized void append(Model model) {
            append(JSONAccessor.impl().read(JSONAccessor.impl().format(model)));
        }

        @Override
        public void meta(Map<String, Object> meta) {
            this.meta=meta;
        }

        @Override
        public synchronized void commit() {

            if (committed) {
                return;
            }

            rowTransLog.fsync();

            long l = rowTransLog.byteSize();
            runtimeStat.byteSize(l);

            for (SegmentFileDescriptor segmentFileDescriptor : segmentFileDescriptorList) {
                if (segmentFileDescriptor.getPath().equalsIgnoreCase(filePath)) {
                    throw new IllegalArgumentException("file is duplicate: "+filePath);
                }
            }

            SegmentFileDescriptor segmentFileDescriptor= new SegmentFileDescriptor();
            segmentFileDescriptor.setPath(filePath);
            segmentFileDescriptor.setName(name);
            segmentFileDescriptor.setTitle(title);
            segmentFileDescriptor.setHeader(header);
            segmentFileDescriptor.setMeta(meta);
            segmentFileDescriptor.setGroup(group);
            segmentFileDescriptor.setRowCount(rowCount);
            segmentFileDescriptor.setFeature(segmentFeature);

            segmentFileDescriptorList.add(segmentFileDescriptor);
            header().commit();

            segmentStat.setRowCount(rowCount);
            if (stat.getSegmentStatList()==null) {
                stat.setSegmentStatList(new ArrayList<>());
            }
            stat.getSegmentStatList().add(segmentStat);

            this.committed=true;

            try {
                log.info("committed , close segment: "+name);
                rowTransLog.close();
                closed=true;
            } catch (IOException e) {
                throw convert2RuntimeException(e);
            }

        }

        @Override
        public Segment.Header header() {
            return header;
        }

        @Override
        public Segment.Title title() {
            return title;
        }

        @Override
        public String group() {
            return group;
        }

        void close() throws Exception{
            if (closed) {
                return;
            }
            rowTransLog.close();
        }

        @Override
        public FeatureOperation feature() {
            return new FeatureOperation() {
                @Override
                public void supportRangeMerge(boolean rangeMerge) {
                    segmentFeature.setRangeMerge(rangeMerge);
                }

                @Override
                public void supportBorder(boolean border) {
                    segmentFeature.setBorder(border);
                }
            };
        }

        @Override
        public List<Map<String,Object>> capture(int n) {
            return rowTransLog.capture(n);
        }
    }

}
