package com.ovopark.iohub.sdk.client;

import com.ovopark.iohub.sdk.client.outstream.ExportEndpoint;
import com.ovopark.iohub.sdk.model.ClientNodeRegisterRequest;
import com.ovopark.iohub.sdk.model.ClientNodeRegisterResponse;
import com.ovopark.iohub.sdk.model.proto.IO;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.CommandLineRunner;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

import static com.ovopark.kernel.shared.Util.*;

@Slf4j
@JobClientActive
@Component
public class ClientNodeRegisterImpl implements ClientNodeRegister, CommandLineRunner {

    @Autowired
    private ClientNodeProvider clientNodeProvider;

    final static ScheduledExecutorService scheduledExecutorService= Executors.newScheduledThreadPool(1);

    @Autowired
    ControlTransport controlTransport;

    @Autowired
    private IOHubClientConfig ioHubClientConfig;

    final boolean verbose= Config.ConfigPriority.option().getBoolean("iohub.client.node.verbose",false);

    final private AtomicLong count=new AtomicLong(0);

    final static LocalDateTime startupTime=LocalDateTime.now();

    @Override
    public void register() {
        schedule(scheduledExecutorService, () -> {

            ClientNode clientNode = clientNodeProvider.clientNode();
            ClientNodeRegisterRequest clientNodeRegisterRequest=new ClientNodeRegisterRequest();
            clientNodeRegisterRequest.setStartupTime(startupTime);
            clientNodeRegisterRequest.setVersion(IO.version);
            clientNodeRegisterRequest.setMinVersion(IO.minVersion);
            clientNodeRegisterRequest.setApp(clientNode.app());
            clientNodeRegisterRequest.setNode(clientNode.node());
            clientNodeRegisterRequest.setIp(clientNode.ip());
            clientNodeRegisterRequest.setPort(clientNode.port());
            clientNodeRegisterRequest.setServletPath(clientNode.servletPath());
            clientNodeRegisterRequest.setTaskModelList(ExportEndpoint.JobTaskManager.getOrCreate().taskList());
            clientNodeRegisterRequest.setNfsPath(ioHubClientConfig.nfsPath());
            String nodeRootPath = ioHubClientConfig.nfsPath() + "/client-" + clientNode.node();
            clientNodeRegisterRequest.setNodeRootPath(nodeRootPath);

            final String nfsCheckContent=IOTtlCache.getOrCreate().putIfAbsentAndGet(ClientNode.UUID_STR + ":nfs-check"
                    , new Function<String,String>() {
                @Override
                public String apply(String k) {
                    String nfsPath = ioHubClientConfig.nfsPath();
                    String nfsCheckContent = uniqueFirstPart();
                    try {
                        File file = new File(nodeRootPath + "/nfs-check");
                        if (!file.getParentFile().exists()) {
                            file.getParentFile().mkdirs();
                        }
                        Util.writeAtomic(file, utf8(nfsCheckContent));
                    } catch (IOException e) {
                        return e.getMessage();
                    }
                    return nfsCheckContent;
                }
            },15,TimeUnit.SECONDS).value();

            if (count.getAndIncrement() % 30==0) {
                IOTtlCache.getOrCreate().putAndGet(ClientNode.UUID_STR + ":ts", new Function<String, String>() {
                    @Override
                    public String apply(String s) {
                        String tsTimeStr = formatTime(LocalDateTime.now());
                        try {
                            File file = new File(nodeRootPath + "/ts");
                            if (!file.getParentFile().exists()) {
                                file.getParentFile().mkdirs();
                            }
                            Util.writeAtomic(file, utf8(tsTimeStr));
                        } catch (IOException e) {
                            log.error(e.getMessage(),e);
                        }
                        return tsTimeStr;
                    }
                });
            }
            clientNodeRegisterRequest.setTsFilePath(nodeRootPath + "/ts");

            clientNodeRegisterRequest.setNfsCheckContent(nfsCheckContent);

            ClientNodeRegisterResponse clientNodeRegisterResponse = controlTransport.heartbeat(clientNodeRegisterRequest);
            log.debug(JSONAccessor.impl().format(clientNodeRegisterResponse));


        },5, TimeUnit.SECONDS, e->{
            log.debug(e.getMessage());
            return true;
        },()->true);
    }

    @Override
    public void run(String... args) throws Exception {
        register();
    }
}
