package com.ovopark.iohub.sdk.client;

import com.ovopark.iohub.sdk.model.*;
import com.ovopark.iohub.sdk.model.instream.*;
import com.ovopark.iohub.sdk.model.outstream.*;
import com.ovopark.iohub.sdk.model.test.ExportRenderResponse;
import com.ovopark.iohub.sdk.model.test.ExportStartResponse;
import com.ovopark.iohub.sdk.model.test.ReplyRenderResponse;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.module.shared.Session;
import com.ovopark.module.shared.spring.rbac.SessionImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.Map;

import static com.ovopark.kernel.shared.Util.isEmpty;
import static com.ovopark.kernel.shared.Util.isNotEmpty;

@Slf4j
public class Client2WorkRestClient implements Client2WorkTransport {


    private final RestTemplate restTemplate;

    private final AppNode appNode;

    public Client2WorkRestClient(AppNode appNode, RestTemplate restTemplate) {
        this.restTemplate=restTemplate;
        this.appNode=appNode;
    }

    @Override
    public ClientNodeTaskRegisterResponse exportHeartbeat(ClientNodeTaskRegisterRequest clientNodeTaskRegisterRequest) {
        return rpc(clientNodeTaskRegisterRequest, "/jobClient/task/export/heartbeat",ClientNodeTaskRegisterResponse.class);
    }

    @Override
    public TaskLogResponse log(TaskLogRequest taskLogRequest) {
        return rpc(taskLogRequest, "/jobClient/task/log",TaskLogResponse.class);
    }

    @Override
    public ExportPushStartResponse start(ExportPushStartRequest exportPushStartRequest) {
        return rpc(exportPushStartRequest, "/jobClient/task/export/start", ExportPushStartResponse.class);
    }

    @Override
    public CreateExportRecordResponse createExportRecord(CreateExportRecordRequest createExportRecordRequest) {
        return rpc(createExportRecordRequest, "/jobClient/task/export/createExportRecord",CreateExportRecordResponse.class);
    }

    @Override
    public ExportPushDataResponse push(ExportPushDataRequest exportPushDataRequest) {
        return rpc(exportPushDataRequest, "/jobClient/task/export/push", ExportPushDataResponse.class);
    }

    @Override
    public ExportPushCommitResponse commit(ExportPushCommitRequest exportPushCommitRequest) {
        return rpc(exportPushCommitRequest, "/jobClient/task/export/commit", ExportPushCommitResponse.class);
    }

    @Override
    public ClientNodeTaskRegisterResponse importHeartbeat(ClientNodeTaskRegisterRequest clientNodeTaskRegisterRequest) {
        return rpc(clientNodeTaskRegisterRequest, "/jobClient/task/import/heartbeat",ClientNodeTaskRegisterResponse.class);
    }

    @Override
    public ImportReplyResponse importReply(ImportReplyRequest importReplyRequest) {
        return rpc(importReplyRequest, "/jobClient/task/import/reply", ImportReplyResponse.class);
    }

    @Override
    public ImportCommitResponse importCommit(ImportCommitRequest importCommitRequest) {
        return rpc(importCommitRequest, "/jobClient/task/import/commit", ImportCommitResponse.class);
    }

    @Override
    public ImportPushDataRequest parseOnlyForTest(String file, String uri) {
        HashMap<String, Object> map = new HashMap<>();
        map.put("uri",uri);
        return restWithFile(map,file,"/jobClient/task/import/test/parse", ImportPushDataRequest.class,null);
    }

    @Override
    public ReplyRenderResponse renderReplyOnlyForTest(ImportReplyRequest importReplyRequest) {
        return rpc(importReplyRequest, "/jobClient/task/import/test/renderReply", ReplyRenderResponse.class);
    }

    @Override
    public ExportStartResponse startOnlyForTest(String uri) {
        HashMap<String, Object> map = new HashMap<>();
        map.put("uri",uri);
        return restWithFile(map,null,"/jobClient/task/export/test/start", ExportStartResponse.class,null);
    }

    @Override
    public ExportRenderResponse renderOnlyForTest(ExportPushCommitRequest exportPushCommitRequest) {
        return rpc(exportPushCommitRequest, "/jobClient/task/export/test/render", ExportRenderResponse.class);
    }

    @Override
    public ExportPushDataResponse pushOnlyForTest(ExportPushDataRequest exportPushDataRequest) {
        return rpc(exportPushDataRequest, "/jobClient/task/export/test/pushData", ExportPushDataResponse.class);
    }

    @Override
    public Session parseSessionOnlyForTest(String token) {
        SessionImpl session = restWithFile(new HashMap<>(), null, "/jobClient/task/export/test/parseSession", SessionImpl.class, token);
        return session;
    }

    @Override
    public ExportPushDataResponse pushJobRender(ExportPushDataRequest exportPushDataRequest) {
        return rpc(exportPushDataRequest, "/jobClient/task/export/pushJobRender", ExportPushDataResponse.class);
    }

    @Override
    public ExportPushCommitResponse commitJobRender(ExportPushCommitRequest exportPushCommitRequest) {
        return rpc(exportPushCommitRequest, "/jobClient/task/export/commitJobRender", ExportPushCommitResponse.class);
    }

    @Override
    public JobRenderHeartbeatResponse heartbeatJobRender(JobRenderHeartbeatRequest jobRenderHeartbeatRequest) {
        return rpc(jobRenderHeartbeatRequest, "/jobClient/task/export/heartbeatJobRender", JobRenderHeartbeatResponse.class);
    }

    private <T> T restWithFile(Map<String,Object> params, String file, String path, Class<T> clazz, String token){
        String url = "http://"+appNode.getIp() + ":" + appNode.getPort() ;
        url+= isEmpty(appNode.getServletPath())?"":(
                appNode.getServletPath().startsWith("/")?appNode.getServletPath()
                        : ("/"+appNode.getServletPath()));
        url+=path;
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type","multipart/form-data");
        if (isNotEmpty(token)) {
            headers.add("ovo-authorization",token);
        }
        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            multiValueMap.add(entry.getKey(),entry.getValue());
        }
        if (isNotEmpty(file)) {
            multiValueMap.add("file",new FileSystemResource(file));
        }

        HttpEntity<MultiValueMap<String, Object>> formEntity = new HttpEntity<>(multiValueMap, headers);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(url, formEntity, String.class);
            int statusCodeValue = responseEntity.getStatusCodeValue();
            if (statusCodeValue==200) {
                return JSONAccessor.impl().read(responseEntity.getBody(), clazz);
            }
            return null;
        } catch (Exception e) {
            log.error(e.getMessage());
            return null;
        }

    }

    private <T> T rpc(Object request, String path, Class<T> clazz) {
        String url = "http://"+appNode.getIp() + ":" + appNode.getPort() ;
        url+= isEmpty(appNode.getServletPath())?"":(
                appNode.getServletPath().startsWith("/")?appNode.getServletPath()
                        : ("/"+appNode.getServletPath()));
        url+=path;
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type","application/json");
        HttpEntity<String> formEntity = new HttpEntity<String>(JSONAccessor.impl().format(request), headers);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(url, formEntity, String.class);
            int statusCodeValue = responseEntity.getStatusCodeValue();
            if (statusCodeValue==200) {
                return JSONAccessor.impl().read(responseEntity.getBody(), clazz);
            }
            return null;
        } catch (Exception e) {
            log.error(e.getMessage());
            return null;
        }

    }

}
