package com.ovopark.iohub.sdk.client.instream;

import com.ovopark.iohub.sdk.client.*;
import com.ovopark.iohub.sdk.client.outstream.RequestParamBody;
import com.ovopark.iohub.sdk.model.AppNode;
import com.ovopark.iohub.sdk.model.JobHint;
import com.ovopark.iohub.sdk.model.JobMeta;
import com.ovopark.iohub.sdk.model.instream.*;
import com.ovopark.iohub.sdk.model.proto.*;
import com.ovopark.iohub.sdk.model.proto.internal.ImportPreDefConfImpl;
import com.ovopark.iohub.sdk.model.proto.internal.SegmentImpl;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import com.ovopark.kernel.shared.kv.KVEngine;
import com.ovopark.kernel.shared.stream.Stream;
import com.ovopark.module.shared.Session;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import static com.ovopark.kernel.shared.Util.*;

/**
 * @since 4.6
 */
@Slf4j
class ReadJobImpl implements ReadJob {

    final boolean renderJobTest= Config.ConfigPriority.option().getBoolean("iohub.client.renderJob.test",false);

    OutStore outStore;

    private final String uri;

    private final boolean nfs;

    final JobMeta jobMeta;

    final JobHint jobHint;

    final private IOHubClientConfig ioHubClientConfig;

    final private Client2ControlTransport client2ControlTransport;

    final ClientNode clientNode;

    final Long taskId;

    private static final ScheduledExecutorService jobHeartbeatExecutor = new ScheduledThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,8)
                    , Config.ConfigPriority.option().getInt("IOHUB_JOB_RENDER_HEARTBEAT_IO", 0))
            , newThreadFactory("iohub_job_render_heartbeat_io")
            , new ThreadPoolExecutor.AbortPolicy());

    private static final ExecutorService jobFutureExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,64)
                    , Config.ConfigPriority.option().getInt("IOHUB_JOB_RENDER_FUTURE_IO",0))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,64)
            , Config.ConfigPriority.option().getInt("IOHUB_JOB_RENDER_FUTURE_IO", 0))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(Integer.MAX_VALUE)
            , newThreadFactory("iohub_job_render_future_io")
            , new ThreadPoolExecutor.AbortPolicy());

    final static KVEngine.TtlFunc<String> readFutureTtl = KVEngine.newTtl(ReadJob.class.getName());

    static {
        readFutureTtl.subscribeTtl(t -> true, (getResult, l, l1) -> {
            String key = getResult.key();
            log.info("onExpired: "+key);
            ReadFutureImpl renderFuture = (ReadFutureImpl) getResult.value();
            renderFuture.setDataRowStreamAndNotify(null,new TimeoutException("seconds: "+TimeUnit.MILLISECONDS.toSeconds(l1)));
        });
    }

    final static RestTemplate restTemplate;
    static {
        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setReadTimeout(45_000);//ms
        factory.setConnectTimeout(15_000);//ms
        restTemplate = new RestTemplate(factory);
        restTemplate.getMessageConverters().add(0, new StringHttpMessageConverter(StandardCharsets.UTF_8));
    }

    public ReadJobImpl(Long taskId, String uri, boolean nfs, JobMeta jobMeta, JobHint jobHint
            , IOHubClientConfig ioHubClientConfig, Client2ControlTransport client2ControlTransport
            , ClientNode clientNode) {
        this.taskId=taskId;
        this.uri = uri;
        this.nfs = nfs;
        this.jobMeta = jobMeta;
        this.jobHint = jobHint;
        this.ioHubClientConfig = ioHubClientConfig;
        this.client2ControlTransport = client2ControlTransport;
        this.clientNode = clientNode;
    }


    @Override
    public ReadFuture readExcel(String file,ReadDef readDef) {
        return read(file, readDef);
    }

    @Override
    public ReadFuture read(String file,ReadDef readDef) {
        ReadJobAssignWorkRequest readJobAssignWorkRequest =new ReadJobAssignWorkRequest();
        readJobAssignWorkRequest.setTaskId(taskId);
        readJobAssignWorkRequest.setClientApp(clientNode.app());
        readJobAssignWorkRequest.setClientNode(clientNode.node());
        readJobAssignWorkRequest.setUri(uri);
        readJobAssignWorkRequest.setTest(renderJobTest);
        ReadJobAssignWorkResponse readJobAssignWorkResponse = client2ControlTransport.assignWorkReadJob(readJobAssignWorkRequest);
        log.info("readJobAssignWorkResponse: "+JSONAccessor.impl().format(readJobAssignWorkResponse));
        if (readJobAssignWorkResponse ==null || !readJobAssignWorkResponse.isSuccess() || readJobAssignWorkResponse.getWorker()==null) {
            throw new IllegalArgumentException("cannot assign work for read job: "+ JSONAccessor.impl().format(readJobAssignWorkResponse));
        }

        AppNode worker= readJobAssignWorkResponse.getWorker();
        Client2WorkTransport client2WorkTransport =new Client2WorkRestClient(worker,restTemplate);

        ReadJobParseRequest readJobParseRequest=new ReadJobParseRequest();
        readJobParseRequest.setUri(uri);
        if (readDef!=null) {
            JobMeta jm = jobMeta0(readDef);
            readJobParseRequest.setJobMeta(jm);
            readJobParseRequest.setJobHint(new JobHint());
        }
        else {
            readJobParseRequest.setJobMeta(jobMeta);
            readJobParseRequest.setJobHint(jobHint);
        }
        readJobParseRequest.setClientNode(clientNode.node());
        readJobParseRequest.setClientApp(clientNode.app());
        readJobParseRequest.setWorkNode(worker.getNode());
        readJobParseRequest.setTaskId(taskId);

        ReadJobParseResponse readJobParseResponse = client2WorkTransport.parseReadJob(file, readJobParseRequest);
        log.info("response readJobParseResponse: "+ JSONAccessor.impl().format(readJobParseResponse));
        if(readJobParseResponse ==null || !readJobParseResponse.isSuccess()){
            throw new IllegalStateException("cannot push file to worker:");
        }

        ReadFutureImpl readFuture=new ReadFutureImpl(file,readDef);
        ReadFutureTask readFutureTask=new ReadFutureTask(taskId,worker,clientNode,readFuture);
        jobHeartbeatExecutor.schedule(catchRunnable(readFutureTask),5,TimeUnit.SECONDS);
        readFutureTtl.putIfAbsentAndGet(String.valueOf(taskId), k->readFutureTask,60*15,TimeUnit.SECONDS);
        return readFuture;
    }

    private static JobMeta jobMeta0(ReadDef readDef) {
        JobMeta jm = new JobMeta();
        jm.setHeaderFromIndex(readDef.headerFromIndex());
        jm.setHeaderToIndex(readDef.headerToIndex());
        jm.setDataStartRowIndex(readDef.dataStartRowIndex());

        jm.setSheetIndexList(readDef.sheetIndexList());
        jm.setHeaderMaxColumnIndexList(readDef.headerMaxColumnIndexList());
        return jm;
    }

    static class ReadFutureImpl implements ReadFuture{

        private ReadListener readListener;

        JobInTaskFlow.DataRowStream<?> dataRowStream;

        private boolean cancelled;

        private boolean completed;

        private Throwable t;

        final ReentrantLock reentrantLock= new ReentrantLock();
        final Condition condition = reentrantLock.newCondition();

        final String file;

        final ReadDef readDef;

        public ReadFutureImpl(String file, ReadDef readDef) {
            this.file = file;
            this.readDef = readDef;
        }

        void setDataRowStreamAndNotify(JobInTaskFlow.DataRowStream<?> dataRowStream, Throwable t) {
            reentrantLock.lock();
            try {
                if (cancelled || completed || this.t!=null) {
                    return;
                }
                this.dataRowStream = dataRowStream;
                this.t=t;
                this.completed= t==null;

                condition.signalAll();
                if (readListener != null) {
                    // other thread , should not in heartbeat thread ,important!!!
                    jobFutureExecutor.execute(catchRunnable(() -> readListener.onRead(dataRowStream,t)));
                }
            }
            finally {
                reentrantLock.unlock();
            }
        }

        void cancelAndNotify() {
            reentrantLock.lock();
            try {
                if (cancelled || completed || this.t!=null) {
                    return;
                }
                this.cancelled = true;
                condition.signalAll();
                if (readListener != null) {
                    // other thread , should not in heartbeat thread ,important!!!
                    jobFutureExecutor.execute(catchRunnable(() -> readListener.onRead(dataRowStream,new CancellationException())));
                }
            }
            finally {
                reentrantLock.unlock();
            }
        }

        @Override
        public JobInTaskFlow.DataRowStream<?> get() throws InterruptedException{
            reentrantLock.lock();
            try {
                if (cancelled) {
                    throw new CancellationException();
                }
                if (t!=null) {
                    throw new RuntimeException(t);
                }
                if (completed) {
                    return dataRowStream;
                }
                try {
                    condition.await();
                    if (cancelled) {
                        throw new CancellationException();
                    }
                    if (t!=null) {
                        throw new RuntimeException(t);
                    }
                    if (completed) {
                        return dataRowStream;
                    }
                    throw new IllegalStateException("unreachable code , error???");
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw e;
                }
            }
            finally {
                reentrantLock.unlock();
            }
        }

        @Override
        public JobInTaskFlow.DataRowStream<?> get(long timeout, TimeUnit timeUnit) throws InterruptedException ,TimeoutException{
            reentrantLock.lock();
            try {
                if (cancelled) {
                    throw new CancellationException();
                }
                if (t!=null) {
                    throw new RuntimeException(t);
                }
                if (completed) {
                    return dataRowStream;
                }

                boolean f ;
                try {
                    f = condition.await(timeout, timeUnit);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw e;
                }
                if (f) {
                    // ok , receive signal
                    if (cancelled) {
                        throw new CancellationException();
                    }
                    if (t!=null) {
                        throw new RuntimeException(t);
                    }
                    if (completed) {
                        return dataRowStream;
                    }
                    throw new IllegalStateException("unreachable code , error???");
                } else {
                    throw new TimeoutException("timeout, ms: "+timeUnit.toSeconds(timeout));
                }
            }
            finally {
                reentrantLock.unlock();
            }
        }

        @Override
        public boolean cancelled() {
            return cancelled;
        }

        @Override
        public void cancel() {
            cancelAndNotify();
        }

        @Override
        public void setReadListener(ReadListener renderListener) {
            reentrantLock.lock();
            try {
                if (cancelled) {
                    // other thread , should not in heartbeat thread ,important!!!
                    jobFutureExecutor.execute(catchRunnable(() -> renderListener.onRead(dataRowStream,new CancellationException())));
                    return;
                }

                if (completed || this.t!=null) {
                    // other thread , should not in heartbeat thread ,important!!!
                    jobFutureExecutor.execute(catchRunnable(() -> renderListener.onRead(dataRowStream,t)));
                    return;
                }

                this.readListener =renderListener;
            }
            finally {
                reentrantLock.unlock();
            }
        }

    }

    static class ReadFutureTask implements Util.CatchRunnable{

        final Long taskId;

        final AppNode worker;

        final ClientNode clientNode;

        final ReadFutureImpl renderFuture;

        final long startMs=System.currentTimeMillis();

        final Client2WorkTransport client2WorkTransport;

        int failCount;

        public ReadFutureTask(Long taskId, AppNode worker, ClientNode clientNode, ReadFutureImpl renderFuture) {
            this.taskId = taskId;
            this.worker = worker;
            this.clientNode = clientNode;
            this.renderFuture = renderFuture;
            //
            SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
            factory.setReadTimeout(45_000);//ms
            factory.setConnectTimeout(15_000);//ms
            RestTemplate restTemplate = new RestTemplate(factory);
            restTemplate.getMessageConverters().add(0,new StringHttpMessageConverter(StandardCharsets.UTF_8));
            client2WorkTransport =new Client2WorkRestClient(worker,restTemplate);
        }

        @Override
        public void run() throws Exception {

            if (renderFuture.cancelled()) {
                log.info("cancel job render by client: "+taskId);
                return;
            }

            ReadJobHeartbeatRequest readJobHeartbeatRequest =new ReadJobHeartbeatRequest();
            readJobHeartbeatRequest.setTaskId(taskId);
            readJobHeartbeatRequest.setClientApp(clientNode.app());
            readJobHeartbeatRequest.setClientNode(clientNode.node());
            readJobHeartbeatRequest.setWorkNode(worker.getNode());

            ReadJobHeartbeatResponse renderJobHeartbeatResponse = client2WorkTransport.heartbeatReadJob(readJobHeartbeatRequest);
            if (renderJobHeartbeatResponse ==null || !renderJobHeartbeatResponse.isSuccess()) {
                log.info("response job render heartbeat: "+JSONAccessor.impl().format(renderJobHeartbeatResponse));
            }
            if(renderJobHeartbeatResponse ==null || !renderJobHeartbeatResponse.isSuccess()){
                if (failCount++>5) {
                    readFutureTtl.delete(String.valueOf(taskId));
                    renderFuture.setDataRowStreamAndNotify(null,new RuntimeException("render server error: "
                            +(renderJobHeartbeatResponse ==null?"": renderJobHeartbeatResponse.getDesc())
                    ));
                    return;
                }
                jobHeartbeatExecutor.schedule(catchRunnable(this),5,TimeUnit.SECONDS);
                return;
            }

            if (renderJobHeartbeatResponse.isReading()) {
                jobHeartbeatExecutor.schedule(catchRunnable(this),5,TimeUnit.SECONDS);
                return;
            }

            if (renderJobHeartbeatResponse.isReadCancelled()) {
                log.info("render job ( "+taskId+" ) was cancelled by work: "+JSONAccessor.impl().format(renderJobHeartbeatResponse));
                readFutureTtl.delete(String.valueOf(taskId));
                renderFuture.cancelAndNotify();
                return;
            }

            if (renderJobHeartbeatResponse.isReadError()) {
                readFutureTtl.delete(String.valueOf(taskId));
                renderFuture.setDataRowStreamAndNotify(null,new RuntimeException(renderJobHeartbeatResponse.getReadErrorDesc()));
                return;
            }

            if (renderJobHeartbeatResponse.isReadCompleted()) {
                readFutureTtl.delete(String.valueOf(taskId));
                ImportPushDataRequest importPushDataRequest = renderJobHeartbeatResponse.getImportPushDataRequest();
                if (importPushDataRequest==null) {
                    renderFuture.setDataRowStreamAndNotify(null,new RuntimeException("render server error"));
                    return;
                }
                renderFuture.setDataRowStreamAndNotify(dataStream0(renderFuture.file,renderFuture.readDef,renderJobHeartbeatResponse.getImportPushDataRequest()),null);
                return;
            }

            throw new RuntimeException("unreachable code , error???");
        }
    }

    @Data
    private static class RequestParamBodyImpl implements RequestParamBody{

        private String file;

        private ReadDef readDef;

    }

    static private JobInTaskFlow.DataRowStream<RequestParamBody> dataStream0(String file,ReadDef readDef, ImportPushDataRequest importPushDataRequest){
        RequestParamBodyImpl requestParamBody=new RequestParamBodyImpl();
        requestParamBody.setFile(file);
        requestParamBody.setReadDef(jobMeta0(readDef));

        // request param
        ImportPreDefConfImpl importPreDefConf=new ImportPreDefConfImpl();
        importPreDefConf.setDataExistStrategy(DataExistStrategy.NONE);

        AtomicBoolean cancelled=new AtomicBoolean(false);
        JobInTaskFlow.ImportContext importContext =new JobInTaskFlow.ImportContext() {
            @Override
            public void cancel() {
                cancelled.set(true);
            }

            @Override
            public synchronized void replyWithSegment(int segment, int fromRow, int toRow, String desc, JobInTaskFlow.Result result) {

            }
        };

        JobInTaskFlow.ImportConf<RequestParamBody> importConf=new JobInTaskFlow.ImportConf<RequestParamBody>() {
            @Override
            public RequestParamBody requestParam() {
                return requestParamBody;
            }

            @Override
            public ImportPreDefConf importPreDefConf() {
                return importPreDefConf;
            }
        };

        final List<SegmentImpl> segmentList = importPushDataRequest.getSegmentList();
        int segmentCount = segmentList.size();
        int rowCount=0;
        for (SegmentImpl segment : segmentList) {
            rowCount+=segment.size();
        }
        int finalRowCount = rowCount;
        LimitLogger limitLogger=new LimitLogger() {
            @Override
            public void log(String content) {
                log.info(content);
            }
        };
        JobInTaskFlow.DataRowStream<RequestParamBody> dataRowStream =new JobInTaskFlow.DataRowStream<RequestParamBody>() {
            final AtomicBoolean reset=new AtomicBoolean(true);

            @Override
            public JobInTaskFlow.DataRowStream<RequestParamBody> reset() {
                cancelled.set(false);
                return this;
            }

            @Override
            public void mark(JobInTaskFlow.StreamProcessResult streamProcessResult, String msg) {

            }

            @Override
            public LimitLogger limitLogger() {
                return limitLogger;
            }

            @Override
            public void subscribe(JobInTaskFlow.DataRowListener dataRowListener) {
                if (!reset.get()) {
                    throw new IllegalStateException("need reset stream before subscribe it.");
                }
                for (int segmentIndex = 0; segmentIndex < segmentList.size(); segmentIndex++) {
                    if (cancelled.get()) {
                        Stream.from("cancel the import flow, segment: "+segmentIndex).subscribe(s -> {
                            log.info(s);
                        });
                        return;
                    }
                    Segment segment=segmentList.get(segmentIndex);
                    final Map<String, Segment.Header.Cell> headerCellMap=new HashMap<>();
                    segment.header().scanUp2Down(new Segment.Header.Scan() {
                        @Override
                        public void scan(Segment.Header.Cell cell) {
                            headerCellMap.put(cell.path(),cell);
                        }
                    });
                    JobInTaskFlow.HeaderLookup headerLookup=new JobInTaskFlow.HeaderLookup() {
                        @Override
                        public Segment.Header.Cell lookup(String path) {
                            return headerCellMap.get(path);
                        }
                    };
                    List<Map<String, Object>> mapList = segment.rowList();
                    for (int rowIndex = 0; rowIndex < mapList.size(); rowIndex++) {
                        if (cancelled.get()) {
                            Stream.from("cancel the import flow, segment: "+segmentIndex).subscribe(s -> {
                                log.info(s);
                            });
                            return;
                        }

                        Map<String, Object> data = mapList.get(rowIndex);
                        boolean allEmpty=true;
                        for (Object v : data.values()) {
                            if (isNotEmpty((String)v)) {
                                allEmpty=false;
                                break;
                            }
                        }

                        if (allEmpty) {
                            continue;
                        }

                        JobInTaskFlow.DataRowImpl dataRow=new JobInTaskFlow.DataRowImpl();
                        dataRow.setSegment(segmentIndex);
                        dataRow.setRowNum(rowIndex);
                        dataRow.setData(data);
                        dataRow.setHeaderLookup(headerLookup);
                        dataRowListener.onRow(dataRow, importContext);
                    }
                }
                reset.set(false);
            }

            @Override
            public int segmentCount() {
                return segmentCount;
            }

            @Override
            public long rowCount() {
                return finalRowCount;
            }

            @Override
            public long rowCount(int segment) {
                return segmentList.get(segment).size();
            }

            @Override
            public Session session() {
                throw new UnsupportedOperationException("read job does not contains session");
            }

            @Override
            public JobInTaskFlow.ImportConf<RequestParamBody> importConf() {
                return importConf;
            }
        };

        return dataRowStream;
    }

}
