package com.ovopark.iohub.sdk.client.outstream;

import com.ovopark.iohub.sdk.client.*;
import com.ovopark.iohub.sdk.model.AppNode;
import com.ovopark.iohub.sdk.model.JobMeta;
import com.ovopark.iohub.sdk.model.outstream.*;
import com.ovopark.iohub.sdk.model.proto.OutStore;
import com.ovopark.iohub.sdk.model.proto.Segment;
import com.ovopark.iohub.sdk.model.proto.internal.*;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import com.ovopark.kernel.shared.concurrent.ListenerFuture;
import com.ovopark.kernel.shared.kv.KVEngine;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;

import static com.ovopark.kernel.shared.Util.catchRunnable;
import static com.ovopark.kernel.shared.Util.newThreadFactory;

@Slf4j
class RenderJobImpl implements RenderJob{

    OutStore outStore;

    private final String uri;

    private final boolean nfs;

    final JobMeta jobMeta;

    final JobHint jobHint;

    final private IOHubClientConfig ioHubClientConfig;

    final private Client2ControlTransport client2ControlTransport;

    final ClientNode clientNode;

    final Long taskId;

    final boolean renderJobTest;

    private static final ScheduledExecutorService jobHeartbeatExecutor = new ScheduledThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,8)
                    , Config.ConfigPriority.option().getInt("IOHUB_RENDER_JOB_HEARTBEAT_IO", 0))
            , newThreadFactory("iohub_render_job_heartbeat_io")
            , new ThreadPoolExecutor.AbortPolicy());

    private static final ExecutorService jobFutureExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,64)
                    , Config.ConfigPriority.option().getInt("IOHUB_RENDER_JOB_FUTURE_IO",0))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,64)
            , Config.ConfigPriority.option().getInt("IOHUB_RENDER_JOB_FUTURE_IO", 0))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(Integer.MAX_VALUE)
            , newThreadFactory("iohub_render_job_future_io")
            , new ThreadPoolExecutor.AbortPolicy());

    final static KVEngine.TtlFunc<String> renderFutureTtl = KVEngine.newTtl(RenderJob.class.getName());

    static {
        renderFutureTtl.subscribeTtl(t -> true, (getResult, l, l1) -> {
            String key = getResult.key();
            log.info("onExpired: "+key);
            RenderFutureImpl renderFuture = (RenderFutureImpl) getResult.value();
            renderFuture.setOutFileUrlAndNotify(null,new TimeoutException("seconds: "+TimeUnit.MILLISECONDS.toSeconds(l1)));
        });
    }

    final static RestTemplate restTemplate;
    static {
        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setReadTimeout(45_000);//ms
        factory.setConnectTimeout(15_000);//ms
        restTemplate = new RestTemplate(factory);
        restTemplate.getMessageConverters().add(0, new StringHttpMessageConverter(StandardCharsets.UTF_8));
    }

    public RenderJobImpl(Long taskId,String uri, boolean nfs, JobMeta jobMeta, JobHint jobHint
            , IOHubClientConfig ioHubClientConfig, Client2ControlTransport client2ControlTransport
            , ClientNode clientNode,boolean renderJobTest) {
        this.taskId=taskId;
        this.uri = uri;
        this.nfs = nfs;
        this.jobMeta = jobMeta;
        this.jobHint = jobHint;
        this.ioHubClientConfig = ioHubClientConfig;
        this.client2ControlTransport = client2ControlTransport;
        this.clientNode = clientNode;
        this.renderJobTest = renderJobTest;
    }

    @Override
    public synchronized OutStore outStore() {
        if (outStore==null) {
            if (nfs) {
                RowTransLogConfig rowTransLogConfig=new RowTransLogConfig();
                rowTransLogConfig.setRegionRowCount(1000);
                outStore=new NFSOutStore(ioHubClientConfig.nfsPath()+"/client-"+ ClientNode.UUID_STR,1,rowTransLogConfig);
                ((NFSOutStore) outStore).setLimitLogger(log::info);
            }
            else {
                outStore=new InMemoryOutStore(8);
                ((InMemoryOutStore) outStore).setLimitLogger(log::info);
            }
        }
        return outStore;
    }

    @Override
    public RenderFuture render2Excel() {
        return render();
    }

    @Override
    public RenderFuture render() {
        return render0(null);
    }

    private RenderFutureImpl render0(JobHint jobHint) {
        RenderJobAssignWorkRequest renderJobAssignWorkRequest =new RenderJobAssignWorkRequest();
        renderJobAssignWorkRequest.setTaskId(taskId);
        renderJobAssignWorkRequest.setClientApp(clientNode.app());
        renderJobAssignWorkRequest.setClientNode(clientNode.node());
        renderJobAssignWorkRequest.setUri(uri);
        renderJobAssignWorkRequest.setTest(renderJobTest);
        RenderJobAssignWorkResponse renderJobAssignWorkResponse = client2ControlTransport.assignWorkRenderJob(renderJobAssignWorkRequest);
        log.info("jobRenderAssignWorkResponse: "+JSONAccessor.impl().format(renderJobAssignWorkResponse));
        if (renderJobAssignWorkResponse ==null || !renderJobAssignWorkResponse.isSuccess() || renderJobAssignWorkResponse.getWorker()==null) {
            throw new IllegalArgumentException("cannot assign work for job render: "+ JSONAccessor.impl().format(renderJobAssignWorkResponse));
        }

        AppNode worker= renderJobAssignWorkResponse.getWorker();
        Client2WorkTransport client2WorkTransport =new Client2WorkRestClient(worker,restTemplate);

        List<Segment.SD> sdList = outStore.sdList();
        if(outStore instanceof InMemoryOutStore){
            //memory
            List<SegmentImpl> list=new ArrayList<>();
            for (Segment.SD sd : sdList) {
                Segment segment = outStore.segment(sd);
                list.add((SegmentImpl) segment);
            }

            ExportPushDataRequest exportPushDataRequest =new ExportPushDataRequest();
            exportPushDataRequest.setSegmentList(list);

            exportPushDataRequest.setTaskId(taskId);
            exportPushDataRequest.setApp(clientNode.app());
            exportPushDataRequest.setNode(clientNode.node());
            exportPushDataRequest.setWorkApp(worker.getApp());
            exportPushDataRequest.setWorkNode(worker.getNode());

            log.info("request push data checkpoint");
            ExportPushDataResponse exportPushDataResponse = client2WorkTransport.pushRenderJob(exportPushDataRequest);
            log.info("response push data checkpoint: "+ JSONAccessor.impl().format(exportPushDataResponse));
            if(exportPushDataResponse ==null || !exportPushDataResponse.isSuccess()){
                throw new IllegalStateException("cannot push data to worker:");
            }

            ExportPushCommitRequest exportPushCommitRequest =new ExportPushCommitRequest();
            exportPushCommitRequest.setFileName(((InMemoryOutStore) outStore).getFileName());

            exportPushCommitRequest.setTaskId(taskId);
            exportPushCommitRequest.setApp(clientNode.app());
            exportPushCommitRequest.setNode(clientNode.node());
            exportPushCommitRequest.setWorkApp(worker.getApp());
            exportPushCommitRequest.setWorkNode(worker.getNode());

            exportPushCommitRequest.setGroupMap(((InMemoryOutStore) outStore).getGroupMap());
            exportPushCommitRequest.setFeature(((InMemoryOutStore) outStore).getFeature());
            exportPushCommitRequest.setJobHint(jobHint);
            log.info("request commit checkpoint");
            ExportPushCommitResponse exportPushCommitResponse = client2WorkTransport.commitRenderJob(exportPushCommitRequest);
            log.info("response commit checkpoint: "+JSONAccessor.impl().format(exportPushCommitResponse));
            if(exportPushCommitResponse ==null || !exportPushCommitResponse.isSuccess()){
                throw new IllegalStateException("cannot commit data to worker:");
            }
        }

        else {
            //NFS
            ExportPushDataRequest exportPushDataRequest =new ExportPushDataRequest();
            exportPushDataRequest.setTaskId(taskId);
            exportPushDataRequest.setApp(clientNode.app());
            exportPushDataRequest.setNode(clientNode.node());
            exportPushDataRequest.setWorkApp(worker.getApp());
            exportPushDataRequest.setWorkNode(worker.getNode());

            log.info("request push data checkpoint");
            ExportPushDataResponse exportPushDataResponse = client2WorkTransport.pushRenderJob(exportPushDataRequest);
            log.info("response push data checkpoint: "+ JSONAccessor.impl().format(exportPushDataResponse));
            if(exportPushDataResponse ==null || !exportPushDataResponse.isSuccess()){
                throw new IllegalStateException("cannot push data to worker:");
            }

            List<NFSSegmentModel> nfsSegmentModelList=new ArrayList<>();
            for (Segment.SD sd : sdList) {
                NFSSegmentModel nfsSegmentModel=new NFSSegmentModel();
                nfsSegmentModel.setName(sd.getName());
                nfsSegmentModel.setUrl(sd.getUrl());
                nfsSegmentModel.setMeta(sd.getMeta());
                nfsSegmentModel.setRowCount(sd.getRowCount());
                Segment segment = outStore.segment(sd);
                nfsSegmentModel.setTitle((TitleImpl) segment.title());
                nfsSegmentModel.setHeader((HeaderImpl) segment.header());
                nfsSegmentModel.setGroup(segment.group());
                nfsSegmentModel.setFeature(segment.feature());

                nfsSegmentModelList.add(nfsSegmentModel);
            }

            ExportPushCommitRequest exportPushCommitRequest =new ExportPushCommitRequest();
            exportPushCommitRequest.setFileName(((NFSOutStore) outStore).getFileName());
            exportPushCommitRequest.setNfsSegmentModelList(nfsSegmentModelList);

            exportPushCommitRequest.setTaskId(taskId);
            exportPushCommitRequest.setApp(clientNode.app());
            exportPushCommitRequest.setNode(clientNode.node());
            exportPushCommitRequest.setWorkApp(worker.getApp());
            exportPushCommitRequest.setWorkNode(worker.getNode());

            exportPushCommitRequest.setGroupMap(((NFSOutStore) outStore).getGroupMap());
            exportPushCommitRequest.setFileIOUrl(((NFSOutStore) outStore).getFileIOUrl());
            exportPushCommitRequest.setFeature(((NFSOutStore) outStore).getFeature());
            exportPushCommitRequest.setJobHint(jobHint);

            log.info("request commit checkpoint");
            ExportPushCommitResponse exportPushCommitResponse = client2WorkTransport.commitRenderJob(exportPushCommitRequest);
            log.info("response commit checkpoint: "+JSONAccessor.impl().format(exportPushCommitResponse));
            if(exportPushCommitResponse ==null || !exportPushCommitResponse.isSuccess()){
                log.error("cannot commit data to worker:"+ JSONAccessor.impl().format(exportPushCommitResponse));
                throw new IllegalStateException("cannot commit data to worker:");
            }

        }

        RenderFutureImpl renderFuture=new RenderFutureImpl();
        RenderFutureTask renderFutureTask=new RenderFutureTask(taskId,worker,clientNode,renderFuture);
        jobHeartbeatExecutor.schedule(catchRunnable(renderFutureTask),5,TimeUnit.SECONDS);
        renderFutureTtl.putIfAbsentAndGet(String.valueOf(taskId), k->renderFutureTask,60*15,TimeUnit.SECONDS);
        return renderFuture;
    }

    @Override
    public RenderFuture render4Test(JobHint jobHint) {
        return render0(jobHint);
    }

    static class RenderFutureImpl implements RenderFuture{

        private final ListenerFuture.ListenerFutureSetter<String> futureSetter=ListenerFuture.newFuture();

        void setOutFileUrlAndNotify(String outFileUrl, Throwable t) {
            futureSetter.setValueAndNotify(outFileUrl, t);
        }

        void cancelAndNotify() {
            futureSetter.cancelAndNotify();
        }

        @Override
        public String get() throws InterruptedException,ExecutionException{
            return futureSetter.get();
        }

        @Override
        public String get(long timeout, TimeUnit timeUnit) throws InterruptedException ,TimeoutException,ExecutionException{
            return futureSetter.get(timeout, timeUnit);
        }

        @Override
        public boolean cancelled() {
            return futureSetter.isCancelled();
        }

        @Override
        public void cancel() {
            futureSetter.cancel(true);
        }

        @Override
        public void setRenderListener(RenderListener renderListener) {
            futureSetter.addListener(renderListener::onRender);
        }

    }

    static class RenderFutureTask implements Util.CatchRunnable{

        final Long taskId;

        final AppNode worker;

        final ClientNode clientNode;

        final RenderFutureImpl renderFuture;

        final long startMs=System.currentTimeMillis();

        final Client2WorkTransport client2WorkTransport;

        int failCount;

        public RenderFutureTask(Long taskId, AppNode worker, ClientNode clientNode, RenderFutureImpl renderFuture) {
            this.taskId = taskId;
            this.worker = worker;
            this.clientNode = clientNode;
            this.renderFuture = renderFuture;
            //
            SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
            factory.setReadTimeout(45_000);//ms
            factory.setConnectTimeout(15_000);//ms
            RestTemplate restTemplate = new RestTemplate(factory);
            restTemplate.getMessageConverters().add(0,new StringHttpMessageConverter(StandardCharsets.UTF_8));
            client2WorkTransport =new Client2WorkRestClient(worker,restTemplate);
        }

        @Override
        public void run() throws Exception {

            if (renderFuture.cancelled()) {
                log.info("cancel job render by client: "+taskId);
                return;
            }

            RenderJobHeartbeatRequest renderJobHeartbeatRequest =new RenderJobHeartbeatRequest();
            renderJobHeartbeatRequest.setTaskId(taskId);
            renderJobHeartbeatRequest.setClientApp(clientNode.app());
            renderJobHeartbeatRequest.setClientNode(clientNode.node());
            renderJobHeartbeatRequest.setWorkNode(worker.getNode());

            RenderJobHeartbeatResponse renderJobHeartbeatResponse = client2WorkTransport.heartbeatRenderJob(renderJobHeartbeatRequest);
            if (renderJobHeartbeatResponse ==null || !renderJobHeartbeatResponse.isSuccess()) {
                log.info("response job render heartbeat: "+JSONAccessor.impl().format(renderJobHeartbeatResponse));
            }
            if(renderJobHeartbeatResponse ==null || !renderJobHeartbeatResponse.isSuccess()){
                if (failCount++>5) {
                    renderFutureTtl.delete(String.valueOf(taskId));
                    renderFuture.setOutFileUrlAndNotify(null,new RuntimeException("render server error: "
                            +(renderJobHeartbeatResponse ==null?"": renderJobHeartbeatResponse.getDesc())
                    ));
                    return;
                }
                jobHeartbeatExecutor.schedule(catchRunnable(this),5,TimeUnit.SECONDS);
                return;
            }

            if (renderJobHeartbeatResponse.isRendering()) {
                jobHeartbeatExecutor.schedule(catchRunnable(this),5,TimeUnit.SECONDS);
                return;
            }

            if (renderJobHeartbeatResponse.isRenderCancelled()) {
                log.info("render job ( "+taskId+" ) was cancelled by work: "+JSONAccessor.impl().format(renderJobHeartbeatResponse));
                renderFutureTtl.delete(String.valueOf(taskId));
                renderFuture.cancelAndNotify();
                return;
            }

            if (renderJobHeartbeatResponse.isRenderError()) {
                renderFutureTtl.delete(String.valueOf(taskId));
                renderFuture.setOutFileUrlAndNotify(null,new RuntimeException(renderJobHeartbeatResponse.getRenderErrorDesc()));
                return;
            }

            if (renderJobHeartbeatResponse.isRenderCompleted()) {
                renderFutureTtl.delete(String.valueOf(taskId));
                String outFileUrl = renderJobHeartbeatResponse.getOutFileUrl();
                renderFuture.setOutFileUrlAndNotify(outFileUrl,null);
                return;
            }

            throw new RuntimeException("unreachable code , error???");
        }
    }

}
