package com.ovopark.iohub.sdk.client.outstream;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.ovopark.iohub.sdk.client.*;
import com.ovopark.iohub.sdk.model.*;
import com.ovopark.iohub.sdk.model.outstream.ExportPushStartRequest;
import com.ovopark.iohub.sdk.model.outstream.ExportPushStartResponse;
import com.ovopark.iohub.sdk.model.proto.IO;
import com.ovopark.iohub.sdk.model.proto.LimitLogger;
import com.ovopark.iohub.sdk.model.proto.NoPrivilegeException;
import com.ovopark.iohub.sdk.model.proto.OutStore;
import com.ovopark.iohub.sdk.model.proto.internal.InMemoryOutStore;
import com.ovopark.iohub.sdk.model.proto.internal.JobHint;
import com.ovopark.iohub.sdk.model.proto.internal.NFSOutStore;
import com.ovopark.iohub.sdk.model.proto.internal.RowTransLogConfig;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.module.shared.spring.rbac.SessionImpl;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Slf4j
@RestController("com.ovopark.iohub.sdk.client.outstream.ExportEndpoint")
@RequestMapping("/feign/iohub-job/processing")
public class ExportEndpoint implements InitializingBean {

    @Autowired
    private Client2ControlTransport jobClient2ControlTransport;

    private static final ExecutorService jobExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,64)
                    , Config.ConfigPriority.option().getInt("IOHUB_EXPORT_IO", 0))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,1024)
            , Config.ConfigPriority.option().getInt("IOHUB_EXPORT_IO", 0))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(1)
            , newThreadFactory("iohub-export-io")
            , new ThreadPoolExecutor.AbortPolicy());

//    private static final KVEngine.TtlFunc<String> C=KVEngine.newTtl("iohub-export-io-c");

    final private Set<Long> executingJobTaskIdSet = ConcurrentHashMap.newKeySet();

    final Map<String,SubmitTask> submitTaskMap=new ConcurrentHashMap<>();

    final AtomicInteger receivedTaskCount=new AtomicInteger(0);

    @Autowired
    private IOHubClientConfig ioHubClientConfig;

    @Autowired
    private JobOutTaskFlowProvider jobOutTaskFlowProvider;

    @Autowired
    private ClientNodeRegister.ClientNodeProvider clientNodeProvider;


    @RequestMapping("/submit")
    @ResponseBody
    public TaskSubmitResponse submit(@RequestBody TaskSubmitRequest taskSubmitRequest){
        final String taskKey = "task_" + taskSubmitRequest.getTaskId();
        MDC.put("traceId", taskKey);
        MDC.put("requestId", taskKey);
        log.info("receive a new task, we can do it???: "+JSONAccessor.impl().format(taskSubmitRequest));
        final TaskSubmitResponse taskSubmitResponse=new TaskSubmitResponse();
        taskSubmitResponse.setSubmitted(false);
        final JobLog client2ControlLog = new JobLogImpl(jobClient2ControlTransport,  taskKey,taskSubmitRequest.getTaskId());
        try {
            lock(taskKey, (Callable<Void>) () -> {
                // same task is submit in the same node , 10min???
                // skip the same task ??? task id + task history id
                if (executingJobTaskIdSet.contains(taskSubmitRequest.getTaskId())) {
                    taskSubmitResponse.setSubmitted(false);
                    taskSubmitResponse.setDesc("Rejected,the previous task is executing");
                    throw new IllegalStateException("the previous task is executing, reject the task: "
                            +JSONAccessor.impl().format(taskSubmitRequest));
                }

                TaskLockRequest taskLockRequest=new TaskLockRequest();
                taskLockRequest.setTaskId(taskSubmitRequest.getTaskId());
                taskLockRequest.setNode(ClientNode.UUID_STR);
                TaskLockResponse taskLockResponse = jobClient2ControlTransport.lockClient(taskLockRequest);
                if (taskLockResponse == null || !taskLockResponse.isSuccess()) {
                    taskSubmitResponse.setSubmitted(false);
                    throw new IllegalStateException("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest));
                }

                receivedTaskCount.incrementAndGet();
                executingJobTaskIdSet.add(taskSubmitRequest.getTaskId());

                //process pre flow ...
                final long taskId=taskSubmitRequest.getTaskId();
                final String uri=taskSubmitRequest.getUri();
                final String args=taskSubmitRequest.getArgs();
                final String session=taskSubmitRequest.getSession();
                final boolean retry=taskSubmitRequest.isRetry();
                final Integer exportTaskId=taskSubmitRequest.getExportTaskId();
                final JobMeta jobMeta=taskSubmitRequest.getJobMeta();
                final AppNode worker=taskSubmitRequest.getWorker();
                final JobHint jobHint = taskSubmitRequest.getJobHint();
                final JobTaskManager jobTaskManager=new JobTaskManager() {
                    @Override
                    public void cancel(long taskId) {
                        SubmitTask submitTask = submitTaskMap.get("task_" +taskId);
                        submitTask.setKilled(true);
                        submitTask.future.cancel(true);
                        logLink("cancel the task").log(log::info).log(client2ControlLog::log);
                    }
                };

                ClientNode clientNode = clientNodeProvider.clientNode();

                SessionImpl si = isEmpty(session)?null: JSONAccessor.impl().read(session, SessionImpl.class);
                SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
                factory.setReadTimeout(45_000);//ms
                factory.setConnectTimeout(15_000);//ms
                RestTemplate restTemplate = new RestTemplate(factory);
                restTemplate.getMessageConverters().add(0,new StringHttpMessageConverter(StandardCharsets.UTF_8));
                Client2WorkTransport client2WorkTransport =new Client2WorkRestClient(worker,restTemplate);

                ExportPushStartRequest exportPushStartRequest =new ExportPushStartRequest();
                exportPushStartRequest.setTaskId(taskId);
                exportPushStartRequest.setApp(clientNode.app());
                exportPushStartRequest.setClientNode(clientNode.node());
                exportPushStartRequest.setWorkApp(worker.getApp());
                exportPushStartRequest.setWorkNode(worker.getNode());

                exportPushStartRequest.setExportTaskId(exportTaskId);
                exportPushStartRequest.setJobHint(jobHint);

                exportPushStartRequest.setSession(session);
                exportPushStartRequest.setUserId(si==null?null:si.getUserId());
                exportPushStartRequest.setUserName(si==null?null:si.getUserName());
                exportPushStartRequest.setUserGroupId(si==null?null:si.getGroupId());
                exportPushStartRequest.setArgs(args);
                exportPushStartRequest.setArgsMd5(md5(args));
                exportPushStartRequest.setUri(uri);
                exportPushStartRequest.setJobMeta(jobMeta);

                logLink("to request resource : ").log(log::info).log(client2ControlLog::log);

                ExportPushStartResponse exportPushStartResponse = client2WorkTransport.start(exportPushStartRequest);
                if (exportPushStartResponse ==null || !exportPushStartResponse.isSuccess()) {
                    logLink("cannot get enough resource , break the export task.").log(log::info).log(client2ControlLog::log);
                    taskSubmitResponse.setSubmitted(false);
                    taskSubmitResponse.setDesc("resource limit,"
                            +(exportPushStartResponse==null?"":exportPushStartResponse.getDesc())
                    );
                    throw new IllegalStateException("resource limit, reject the task: "
                            +JSONAccessor.impl().format(taskSubmitRequest));
                }

                logLink("get request resource : ").log(log::info).log(client2ControlLog::log);
                JobOutTaskFlow<RequestParamBody> jobOutTaskFlow = jobOutTaskFlowProvider.find(uri);
                boolean taskSubmit=false;
                try {
                    //prepared
                    jobOutTaskFlow.prepared();
                    logLink("prepare bean : ").log(log::info).log(client2ControlLog::log);
                    RequestParamBody requestParamBody=null;
                    if (retry) {
                        requestParamBody=JSONAccessor.impl().read(args,jobOutTaskFlow.paramType());
                    }
                    else {
                        // create export record
                        try {
                            requestParamBody = jobOutTaskFlow.requestParamBody(args, si);
                        } catch (NoPrivilegeException e) {
                            taskSubmitResponse.setSubmitted(false);
                            taskSubmitResponse.setDesc("no privilege");
                            throw new IllegalStateException("no privilege, reject the task: "
                                    +JSONAccessor.impl().format(taskSubmitRequest));
                        }catch (Exception e){
                            client2ControlLog.flush();
                            errorStackList(e).forEach(client2ControlLog::log);
                            client2ControlLog.flush();
                            taskSubmitResponse.setSubmitted(false);
                            taskSubmitResponse.setDesc(e.getMessage());
                            log.error(e.getMessage(),e);
                            throw new IllegalStateException("error, reject the task: "+e.getMessage()
                                    +JSONAccessor.impl().format(taskSubmitRequest));
                        }

                        JobOutTaskFlow.ExportTaskInfoProvider exportTaskInfoProvider = jobOutTaskFlow.exportTaskInfoProvider(requestParamBody, si);


                        CreateExportRecordRequest createExportRecordRequest = new CreateExportRecordRequest();
                        createExportRecordRequest.setRequestParamBody(requestParamBody);

                        createExportRecordRequest.setExportTaskName(exportTaskInfoProvider==null?null:exportTaskInfoProvider.name());
                        createExportRecordRequest.setTaskId(taskId);
                        createExportRecordRequest.setApp(clientNode.app());
                        createExportRecordRequest.setNode(clientNode.node());
                        createExportRecordRequest.setWorkApp(worker.getApp());
                        createExportRecordRequest.setWorkNode(worker.getNode());

                        createExportRecordRequest.setUserId(si==null?null:si.getUserId());
                        createExportRecordRequest.setGroupId(si==null?null:si.getGroupId());

                        createExportRecordRequest.setJobMeta(jobMeta);

                        CreateExportRecordResponse exportRecord = client2WorkTransport.createExportRecord(createExportRecordRequest);
                        logLink("createExportRecord: "+JSONAccessor.impl().format(exportRecord)).log(log::info).log(client2ControlLog::log);
                        if (exportRecord==null || !exportRecord.isSuccess() || exportRecord.getExportTaskId()==null) {
                            logLink("cannot get exportRecord id , break the export task.")
                                    .log(log::info).log(client2ControlLog::log);
                            taskSubmitResponse.setSubmitted(false);
                            taskSubmitResponse.setDesc("cannot create exportRecord");
                            throw new IllegalStateException("cannot create exportRecord: "
                                    +JSONAccessor.impl().format(taskSubmitRequest));
                        }
                        logLink("created export record  , export record id: "+exportRecord.getExportTaskId())
                                .log(log::info).log(client2ControlLog::log);
                        taskSubmitResponse.setExportTaskId(exportRecord.getExportTaskId());
                    }
                    //query & export ???
                    OutStore outStore;
                    boolean nfs = jobMeta.isNfs();
                    if (jobHint!=null && isNotEmpty(jobHint.getOutStore())) {
                        nfs="nfs".equalsIgnoreCase(jobHint.getOutStore()) || "nas".equalsIgnoreCase(jobHint.getOutStore());
                    }
                    if (nfs) {
                        RowTransLogConfig rowTransLogConfig=new RowTransLogConfig();
                        int regionRowCount=jobMeta.getRegionRowCount()>0?jobMeta.getRegionRowCount():1000;
                        if (jobHint != null && jobHint.getRegionRowCount() > 0) {
                            regionRowCount = jobHint.getRegionRowCount();
                        }
                        rowTransLogConfig.setRegionRowCount(regionRowCount);
                        rowTransLogConfig.setCompressed(jobMeta.isCompressed());
                        outStore=new NFSOutStore(ioHubClientConfig.nfsPath()+"/client-"+ClientNode.UUID_STR
                                +"/export/"+taskId
                                ,1,rowTransLogConfig);
                    }
                    else {
                        logLink("create memory out store, maxRowCountInMemory: "+jobMeta.getMaxRowCountInMemory())
                                .log(log::info).log(client2ControlLog::log);
                        outStore=new InMemoryOutStore(Math.min(jobMeta.getSegmentCount(),10),jobMeta.getMaxRowCountInMemory());
                    }
                    logLink("create out store , nfs ? : "+ nfs).log(log::info).log(client2ControlLog::log);
                    ClientNodeTaskRegisterRequest clientNodeTaskRegisterRequest=new ClientNodeTaskRegisterRequest();
                    clientNodeTaskRegisterRequest.setTaskId(taskId);
                    clientNodeTaskRegisterRequest.setApp(clientNode.app());
                    clientNodeTaskRegisterRequest.setNode(clientNode.node());
                    clientNodeTaskRegisterRequest.setWorkApp(worker.getApp());
                    clientNodeTaskRegisterRequest.setWorkNode(worker.getNode());
                    clientNodeTaskRegisterRequest.setVersion(IO.version);
                    clientNodeTaskRegisterRequest.setMinVersion(IO.minVersion);

                    RequestParamBody finalRequestParamBody=requestParamBody;
                    //ok , we create a submit task locally
                    final SubmitTask submitTask=new SubmitTask();
                    submitTask.setTaskModel(new TaskModel());
                    final TaskModel taskModel = submitTask.getTaskModel();
                    taskModel.setJobMeta(taskSubmitRequest.getJobMeta());
                    taskModel.setJobHint(taskSubmitRequest.getJobHint());
                    taskModel.setExportClientStat(new TaskModel.ExportClientStat());
                    final TaskModel.ExportClientStat exportClientStat = taskModel.getExportClientStat();

                    submitTask.setRuntimeStat(outStore.runtimeStat());

                    submitTask.setTaskId(taskSubmitRequest.getTaskId());
                    taskModel.setJobTaskId(taskSubmitRequest.getTaskId());
                    taskModel.setExportTaskId(taskSubmitRequest.getExportTaskId());
                    taskModel.setWorkNode(taskSubmitRequest.getWorker().getNode());
                    taskModel.setClientNode(ClientNode.UUID_STR);

                    taskModel.setExport(true);
                    taskModel.setUri(uri);
                    taskModel.setClientApp(clientNode.app());
                    taskModel.setSession(session);
                    taskModel.setUserId(si==null?null:si.getUserId());
                    taskModel.setUserName(si==null?null:si.getUserName());
                    taskModel.setUserGroupId(si==null?null:si.getGroupId());

                    exportClientStat.setArgs(args);
                    exportClientStat.setArgsMd5(md5(args));
                    exportClientStat.setConvertArgs(JSONAccessor.impl().format(finalRequestParamBody));

                    try {
                        Future<?> future = jobExecutor.submit(catchRunnable(() -> {
                            submitTask.setIoThread(Thread.currentThread());
                            submitTask.setOutStore(outStore);

                            final JobLog jobLog2Control = new JobLogImpl(jobClient2ControlTransport, taskKey,taskId);
                            taskModel.setIoThreadName(Thread.currentThread().getName());
                            taskModel.setTransientFunc(() -> {
                                OutStore.RuntimeStat runtimeStat = outStore.runtimeStat();
                                int sumRowCount = exportClientStat.getSumRowCount();
                                int rowCount = runtimeStat.rowCount();
                                exportClientStat.setRowCountAdded(rowCount-sumRowCount);
                                exportClientStat.setSumRowCount(rowCount);

                                String currentSegment = runtimeStat.currentSegment();
                                exportClientStat.setCurrentSegment(currentSegment);
                                exportClientStat.setSegmentCount(runtimeStat.segmentCount());

                                long lastIoTimeMs = runtimeStat.lastIoTimeMs();
                                exportClientStat.setLastIoTimeStr(formatTime(dateTime(lastIoTimeMs)));

                                long byteSize = runtimeStat.byteSize();
                                exportClientStat.setByteSize(byteSize);

                                exportClientStat.setJsonByteSize(runtimeStat.jsonByteSize());
                                exportClientStat.setCompressedByteSize(runtimeStat.compressedByteSize());
                                exportClientStat.setMaxRowRegionByteSize(runtimeStat.maxRowRegionByteSize());
                            });
                            clientNodeTaskRegisterRequest.setIoThreadName(Thread.currentThread().getName());

                            //OK , start doing real task logic
                            ScheduledThreadPoolExecutor scheduledThreadPoolExecutor
                                    =new ScheduledThreadPoolExecutor(
                                    1,newThreadFactory("tmp"),new ThreadPoolExecutor.CallerRunsPolicy());
                            scheduledThreadPoolExecutor.setKeepAliveTime(60,TimeUnit.SECONDS);
                            scheduledThreadPoolExecutor.allowCoreThreadTimeOut(true);
                            try {
                                doTask((Callable<Void>) () -> {
                                    final int maxClientLogCount = jobMeta.getMaxClientLogCount();
                                    logLink("create limit logger, maxClientLogCount: "+ maxClientLogCount)
                                            .log(log::info).log(jobLog2Control::log);
                                    ((LimitLogger.LimitLoggerSetter) outStore).setLimitLogger(new LimitLogger() {
                                        final AtomicLong limit=new AtomicLong();
                                        @Override
                                        public void log(String content) {
                                            log.info(content);
                                            if (limit.incrementAndGet()> maxClientLogCount) {
                                                log.warn("limit logger,cannot push log to control, exceed max client log count: "+ maxClientLogCount);
                                                return;
                                            }
                                            jobLog2Control.log(content);
                                        }
                                    });
                                    logLink("to get all data ").log(log::info).log(jobLog2Control::log);
                                    jobLog2Control.flush();
                                    try {
                                        jobOutTaskFlow.execute(finalRequestParamBody,outStore);
                                        outStore.commit();
                                    }
                                    catch (Throwable t){
                                        log.error("job out task flow error: "+t.getMessage(),t);
                                        jobLog2Control.log("job out task flow error: "+t.getMessage());
                                        jobLog2Control.flush(true);
                                        errorStackList(t).forEach(jobLog2Control::log);
                                        jobLog2Control.flush(true);
                                        throw convert2RuntimeException(t);
                                    }
                                    finally {
                                        outStore.close();
                                    }
                                    logLink("got all data , then push meta to work").log(log::info).log(jobLog2Control::log);
                                    jobLog2Control.flush();

                                    OutStore.Stat stat = outStore.stat();
                                    if (isNotEmpty(stat.getSegmentStatList())) {
                                        StringBuilder stringBuilder=new StringBuilder();
                                        for (int i = 0; i < stat.getSegmentStatList().size(); i++) {
                                            OutStore.SegmentStat segmentStat=stat.getSegmentStatList().get(i);
                                            int rowCount = segmentStat.getRowCount();
                                            stringBuilder.append(",s"+i+": "+rowCount);
                                        }
                                        log.info(stringBuilder.toString());
                                        jobLog2Control.log(stringBuilder.toString());
                                    }
                                    jobLog2Control.flush();

                                    if (submitTask.killed) {
                                        throw new CancellationException("task is cancelled");
                                    }

                                    HttpWriter httpWriter=new HttpWriter(client2WorkTransport);
                                    httpWriter.writeAndCommit(outStore.sdList(),outStore,taskId,clientNode,worker);
                                    logLink("task completed, and all data pushed to worker, wait worker generate xlsx???")
                                            .log(log::info).log(jobLog2Control::log);
                                    return null;
                                }
                                , client2WorkTransport
                                ,clientNodeTaskRegisterRequest
                                ,scheduledThreadPoolExecutor
                                ,jobTaskManager,submitTask);
                            }
                            catch (Throwable e){
                                logLink("error: "+e.getMessage(),e).logError(log::error).log(jobLog2Control::log);
                            }
                            finally {
                                try {
                                    scheduledThreadPoolExecutor.shutdown();
                                } catch (Exception e) {
                                    logLink("error: "+e.getMessage(),e).logError(log::error).log(jobLog2Control::log);
                                }
                                submitTaskMap.remove(taskKey);
                                executingJobTaskIdSet.remove(taskLockRequest.getTaskId());
                                // flush net log
                                try {
                                    log.info(JSONAccessor.impl().format(submitTask.getTaskModel()));
                                    jobLog2Control.log(JSONAccessor.impl().format(submitTask.getTaskModel()));
                                    jobLog2Control.log(taskKey + " > completed: " + formatTime(LocalDateTime.now()));
                                    jobLog2Control.flush();
                                    jobLog2Control.close();
                                } catch (Exception e) {
                                    logLink("error: "+e.getMessage(),e).logError(log::error).log(jobLog2Control::log);
                                }
                                // on close
                                try {
                                    jobOutTaskFlow.close();
                                } catch (Exception e) {
                                    logLink("error: "+e.getMessage(),e).logError(log::error).log(jobLog2Control::log);
                                }
                            }
                        }));

                        taskSubmit=true;
                        submitTask.setFuture(future);
                        taskModel.setAcceptTime(System.currentTimeMillis());
                        taskModel.setAcceptTimeStr(formatTime(LocalDateTime.now()));
                        submitTaskMap.put(taskKey,submitTask);
                        logLink("OK, the task is scheduled: "+taskKey).log(log::info).log(client2ControlLog::log);
                        //OK ,  resource is available , executor accepts the task
                        taskSubmitResponse.setSubmitted(true);
                    } catch (RejectedExecutionException e) {
                        // resource is unavailable
                        executingJobTaskIdSet.remove(taskLockRequest.getTaskId());
                        taskSubmitResponse.setSubmitted(false);
                        taskSubmitResponse.setDesc("Rejected, resource is unavailable ");
                        throw convert2RuntimeException(e);
                    }

                }
                catch (Exception e){
                    logLink("error: "+e.getMessage(),e).logError(log::error).log(client2ControlLog::log);
                    client2ControlLog.flush();
                    errorStackList(e).forEach(client2ControlLog::log);
                    client2ControlLog.flush();
                    throw convert2RuntimeException(e);
                }
                finally {
                    if (!taskSubmit) {
                        jobOutTaskFlow.close();
                    }
                    client2ControlLog.close();
                }
                return null;
            },10,TimeUnit.SECONDS);

        }
        catch (Exception e) {
            log.error(e.getMessage(),e);
        }
        finally {
            try {
                client2ControlLog.close();
            }
            catch (Exception e){
                log.error(e.getMessage(),e);
            }
            finally {
                MDC.remove("traceId");
                MDC.remove("requestId");
            }
        }
        return taskSubmitResponse;
    }

    @RequestMapping("/cancel")
    @ResponseBody
    public TaskCancelResponse cancel(@RequestBody TaskCancelRequest taskCancelRequest){
        TaskCancelResponse taskCancelResponse=new TaskCancelResponse();
        taskCancelResponse.setApp(clientNodeProvider.clientNode().app());
        taskCancelResponse.setNode(ClientNode.UUID_STR);

        long taskId = taskCancelRequest.getTaskId();
        if (!executingJobTaskIdSet.contains(taskId)) {
            taskCancelResponse.setFound(false);
            taskCancelResponse.setSuccess(false);
            taskCancelResponse.setDesc("cannot find task");
            return taskCancelResponse;
        }

        SubmitTask submitTask = submitTaskMap.get("task_" +taskId);
        if (submitTask==null) {
            taskCancelResponse.setFound(false);
            taskCancelResponse.setSuccess(false);
            taskCancelResponse.setDesc("cannot find task");
            return taskCancelResponse;
        }
        // the task need be cancelled
        submitTask.setKilled(true);
        submitTask.runtimeStat.cancel();

        taskCancelResponse.setFound(true);
        Future<?> future = submitTask.future;
        if (future==null) {
            taskCancelResponse.setSuccess(false);
            taskCancelResponse.setDesc("cannot find io thread");
            return taskCancelResponse;
        }

        future.cancel(true);
        taskCancelResponse.setSuccess(future.isCancelled());
        taskCancelResponse.setDesc("done cancel");
        return taskCancelResponse;
    }

    @RequestMapping("/getStack")
    @ResponseBody
    public TaskStackGetResponse getStack(@RequestBody TaskStackGetRequest taskStackGetRequest){
        TaskStackGetResponse taskStackGetResponse=new TaskStackGetResponse();
        long taskId = taskStackGetRequest.getTaskId();
        if (!executingJobTaskIdSet.contains(taskId)) {
            taskStackGetResponse.setFound(false);
            taskStackGetResponse.setSuccess(false);
            taskStackGetResponse.setDesc("cannot find task");
            return taskStackGetResponse;
        }

        SubmitTask submitTask = submitTaskMap.get("task_" +taskId);
        if (submitTask==null) {
            taskStackGetResponse.setFound(false);
            taskStackGetResponse.setSuccess(false);
            taskStackGetResponse.setDesc("cannot find task");
            return taskStackGetResponse;
        }

        Thread ioThread = submitTask.ioThread;
        if (ioThread==null) {
            taskStackGetResponse.setFound(false);
            taskStackGetResponse.setSuccess(false);
            taskStackGetResponse.setDesc("cannot find io thread");
            return taskStackGetResponse;
        }

        taskStackGetResponse.setFound(true);
        taskStackGetResponse.setSuccess(true);
        StackTraceElement[] stackTraceElements = ioThread.getStackTrace();
        List<String> statckList=new ArrayList<>(stackTraceElements.length);
        for (StackTraceElement stackTraceElement : stackTraceElements) {
            statckList.add(stackTraceElement.toString());
        }
        taskStackGetResponse.setThread(ioThread.getName());
        taskStackGetResponse.setStackList(statckList);
        return taskStackGetResponse;
    }


    @RequestMapping("/capture")
    @ResponseBody
    public TaskCaptureResponse capture(@RequestBody TaskCaptureRequest taskCaptureRequest){
        TaskCaptureResponse taskCaptureResponse=new TaskCaptureResponse();
        long taskId = taskCaptureRequest.getTaskId();
        taskCaptureResponse.setTaskId(taskId);

        if (!executingJobTaskIdSet.contains(taskId)) {
            taskCaptureResponse.setFound(false);
            taskCaptureResponse.setSuccess(false);
            taskCaptureResponse.setDesc("cannot find task");
            return taskCaptureResponse;
        }

        SubmitTask submitTask = submitTaskMap.get("task_" +taskId);
        if (submitTask==null) {
            taskCaptureResponse.setFound(false);
            taskCaptureResponse.setSuccess(false);
            taskCaptureResponse.setDesc("cannot find task");
            return taskCaptureResponse;
        }

        OutStore outStore = submitTask.outStore;
        if (outStore==null) {
            taskCaptureResponse.setFound(false);
            taskCaptureResponse.setSuccess(false);
            taskCaptureResponse.setDesc("cannot find out store");
            return taskCaptureResponse;
        }

        taskCaptureResponse.setFound(true);
        taskCaptureResponse.setSuccess(true);

        try {
            Map<String, List<Map<String, Object>>> listMap = outStore.capture(taskCaptureRequest.getN());
            taskCaptureResponse.setData(listMap);
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            taskCaptureResponse.setDesc(e.getMessage());
        }
        return taskCaptureResponse;
    }

    @Data
    public static class SubmitTask{

        private long taskId;

        private Future<?> future;

        private TaskModel taskModel;

        private boolean killed;

        @JsonIgnore
        private transient OutStore.RuntimeStat runtimeStat;

        @JsonIgnore
        private transient Thread ioThread;

        /**
         * @since 5.2
         */
        @JsonIgnore
        private transient OutStore outStore;

    }

    public interface JobTaskManager{

        void cancel(long taskId);

        default List<TaskModel> taskList(){throw new UnsupportedOperationException();}

        static JobTaskManager getOrCreate(){
            return instance;
        }

        default int receivedTaskCount(){return 0;};

    }

    @Override
    public void afterPropertiesSet() throws Exception {
        instance=simpleJobTaskManager;
    }

    private static JobTaskManager instance;

    final private SimpleJobTaskManager simpleJobTaskManager=new SimpleJobTaskManager();

    private class SimpleJobTaskManager implements JobTaskManager{

        @Override
        public void cancel(long taskId) {
            SubmitTask submitTask = submitTaskMap.get("task_" +taskId);
            if (submitTask==null) {
                return;
            }
            submitTask.setKilled(true);
            if (submitTask.future==null) {
                return;
            }
            submitTask.future.cancel(true);
            log.info("cancel the task");
        }

        @Override
        public List<TaskModel> taskList() {
            List<TaskModel> taskModelList=new ArrayList<>(submitTaskMap.size());
            for (SubmitTask submitTask : submitTaskMap.values()) {
                TaskModel taskModel = submitTask.getTaskModel();
                try {
                    TransientFunc transientFunc = taskModel.getTransientFunc();
                    transientFunc.run();
                }catch (Exception e){
                    log.error(e.getMessage(),e);
                }
                taskModelList.add(taskModel);
            }
            return taskModelList;
        }

        @Override
        public int receivedTaskCount() {
            return receivedTaskCount.get();
        }


    }


    private <T> T doTask(Callable<T> callable, Client2WorkTransport client2WorkTransport
            , ClientNodeTaskRegisterRequest clientNodeTaskRegisterRequest
            , ScheduledExecutorService taskHeartbeat
            , JobTaskManager jobTaskManager
            , SubmitTask submitTask
    ){
        final Runnable runnable = catchRunnable(new CatchRunnable() {

            int failCount = 0;

            @Override
            public void run() throws Exception {
                long taskId = submitTask.taskId;
                TaskModel taskModel = submitTask.taskModel;
                try {
                    TransientFunc transientFunc = taskModel.getTransientFunc();
                    transientFunc.run();
                }catch (Exception e){
                    log.error(e.getMessage(),e);
                }
                TaskModel.ExportClientStat exportClientStat = taskModel.getExportClientStat();
                clientNodeTaskRegisterRequest.setExportClientStat(exportClientStat);

                ClientNodeTaskRegisterResponse clientNodeTaskRegisterResponse = client2WorkTransport.exportHeartbeat(clientNodeTaskRegisterRequest);
                log.info("heartbeat from work(" + clientNodeTaskRegisterRequest.getWorkApp() + ":" + clientNodeTaskRegisterRequest.getWorkNode() + ") : " + JSONAccessor.impl().format(clientNodeTaskRegisterResponse));
                if (clientNodeTaskRegisterResponse == null || !clientNodeTaskRegisterResponse.isSuccess()) {
                    failCount++;
                    if (failCount > 5) {
                        //OK , we lose connect to work... , break the task ???
                        // true , if memory , nfs ???
                        jobTaskManager.cancel(taskId);
                    }
                    return;
                }
                failCount = 0;
            }
        });
        ScheduledFuture<?> scheduledFuture = taskHeartbeat.scheduleWithFixedDelay(runnable, 0, 5, TimeUnit.SECONDS);
        try {
            return callable.call();
        } catch (Exception e) {
            throw convert2RuntimeException(e);
        } finally {
            try {
                runnable.run();
            }
            finally {
                scheduledFuture.cancel(true);
                log.info("cancel heartbeat thread");
            }
        }

    }


}
