package com.ovopark.iohub.sdk.client;

import com.ovopark.iohub.sdk.model.ClientConfigRequest;
import com.ovopark.iohub.sdk.model.ClientConfigResponse;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import com.ovopark.kernel.shared.vclient.ClientNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.ovopark.kernel.shared.Util.isEmpty;
import static com.ovopark.kernel.shared.Util.isNotEmpty;

@Slf4j
@Component
public class NacosControlNode implements ControlNode, InitializingBean {

    @Autowired
    private ServiceInstance serviceInstance;

    @Autowired
    private DiscoveryClient discoveryClient;

    @Value("${spring.application.name}")
    private String app;

    final static ScheduledExecutorService scheduledExecutorService= Executors.newScheduledThreadPool(1);

    private static final String PATH="/iohub-control/jobClient/control-config";

    private String ip="127.0.0.1";

    private int port=13910;

    final boolean verbose= Config.ConfigPriority.option().getBoolean("iohub.client.node.verbose",false);

    private int uiPort=13900;

    @Override
    public void afterPropertiesSet() throws Exception {

        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setReadTimeout(45_000);//ms
        factory.setConnectTimeout(15_000);//ms
        RestTemplate restTemplate = new RestTemplate(factory);

        ClientConfigRequest clientConfigRequest=new ClientConfigRequest();
        clientConfigRequest.setApp(app);
        clientConfigRequest.setNode(ClientNode.UUID_STR);

        Util.schedule(scheduledExecutorService, () -> {
            List<ServiceInstance> serviceInstanceList = new ArrayList<>(discoveryClient.getInstances("iohub-control"));

            if (isEmpty(serviceInstanceList)) {
                return;
            }

            ServiceInstance instance = serviceInstanceList.get(0);
            URI instanceUri = instance.getUri();

            HttpHeaders headers = new HttpHeaders();
            headers.add("Content-Type","application/json");
            HttpEntity<String> formEntity = new HttpEntity<>(JSONAccessor.impl().format(clientConfigRequest), headers);
            try {
                ResponseEntity<String> responseEntity = restTemplate.postForEntity(instanceUri + PATH, formEntity, String.class);
                int statusCodeValue = responseEntity.getStatusCodeValue();
                if (statusCodeValue==200) {
                    ClientConfigResponse clientConfigResponse = JSONAccessor.impl().read(responseEntity.getBody(), ClientConfigResponse.class);
                    if (isNotEmpty(clientConfigResponse.getControlIp())) {
                        this.ip=clientConfigResponse.getControlIp();
                    }

                    if (clientConfigResponse.getControlPort()>0) {
                        this.port=clientConfigResponse.getControlPort();
                    }
                    if (clientConfigResponse.getControlUiPort()>0) {
                        this.uiPort=clientConfigResponse.getControlUiPort();
                    }
                    if (verbose) {
                        log.info(PATH+ " result: "+ responseEntity.getBody());
                    }
                }
            } catch (RestClientException e) {
                if (verbose) {
                    log.error(e.getMessage(),e);
                }
                else {
                    log.debug(e.getMessage());
                }
            }
        },5, TimeUnit.SECONDS, e->{
            if (verbose) {
                log.error(e.getMessage(),e);
            }
            return true;
        },()->true);


    }

    @Override
    public String ip() {
        return ip;
    }

    @Override
    public int port() {
        return port;
    }

    @Override
    public int uiPort() {
        return uiPort;
    }
}
