package com.ovopark.iohub.sdk.client.instream.test;

import com.ovopark.iohub.sdk.client.*;
import com.ovopark.iohub.sdk.client.instream.JobInTaskFlow;
import com.ovopark.iohub.sdk.client.instream.JobInTaskFlowProvider;
import com.ovopark.iohub.sdk.client.outstream.RequestParamBody;
import com.ovopark.iohub.sdk.model.AppNode;
import com.ovopark.iohub.sdk.model.instream.ImportPushDataRequest;
import com.ovopark.iohub.sdk.model.instream.ImportReplyRequest;
import com.ovopark.iohub.sdk.model.proto.DataExistStrategy;
import com.ovopark.iohub.sdk.model.proto.ImportPreDefConf;
import com.ovopark.iohub.sdk.model.proto.LimitLogger;
import com.ovopark.iohub.sdk.model.proto.Segment;
import com.ovopark.iohub.sdk.model.proto.internal.ImportPreDefConfImpl;
import com.ovopark.iohub.sdk.model.proto.internal.ReplyModel;
import com.ovopark.iohub.sdk.model.proto.internal.ReplyRowModel;
import com.ovopark.iohub.sdk.model.proto.internal.SegmentImpl;
import com.ovopark.iohub.sdk.model.test.ReplyRenderResponse;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import com.ovopark.kernel.shared.stream.Stream;
import com.ovopark.module.shared.BaseResult;
import com.ovopark.module.shared.Session;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.ovopark.kernel.shared.Util.*;

@IOHubClientActive
@Slf4j
@RestController("com.ovopark.iohub.sdk.client.instream.test.ImportLocalTestEndpoint")
@RequestMapping("/feign/iohub-job/processing/only-test/import")
public class ImportLocalTestEndpoint {


    final String workIp= Config.ConfigPriority.option().getString("iohub.client.io.work.ip","8.139.4.155");

    final int workPort= Config.ConfigPriority.option().getInt("iohub.client.io.work.port",13950);

    @Autowired
    private Client2ControlTransport client2ControlTransport;
    @Autowired
    private IOHubClientConfig ioHubClientConfig;

    @Autowired
    private JobInTaskFlowProvider jobInTaskFlowProvider;

    @Autowired
    private ClientNodeProvider clientNodeProvider;


    @RequestMapping("/submit")
    public BaseResult<?> submit(ImportTaskVo importTaskVo
            , @RequestHeader(value = "ovo-authorization",required = false) String token){

        final String uri = importTaskVo.getUri();
        final String args = importTaskVo.getParams();

        String localFile = write2LocalFile(0, importTaskVo);

        SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
        factory.setReadTimeout(45_000);//ms
        factory.setConnectTimeout(15_000);//ms
        RestTemplate restTemplate = new RestTemplate(factory);
        restTemplate.getMessageConverters().add(0,new StringHttpMessageConverter(StandardCharsets.UTF_8));
        AppNode appNode=new AppNode();
        appNode.setIp(workIp);
        appNode.setPort(workPort);
        appNode.setServletPath("iohub-work");
        Client2WorkRestClient client2WorkRestClient=new Client2WorkRestClient(appNode,restTemplate);

        JobInTaskFlow<RequestParamBody> jobInTaskFlow = jobInTaskFlowProvider.find(uri);
        if (jobInTaskFlow==null) {
            jobInTaskFlow=jobInTaskFlowProvider.find("/iohub-test/import/testAnyImport");
            if (jobInTaskFlow==null) {
                return BaseResult.error(null,"cannot find spring bean: "+uri);
            }
            log.info("found flow impl: "+jobInTaskFlow.getClass().getName());
        }

        jobInTaskFlow.prepared();
        Session si = Session.getOrCreate().get();
        if (si==null && isNotEmpty(token)) {
            si=client2WorkRestClient.parseSessionOnlyForTest(token);
        }

        if (si==null) {
            log.error("cannot parse session , invalid token ???: "+token);
            return BaseResult.error(null,"cannot parse session , invalid token ???: "+token);
        }

        RequestParamBody requestParamBody = isEmpty(args)?null:jobInTaskFlow.requestParamBody(args, si);
        JobInTaskFlow.ImportTaskInfoProvider importTaskInfoProvider = jobInTaskFlow.importTaskInfoProvider(requestParamBody, si);

        // request param
        ImportPreDefConfImpl importPreDefConf=new ImportPreDefConfImpl();
        importPreDefConf.setDataExistStrategy(DataExistStrategy.valueOf(importTaskVo.getDataExistStrategy()));

        ReplyModel replyModel=new ReplyModel();
        final Map<String, ReplyRowModel> replyModelIndex=new HashMap<>();

        AtomicBoolean cancelled=new AtomicBoolean(false);
        JobInTaskFlow.ImportContext importContext =new JobInTaskFlow.ImportContext() {
            @Override
            public void cancel() {
                cancelled.set(true);
            }

            @Override
            public synchronized void replyWithSegment(int segment, int fromRow, int toRow, String desc, JobInTaskFlow.Result result) {
                if (result== JobInTaskFlow.Result.EXISTS) {
                    replyModel.setRepeatCount(replyModel.getRepeatCount()+1);
                }
                else if (result== JobInTaskFlow.Result.SUCCESS) {
                    replyModel.setSuccessCount(replyModel.getSuccessCount()+1);
                }
                else if (result== JobInTaskFlow.Result.FAIL) {
                    replyModel.setErrorCount(replyModel.getErrorCount()+1);
                }
                String key=segment+":"+fromRow+":"+toRow;
                ReplyRowModel rrm = replyModelIndex.get(key);
                if (rrm==null) {
                    ReplyRowModel replyRowModel = ReplyRowModel.replyWithSegment(segment, fromRow, toRow, desc);
                    replyModel.add(replyRowModel);
                    replyModelIndex.put(key,replyRowModel);
                }
                else {
                    if (rrm.getDescList().size()>100) {
                        return;
                    }
                    rrm.getDescList().add(desc);
                }
            }
        };

        JobInTaskFlow.ImportConf<RequestParamBody> importConf=new JobInTaskFlow.ImportConf<RequestParamBody>() {
            @Override
            public RequestParamBody requestParam() {
                return requestParamBody;
            }

            @Override
            public ImportPreDefConf importPreDefConf() {
                return importPreDefConf;
            }
        };

        ImportPushDataRequest importPushDataRequest = client2WorkRestClient.parseOnlyForTest(localFile, uri,importTaskVo.getParams());

        if (importPushDataRequest==null) {
            log.error("cannot parse excel , invalid excel ???, local file: "+localFile);
            return BaseResult.error(null,"cannot parse excel , invalid excel ???, local file: "+localFile);
        }
        final List<SegmentImpl> segmentList = importPushDataRequest.getSegmentList();
        int segmentCount = segmentList.size();
        int rowCount=0;
        for (SegmentImpl segment : segmentList) {
            rowCount+=segment.size();
        }
        int finalRowCount = rowCount;
        Session finalSi = si;
        LimitLogger limitLogger=new LimitLogger() {
            @Override
            public void log(String content) {
                log.info(content);
            }
        };
        JobInTaskFlow.DataRowStream<RequestParamBody> dataRowStream =new JobInTaskFlow.DataRowStream<RequestParamBody>() {
            final AtomicBoolean reset=new AtomicBoolean(true);

            @Override
            public JobInTaskFlow.DataRowStream<RequestParamBody> reset() {
                cancelled.set(false);
                return this;
            }

            @Override
            public void mark(JobInTaskFlow.StreamProcessResult streamProcessResult, String msg) {
                if (streamProcessResult== JobInTaskFlow.StreamProcessResult.SUCCESS) {
                    replyModel.setStatus(1);
                }
                else if (streamProcessResult== JobInTaskFlow.StreamProcessResult.FAIL) {
                    replyModel.setStatus(-1);
                    replyModel.setFailMsg(msg);
                }
            }

            @Override
            public LimitLogger limitLogger() {
                return limitLogger;
            }

            @Override
            public void subscribe(JobInTaskFlow.DataRowListener dataRowListener) {
                if (!reset.get()) {
                    throw new IllegalStateException("need reset stream before subscribe it.");
                }
                for (int segmentIndex = 0; segmentIndex < segmentList.size(); segmentIndex++) {
                    if (cancelled.get()) {
                        Stream.from("cancel the import flow, segment: "+segmentIndex).subscribe(s -> {
                            log.info(s);
                        });
                        return;
                    }
                    Segment segment=segmentList.get(segmentIndex);
                    final Map<String, Segment.Header.Cell> headerCellMap=new HashMap<>();
                    segment.header().scanUp2Down(new Segment.Header.Scan() {
                        @Override
                        public void scan(Segment.Header.Cell cell) {
                            headerCellMap.put(cell.path(),cell);
                        }
                    });
                    JobInTaskFlow.HeaderLookup headerLookup=new JobInTaskFlow.HeaderLookup() {
                        @Override
                        public Segment.Header.Cell lookup(String path) {
                            return headerCellMap.get(path);
                        }
                    };
                    List<Map<String, Object>> mapList = segment.rowList();
                    for (int rowIndex = 0; rowIndex < mapList.size(); rowIndex++) {
                        if (cancelled.get()) {
                            Stream.from("cancel the import flow, segment: "+segmentIndex).subscribe(s -> {
                                log.info(s);
                            });
                            return;
                        }

                        Map<String, Object> data = mapList.get(rowIndex);
                        boolean allEmpty=true;
                        for (Object v : data.values()) {
                            if (isNotEmpty((String)v)) {
                                allEmpty=false;
                                break;
                            }
                        }

                        if (allEmpty) {
                            continue;
                        }

                        JobInTaskFlow.DataRowImpl dataRow=new JobInTaskFlow.DataRowImpl();
                        dataRow.setSegment(segmentIndex);
                        dataRow.setRowNum(rowIndex);
                        dataRow.setData(data);
                        dataRow.setHeaderLookup(headerLookup);
                        dataRowListener.onRow(dataRow, importContext);
                    }
                }
                reset.set(false);
            }

            @Override
            public int segmentCount() {
                return segmentCount;
            }

            @Override
            public long rowCount() {
                return finalRowCount;
            }

            @Override
            public long rowCount(int segment) {
                return segmentList.get(segment).size();
            }

            @Override
            public Session session() {
                return finalSi;
            }

            @Override
            public JobInTaskFlow.ImportConf<RequestParamBody> importConf() {
                return importConf;
            }
        };
        jobInTaskFlow.execute(dataRowStream);

        log.info("replyModel size: "+replyModel.getReplyMap().size());

        ImportReplyRequest importReplyRequest=new ImportReplyRequest();
        importReplyRequest.setTaskId(importPushDataRequest.getTaskId());
        importReplyRequest.setApp(null);
        importReplyRequest.setNode(null);
        importReplyRequest.setWorkApp(null);
        importReplyRequest.setWorkNode(null);

        importReplyRequest.setReplyModel(replyModel);

        ReplyRenderResponse importReplyResponse = client2WorkRestClient.renderReplyOnlyForTest(importReplyRequest);
        log.info("reply result: "+ JSONAccessor.impl().format(importReplyResponse));
        log.info("reply url: "+importReplyResponse.getUrl());

        return BaseResult.success(true);
    }




    private String write2LocalFile(long jobTaskId,ImportTaskVo importTaskVo) {
        String basePath=ioHubClientConfig.nfsPath();
        String osName = System.getProperty("os.name").toLowerCase();
        if (osName.contains("win")) { // 包含 "win" 可以匹配 "Windows", "windows", "Win", 等
            log.info("This is a Windows operating system.");
            if (basePath.startsWith("/")) {
                basePath="d:/iohub";
            }
        } else {
            log.info("This is not a Windows operating system.");
            log.warn("you must config: 'iohub.client.io.nfsPath=/xxx' in environment");
        }
        log.warn(basePath+" , directory exists????");
        String jobTaskFilePath = basePath+"/test-client"
                +"/"+ formatTime(LocalDateTime.now(), yyyyMMddHHmmss)
                + "/" + jobTaskId;
        File file=new File(jobTaskFilePath);
        if (!file.exists()) {
            file.mkdirs();
        }
        String filePath= jobTaskFilePath
                + "/import-"+ Util.uniqueFirstPart()+".xlsx";
        log.info("to write data to disk: "+filePath);
        try {
            importTaskVo.getFile().transferTo(new File(filePath));
        } catch (IOException e) {
            throw convert2RuntimeException(e);
        }
        log.info("write data to disk ("+filePath+"), cost time ");
        return filePath;
    }


}
