package com.ovopark.iohub.sdk.model.proto;

import com.ovopark.kernel.shared.Model;
import com.ovopark.kernel.shared.Util;
import lombok.Data;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;

import static com.ovopark.kernel.shared.Util.isEmpty;
import static com.ovopark.kernel.shared.Util.isNotEmpty;

@Data
public class HeaderImpl implements Segment.Header , Model {

    private boolean committed;

    private boolean fixed;

    final private List<HeaderCellImpl> cellList = new ArrayList<>();

    private int height;

    private int defaultColumnWidth=15;

    private int maxHeadIndex;

    private int maxPossibleHeadIndex;

    @Override
    public int maxHeadIndex() {
        return maxHeadIndex;
    }

    @Override
    public void maxPossibleHeadIndex(int maxPossibleHeadIndex) {
        this.maxPossibleHeadIndex=maxPossibleHeadIndex;
    }

    @Override
    public int maxPossibleHeadIndex() {
        return maxPossibleHeadIndex;
    }

    public void root(String property) {
        if (committed) {
            throw new IllegalArgumentException("header was fixed, cannot modify");
        }
        upset(null,property,null,null);
    }

    public void root(List<String> propertyList) {
        if (committed) {
            throw new IllegalArgumentException("header was fixed, cannot modify");
        }
        propertyList.forEach(this::root);
    }

    public void i18n(I18n i18n) {
        func(i18n::i18n, cellList, (s, cell) -> cell.setName(s),headerCell -> isNotEmpty(headerCell.getI18Key())?headerCell.getI18Key():headerCell.getPath());
    }

    @Override
    public void width(Width width) {
        func(width::width, cellList, (s, cell) -> cell.setWidth(s),null);
    }


    interface CellFuncSet<R>{
        void set(R r, HeaderCellImpl cell);
    }

    private <R> void func(Function<String,R> func, List<HeaderCellImpl> cellList, CellFuncSet<R> cellFuncSet,Function<HeaderCellImpl,String> pathGet) {
        for (HeaderCellImpl cell : cellList) {
            R apply = func.apply(pathGet==null?cell.getPath():pathGet.apply(cell));
            cellFuncSet.set(apply,cell);
            if (isNotEmpty(cell.getCellList())) {
                func(func, cell.getCellList(),cellFuncSet,pathGet);
            }
        }
    }


    public void children(String path, List<String> propertyList) {
        for (String p : propertyList) {
            upset(path,p,null,null);
        }
    }

    @Override
    public void children(String path, String property) {
        children(path,property,null);
    }

    @Override
    public void children(String path, String property, Width width) {
        children(path,property,width,null);
    }

    @Override
    public void children(String path, String property, Width width, I18n i18n) {
        children(path,property,width,i18n,null);
    }

    @Override
    public void children(String path, String property, Width width, I18n i18n, CellType cellType) {
        upset(path,property,width,i18n,cellType);
    }

    @Override
    public void upset(String path, String property, Width width) {
        upset(path,property,width,null);
    }

    @Override
    public void upset(String path, String property, Width width, I18n i18n) {
        upset(path,property,width,i18n,null);
    }

    @Override
    public void upset(String path, String property, Width width, I18n i18n,CellType cellType) {
        upset0(path, property, width, i18n, cellType,c->{});
    }

    private void upset0(String path, String property, Width width, I18n i18n, CellType cellType, Consumer<HeaderCellImpl> headerCellConsumer) {
        if (committed) {
            throw new IllegalArgumentException("header was fixed, cannot modify");
        }

        if (isEmpty(path)) {
            HeaderCellImpl cell = new HeaderCellImpl();
            cell.setProperty(property);
            cell.setLevel(0);
            cell.setPath(property);
            if (width !=null) {
                cell.setWidth(width.width(property));
            }
            if (i18n != null) {
                cell.setName(i18n.i18n(property));
            }
            if (cellType != null) {
                cell.setCellType(cellType.cellType(property));
            }
            headerCellConsumer.accept(cell);
            cellList.add(cell);
            return;
        }

        HeaderCellImpl pathCell = null;
        List<HeaderCellImpl> queryCellList = cellList;
        String[] pl = path.split("\\.");
        for (int i = 0; i < pl.length; i++) {
            String p = pl[i];
            HeaderCellImpl next = null;
            for (HeaderCellImpl cell : queryCellList) {
                if (cell.getProperty().equals(p)) {
                    next = cell;
                    break;
                }
            }

            if (next == null) {
                throw new IllegalArgumentException("sub path: '" + p + "' is missing? " + path);
            }

            queryCellList = next.getCellList();
            pathCell = next;
        }

        if (pathCell == null) {
            throw new IllegalArgumentException("path is missing: " + path);
        }

        List<HeaderCellImpl> cellList =pathCell.getCellList();
        if (cellList==null) {
            cellList=new ArrayList<>();
        }
        HeaderCellImpl cell = new HeaderCellImpl();
        cell.setProperty(property);
        cell.setPath(path + "." + property);
        cell.setLevel(pl.length);
        if (width !=null) {
            cell.setWidth(width.width(cell.getPath()));
        }
        if (i18n != null) {
            cell.setName(i18n.i18n(cell.getPath()));
        }
        if (cellType != null) {
            cell.setCellType(cellType.cellType(cell.getPath()));
        }
        headerCellConsumer.accept(cell);
        height = Math.max(height, cell.getLevel());
        cellList.add(cell);

        pathCell.setCellList(cellList);
    }

    @Override
    public void defaultColumnWidth(int width) {
        this.defaultColumnWidth=width;
    }

    @Override
    public int defaultColumnWidth() {
        return defaultColumnWidth;
    }

    @Override
    public void all(Class<? extends Model> clazz) {
        all(clazz,a->null);
    }

    @Override
    public void all(Class<? extends Model> clazz, I18n i18n) {
        boolean javaTypeAsDataType=true;
        HeaderDef headerDef = clazz.getDeclaredAnnotation(HeaderDef.class);
        if (headerDef != null) {
            setDefaultColumnWidth(headerDef.defaultColumnWidth()>0?headerDef.defaultColumnWidth():15);
            if (headerDef.fixed()) {
                fixed();
            }
            javaTypeAsDataType = headerDef.javaTypeAsDataType();
        }
        final boolean javaTypeAsDataTypeDef=javaTypeAsDataType;
        Map<String, Field> fieldMap = Util.flatProperty(clazz,
                field -> field.getDeclaredAnnotation(ExcludeInHeader.class)==null
                        && !CellValue.class.isAssignableFrom(field.getDeclaringClass()));
        for (Map.Entry<String, Field> entry : fieldMap.entrySet()) {
            String c = entry.getKey();
            String parentPath = c.substring(0, Math.max(c.lastIndexOf("."), 0));
            String subPath=c.substring(Math.max(c.lastIndexOf(".")+1,0));
            Field field = entry.getValue();
            CellDef cellDef = field.getDeclaredAnnotation(CellDef.class);
            upset0(parentPath, subPath, new Width() {
                @Override
                public int width(String path) {
                    return cellDef==null?0:cellDef.width();
                }
            }, null, new CellType() {
                @Override
                public CellDataType cellType(String path) {
                    CellDataType def= cellDef == null ? CellDataType.NONE : cellDef.cellDataType();
                    if (def== CellDataType.NONE && javaTypeAsDataTypeDef) {
                        Class<?> type = field.getType();
                        if (Boolean.class.isAssignableFrom(type) || boolean.class==type) {
                            def=CellDataType.BOOLEAN;
                        }
                        else if (Number.class.isAssignableFrom(type)
                                || int.class==type || long.class==type
                                || short.class==type || byte.class==type
                                || float.class==type || double.class==type
                        ) {
                            def=CellDataType.NUMERIC;
                        }
                        else {
                            def=CellDataType.STRING;
                        }
                    }
                    return def;
                }
            },headerCell -> {
                headerCell.setI18Key(cellDef==null?null:cellDef.i18Key());
                String showName = cellDef == null ? null : cellDef.showName();
                if (isNotEmpty(showName)) {
                    //set as real show name
                    headerCell.setName(showName);
                }

                else if(isNotEmpty(headerCell.getI18Key())){
                    showName = i18n.i18n(headerCell.getI18Key());
                    headerCell.setName(showName);
                }

                else {
                    showName = i18n.i18n(headerCell.getPath());
                    headerCell.setName(showName);
                }

            });
        }

    }

    @Override
    public void fixed() {
        this.fixed=true;
    }

    public void commit() {

        if (committed) {
            return;
        }

        committed = true;

        for (HeaderCellImpl cell : cellList) {
            int i = cell.calLeafCount();
            cell.setLeafCount(i);
        }


        for (int i = 0; i < cellList.size(); i++) {
            HeaderCellImpl cell = cellList.get(i);
            if (i==0) {
                cell.calPosition(0);
            }
            else {
                HeaderCellImpl b = cellList.get(i - 1);
                cell.calPosition(b.getPosition()+b.leafCount());
            }
        }

        scanUp2Down(new Scan() {
            @Override
            public void scan(Cell cell) {
                maxHeadIndex =Math.max(maxHeadIndex,cell.position());
            }
        });

    }

    public int height() {
        return height;
    }

    public void scanUp2Down(Scan scan) {
        scanUp2Down0(scan, cellList);
    }

    private void scanUp2Down0(Scan scan, List<HeaderCellImpl> cellList) {
        List<HeaderCellImpl> nextList = new ArrayList<>();
        for (HeaderCellImpl cell : cellList) {
            scan.scan(cell);
            if (isNotEmpty(cell.getCellList())) {
                nextList.addAll(cell.getCellList());
            }
        }
        if (isNotEmpty(nextList)) {
            scanUp2Down0(scan, nextList);
        }
    }


}
