package com.ovopark.iohub.sdk.model.proto;

import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Model;
import lombok.Getter;

import java.util.*;

public class InMemoryOutStore implements OutStore {

    @Getter
    private String fileName;

    private final List<Segment> segmentList;

    private boolean committed;

    final Stat stat=new Stat();

    final RuntimeStat runtimeStat=new RuntimeStatImpl();

    @Getter
    final private Feature feature=new Feature();

    @Getter
    final private Map<String,Group> groupMap=new HashMap<>();

    public InMemoryOutStore(int size) {
        this.segmentList = new ArrayList<>(size);
    }

    @Override
    public SegmentWriter createSegment(String name) {
        return createSegment(name,10);
    }

    @Override
    public SegmentWriter createSegment(String name,int rowSize) {
        return createSegment(name,rowSize,"default");
    }

    @Override
    public SegmentWriter createSegment(String name, String group) {
        return createSegment(name,10,group);
    }

    @Override
    public SegmentWriter createSegment(String name, int rowSize, String group) {
        check();
        groupMap.putIfAbsent(group,new Group());
        SegmentWriterImpl segmentWriter = new SegmentWriterImpl(name, rowSize, group);
        runtimeStat.segmentAdd(group+":"+name);
        runtimeStat.segmentCountAdd(1);
        return segmentWriter;
    }

    @Override
    public void commit() {
        this.committed=true;
    }

    @Override
    public void close() {
        for (Segment segment : segmentList) {
            segment.commit();
        }
    }

    @Override
    public Segment segment(Segment.SD sd) {
        for (Segment segment : segmentList) {
            if (segment.name().equals(sd.getName()) && segment.group().equals(sd.getGroup())) {
                return segment;
            }
        }
        return null;
    }

    @Override
    public void fileName(String fileName) {
        this.fileName=fileName;
    }

    @Override
    public GroupOperation group(String group) {
        return new GroupOperation() {
            @Override
            public void fileName(String fileName) {
                groupMap.get(group).setFileName(fileName);
            }
        };
    }

    @Override
    public FeatureOperation feature() {
        return new FeatureOperation() {
            @Override
            public void supportRangeMerge(boolean rangeMerge) {
                feature.setRangeMerge(rangeMerge);
            }
        };
    }

    @Override
    public Stat stat() {
        return stat;
    }

    @Override
    public RuntimeStat runtimeStat() {
        return runtimeStat;
    }

    @Override
    public List<Segment.SD> sdList() {
        List<Segment.SD> sdList=new ArrayList<>(segmentList.size());
        for (Segment segment : segmentList) {
            Segment.SD sd=new Segment.SD();
            sd.setMemory(true);
            sd.setName(segment.name());
            sd.setGroup(segment.group());
            sdList.add(sd);
        }
        return sdList;
    }

    @Override
    public CellFile file(byte[] data, String name, String type) {
        CellFile cellFile=new CellFile();
        cellFile.setName(name);
        cellFile.setType(type);
        cellFile.setBase64(true);
        cellFile.setUri(Base64.getEncoder().encodeToString(data));
        return cellFile;
    }

    private void check(){
        if (committed) {
            throw new UnsupportedOperationException("all committed???");
        }
    }

    private class SegmentWriterImpl implements SegmentWriter{


        private final SegmentImpl segment;

        private boolean committed;

        final private int maxRowCount=10000;

        final private SegmentStat segmentStat=new SegmentStat();

        public SegmentWriterImpl(String name,int size,String group) {
            this.segment = new SegmentImpl(name,size,group);
        }

        @Override
        public synchronized void append(Map<String, Object> data) {
            check();
            segment.append(data);
            runtimeStat.rowCountAdd(1);
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
            if (segment.size()>maxRowCount) {
                throw new IllegalStateException("too many row: "+segment.size()+", max: "+maxRowCount);
            }
        }

        @Override
        public synchronized void append(List<Map<String, Object>> data) {
            check();
            for (Map<String, Object> datum : data) {
                segment.append(datum);
            }
            runtimeStat.rowCountAdd(data.size());
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
            if (segment.size()>maxRowCount) {
                throw new IllegalStateException("too many row: "+segment.size()+", max: "+maxRowCount);
            }
        }

        @Override
        public void meta(Map<String, Object> meta) {
            segment.setMeta(meta);
        }

        @Override
        public synchronized void append(Model model) {
            append(JSONAccessor.impl().read(JSONAccessor.impl().format(model)));
        }

        @Override
        public synchronized void commit() {
            if (committed) {
                return;
            }
            segmentList.add(segment);
            segment.header().commit();
            segment.commit();

            segmentStat.setRowCount(segment.getRowList().size());
            if (stat.getSegmentStatList()==null) {
                stat.setSegmentStatList(new ArrayList<>());
            }
            stat.getSegmentStatList().add(segmentStat);

            this.committed=true;
        }

        @Override
        public Segment.Title title() {
            return segment.getTitle();
        }


        private void check(){
            if (committed) {
                throw new UnsupportedOperationException("segment committed???");
            }
        }

        @Override
        public Segment.Header header() {
            return segment.header();
        }

        @Override
        public String group() {
            return segment.group();
        }
    }

}
