package com.ovopark.iohub.sdk.model.proto;

import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Model;
import com.ovopark.kernel.shared.Util;
import com.ovopark.kernel.shared.sequencefile.FileIO;
import com.ovopark.kernel.shared.sequencefile.SimpleFileIO;
import com.ovopark.kernel.shared.sequencefile.TransLogConfig;
import lombok.Data;
import lombok.Getter;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.ovopark.kernel.shared.Util.*;

/**
 *  FILE PATH , DATA VIA NFS
 */
public class NFSOutStore implements OutStore{

    @Getter
    private String fileName;

    @Getter
    private final String path;

    final private List<SegmentFileDescriptor> segmentFileDescriptorList;

    private boolean committed;

    final private List<SegmentFileWriter> segmentFileWriterList=new ArrayList<>();

    final private FileIO fileIO;

    @Getter
    final private String fileIOUrl;

    @Getter
    final private Map<String,Group> groupMap=new HashMap<>();

    final private RowTransLogConfig rowTransLogConfig;

    final Stat stat=new Stat();

    final RuntimeStat runtimeStat=new RuntimeStatImpl();

    @Getter
    final private Feature feature=new Feature();

    public NFSOutStore(String path,int size){
        this(path,size,null);
    }


    public NFSOutStore(String path,int size,RowTransLogConfig rowTransLogConfig) {
        this.path = path+"/"+Util.uniqueFirstPart();
        File file=new File(this.path);
        if (!file.exists()) {
            file.mkdirs();
        }
        segmentFileDescriptorList=new ArrayList<>(size);
        this.rowTransLogConfig=rowTransLogConfig==null?new RowTransLogConfig():rowTransLogConfig;
        TransLogConfig transLogConfig=new TransLogConfig();
        this.fileIOUrl=this.path+"/"+uniqueFirstPart()+"-file.bf";
        this.fileIO=new SimpleFileIO("nfs",fileIOUrl,transLogConfig,false);
    }

    @Override
    public SegmentWriter createSegment(String name) {
        return createSegment(name,1);
    }

    @Override
    public SegmentWriter createSegment(String name,int rowSize) {
        return createSegment(name,rowSize,"default");
    }

    @Override
    public SegmentWriter createSegment(String name, String group) {
        return createSegment(name,1,group);
    }

    @Override
    public SegmentWriter createSegment(String name, int rowSize, String group) {
        check();

        if (segmentFileWriterList.size()>256) {
            throw new IllegalArgumentException("exceed max segment size 256 ");
        }

        SegmentFileWriter segmentFileWriter = new SegmentFileWriter(name, group, rowSize);
        segmentFileWriterList.add(segmentFileWriter);
        groupMap.putIfAbsent(group,new Group());

        runtimeStat.segmentAdd(group+":"+name);
        runtimeStat.segmentCountAdd(1);
        return segmentFileWriter;
    }

    @Override
    public void commit() {
        committed=true;
    }

    @Override
    public void close() {
        Exception exception=null;
        for (SegmentFileWriter segmentFileWriter : segmentFileWriterList) {
            try {
                segmentFileWriter.close();
            }catch (Exception e){
                //
                exception=e;
            }
        }

        try {
            fileIO.close();
        } catch (Exception e) {
            //
            exception=e;
        }

        if (exception!=null) {
            throw convert2RuntimeException(exception);
        }
    }

    private void check(){
        if (committed) {
            throw new UnsupportedOperationException("all committed???");
        }
    }

    @Override
    public Segment segment(Segment.SD sd) {
        for (SegmentFileDescriptor segmentFileDescriptor : segmentFileDescriptorList) {
            if (segmentFileDescriptor.getName().equals(sd.getName()) && segmentFileDescriptor.getGroup().equals(sd.getGroup())) {
                return new Segment() {
                    @Override
                    public List<Map<String, Object>> rowList() {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public int size() {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public void append(Map<String, Object> row) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public String name() {
                        return segmentFileDescriptor.getName();
                    }

                    @Override
                    public Map<String, Object> meta() {
                        return segmentFileDescriptor.getMeta();
                    }

                    @Override
                    public Title title() {
                        return segmentFileDescriptor.title;
                    }

                    @Override
                    public Header header() {
                        return segmentFileDescriptor.header;
                    }

                    @Override
                    public String group() {
                        return segmentFileDescriptor.getGroup();
                    }

                    @Override
                    public void commit() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        }
        return null;
    }

    @Override
    public List<Segment.SD> sdList() {
        List<Segment.SD> sdList=new ArrayList<>(segmentFileDescriptorList.size());
        for (SegmentFileDescriptor segmentFileDescriptor : segmentFileDescriptorList) {
            Segment.SD sd=new Segment.SD();
            sd.setName(segmentFileDescriptor.getName());
            sd.setMemory(false);
            sd.setUrl(segmentFileDescriptor.getPath());
            sd.setRowCount(segmentFileDescriptor.getRowCount());
            sd.setGroup(segmentFileDescriptor.getGroup());
            sdList.add(sd);
        }
        return sdList;
    }

    @Override
    public CellFile file(byte[] data, String name, String type) {
        Map<String,Object> meta=new HashMap<>();
        meta.put("name",name);
        meta.put("type",type);
        FileIO.FilePutResult filePutResult = fileIO.put(meta, data);
        CellFile cellFile=new CellFile();
        cellFile.setName(name);
        cellFile.setType(type);
        cellFile.setBase64(false);
        cellFile.setUri(filePutResult.seq());
        return cellFile;
    }

    @Override
    public void fileName(String fileName) {
        this.fileName=fileName;
    }

    @Override
    public GroupOperation group(String group) {
        return new GroupOperation() {
            @Override
            public void fileName(String fileName) {
                groupMap.get(group).setFileName(fileName);
            }
        };
    }

    @Override
    public FeatureOperation feature() {
        return new FeatureOperation() {
            @Override
            public void supportRangeMerge(boolean rangeMerge) {
                feature.setRangeMerge(rangeMerge);
            }

            @Override
            public void supportBorder(boolean border) {
                feature.setBorder(border);
            }
        };
    }

    @Override
    public Stat stat() {
        return stat;
    }

    @Override
    public RuntimeStat runtimeStat() {
        return runtimeStat;
    }

    @Data
    static class SegmentFileDescriptor implements Model{

        private String path;

        private String name;

        private Segment.Title title;

        private Segment.Header header;

        private Map<String, Object> meta;

        private String group;

        private int rowCount;
    }

    class SegmentFileWriter implements SegmentWriter{

        final private String name;

        final private String group;

        final private RowTransLog rowTransLog;

        final String filePath;

        final private Segment.Header header=new HeaderImpl();

        final private Segment.Title title=new TitleImpl();

        private Map<String, Object> meta;

        private int rowCount;

        private boolean committed;

        final private SegmentStat segmentStat=new SegmentStat();

        public SegmentFileWriter(String name,String group,int size) {
            this.name=name;
            this.group=group;
//            RowTransLogConfig rowTransLogConfig =new RowTransLogConfig();
//            rowTransLogConfig.setWalBufferSizeMb(10);
//            rowTransLogConfig.setWalDiskSizeMb(1024);
//            rowTransLogConfig.setWalIntervalSec(1);
            this.filePath=path+"/"+md5(group+":"+name);
            rowTransLog =new MixedRowTransLogImpl("segment",filePath, rowTransLogConfig);
        }

        @Override
        public synchronized void append(Map<String, Object> data) {
            rowTransLog.append(data);
            rowCount++;
            runtimeStat.rowCountAdd(1);
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
        }

        @Override
        public synchronized void append(List<Map<String, Object>> data) {
            rowTransLog.append(data);
            rowCount+=data.size();
            runtimeStat.rowCountAdd(data.size());
            runtimeStat.lastIoTimeMs(System.currentTimeMillis());
        }

        @Override
        public synchronized void append(Model model) {
            append(JSONAccessor.impl().read(JSONAccessor.impl().format(model)));
        }

        @Override
        public void meta(Map<String, Object> meta) {
            this.meta=meta;
        }

        @Override
        public synchronized void commit() {

            if (committed) {
                return;
            }

            rowTransLog.fsync();

            for (SegmentFileDescriptor segmentFileDescriptor : segmentFileDescriptorList) {
                if (segmentFileDescriptor.getPath().equalsIgnoreCase(filePath)) {
                    throw new IllegalArgumentException("file is duplicate: "+filePath);
                }
            }

            SegmentFileDescriptor segmentFileDescriptor= new SegmentFileDescriptor();
            segmentFileDescriptor.setPath(filePath);
            segmentFileDescriptor.setName(name);
            segmentFileDescriptor.setTitle(title);
            segmentFileDescriptor.setHeader(header);
            segmentFileDescriptor.setMeta(meta);
            segmentFileDescriptor.setGroup(group);
            segmentFileDescriptor.setRowCount(rowCount);

            segmentFileDescriptorList.add(segmentFileDescriptor);
            header().commit();

            segmentStat.setRowCount(rowCount);
            if (stat.getSegmentStatList()==null) {
                stat.setSegmentStatList(new ArrayList<>());
            }
            stat.getSegmentStatList().add(segmentStat);

            this.committed=true;
        }

        @Override
        public Segment.Header header() {
            return header;
        }

        @Override
        public Segment.Title title() {
            return title;
        }

        @Override
        public String group() {
            return group;
        }

        void close() throws Exception{
            rowTransLog.close();
        }
    }

}
