package com.ovopark.iohub.sdk.model.proto;

import lombok.Data;

import java.util.List;

import static com.ovopark.kernel.shared.Util.isEmpty;
import static com.ovopark.kernel.shared.Util.isNotEmpty;

/**
 * <PRE>
 * level (不可变) 定义了节点在 Y 方向的坐标
 * position（不可变） 定义了节点在 X 方向的坐标
 *
 * - virtual
 *   虚拟化， 虚拟节点不会渲染在excel，其子节点占有这个节点的空间
 * - excelY
 *   节点在Y方向的最终坐标（如果路径上有虚拟节点，会做 Y 方向上的调整）
 *
 * x: {@link #excelX()} , y: {@link #excelY()}
 * , deltaX: {@link #leafCount()} , deltaY: {@link #level()} -  {@link #excelY()}
 */
@Data
public class HeaderCellImpl implements Segment.Header.Cell {

    /**
     *  vertical/Y , H ~ 0 ,1, 2 ,3 ,4 ,5, ...
     */
    private int level;

    private String path;

    private String property;

    private String name;

    private List<HeaderCellImpl> cellList;

    private int leafCount;

    /**
     * horizontal/X  , V ~ 0, 1 ,2 ,3, 4, 5, ...
     */
    private int position;

    private int width;

    private CellDataType cellType;

    private String[] pathLink;

    private String i18Key;

    private String tip;

    /**
     * 虚拟节点不会渲染在excel，其子节点占有这个节点的空间
     */
    private boolean virtual;

    private Integer excelY;

    @Override
    public String[] pathLink() {
        if (pathLink==null) {
            pathLink=path.split("\\.");
        }
        return pathLink;
    }

    @Override
    public int width() {
        return width;
    }

    @Override
    public int position() {
        return position;
    }

    @Override
    public CellDataType cellType() {
        return cellType;
    }

    @Override
    public boolean virtual() {
        return virtual;
    }

    public int calLeafCount() {
        if (isEmpty(cellList)) {
            leafCount = 1;
            return 1;
        }
        int sum = 0;
        for (HeaderCellImpl cell : cellList) {
            int i = cell.calLeafCount();
            sum += i;
        }
        leafCount = sum;
        return sum;
    }

    void shiftUpY(int y, boolean parentIsVirtual){
        if (parentIsVirtual) {
            excelY=y;
        }
        if (isNotEmpty(cellList)) {
            for (HeaderCellImpl headerCell : cellList) {
                headerCell.shiftUpY(excelY,this.isVirtual());
            }
        }
    }

    boolean matchYInAllTopCell(int y , int maxY){
        // y is not 0 , i.e. it is not top cell
        if (excelY()!=0) {
            return true;
        }
        if (isVirtual()
                || (isNotEmpty(cellList) && excelY<=y && y<=level && excelY()==0)
                || (isEmpty(cellList) && excelY<=y && y<=maxY  && excelY()==0)
        ) {
            boolean f=true;
            if (!isEmpty(cellList)) {
                for (HeaderCellImpl headerCell : cellList) {
                    if (!(f=headerCell.matchYInAllTopCell(y,maxY))) {
                        break;
                    }
                }
            }
            return f;
        }
        return false;
    }


    void calPosition(int position){
        setPosition(position);
        if (isEmpty(cellList)) {
            return;
        }
        for (int i = 0; i < cellList.size(); i++) {
            HeaderCellImpl cell = cellList.get(i);
            if (i==0) {
                cell.calPosition(position);
            }
            else {
                HeaderCellImpl b = cellList.get(i - 1);
                cell.calPosition(b.getPosition()+b.leafCount());
            }
        }

    }

    @Override
    public int level() {
        return getLevel();
    }

    @Override
    public String path() {
        return getPath();
    }

    @Override
    public String property() {
        return getProperty();
    }

    @Override
    public String name() {
        return getName();
    }

    @Override
    public List<? extends Segment.Header.Cell> cellList() {
        return getCellList();
    }

    @Override
    public int leafCount() {
        return getLeafCount();
    }

    @Override
    public int excelX() {
        return position;
    }

    @Override
    public int excelY() {
        return excelY==null?level:excelY;
    }
}
