package com.ovopark.iohub.sdk.model.proto;

import com.ovopark.kernel.shared.Model;
import lombok.Data;

import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

import static com.ovopark.kernel.shared.Util.*;

@Data
public class HeaderImpl implements Segment.Header , Model {

    private boolean committed;

    private boolean fixed;

    final private List<HeaderCellImpl> cellList = new ArrayList<>();

    // start from 0 , 1, 2, 3
    private int height;

    private int defaultColumnWidth=15;

    private int maxHeadIndex;

    private int maxPossibleHeadIndex;

    private int deltaY;

    @Override
    public int maxHeadIndex() {
        return maxHeadIndex;
    }

    @Override
    public void maxPossibleHeadIndex(int maxPossibleHeadIndex) {
        this.maxPossibleHeadIndex=maxPossibleHeadIndex;
    }

    @Override
    public int maxPossibleHeadIndex() {
        return maxPossibleHeadIndex;
    }

    public Segment.HeaderFeatureOperation root(String property) {
        if (committed) {
            throw new IllegalArgumentException("header was fixed, cannot modify");
        }
        return upset(null,property,null,null);
    }

    public void root(List<String> propertyList) {
        if (committed) {
            throw new IllegalArgumentException("header was fixed, cannot modify");
        }
        propertyList.forEach(this::root);
    }

    public void i18n(I18n i18n) {
        func(i18n::i18n, cellList, (s, cell) -> cell.setName(s)
                ,headerCell -> isNotEmpty(headerCell.getI18Key())?headerCell.getI18Key():headerCell.getPath()
                ,cell->isEmpty(cell.getName())
        );
    }

    @Override
    public void width(Width width) {
        func(width::width, cellList, (s, cell) -> cell.setWidth(s),null,cell->true);
    }

    @Override
    public Cell cell(String path) {
        String[] pathList = path.split("\\.");
        List<HeaderCellImpl> searchList=cellList;
        Cell mayMatchCell=null;
        for(int i=0;i<pathList.length;i++) {
            String matchProperty=pathList[i];
            boolean found=false;
            for (HeaderCellImpl headerCell : searchList) {
                if (matchProperty.equals(headerCell.getProperty())) {
                    searchList=headerCell.getCellList();
                    found=true;
                    mayMatchCell=headerCell;
                    break;
                }
            }
            if (!found) {
                return null;
            }
        }
        return mayMatchCell!=null?(path.equals(mayMatchCell.path())?mayMatchCell:null):null;
    }

    @Override
    public void printPretty() {
        printPretty(System.out);
    }

    @Override
    public void printPretty(OutputStream outputStream) {
        try {
            outputStream.write(utf8("================================heard def start========================\r\n"));
            scanUp2Down(new Scan() {
                @Override
                public void scan(Cell cell) {
                    HeaderCellImpl headerCell = (HeaderCellImpl) cell;
                    try {
                        outputStream.write(utf8(headerCell.getI18Key()+"="+headerCell.getTip()+"\r\n"));
                    } catch (IOException e) {
                        throw convert2RuntimeException(e);
                    }
                }
            });
            outputStream.write(utf8("================================heard def end========================\r\n"));
        }catch (Exception e){
            throw convert2RuntimeException(e);
        }
    }

    interface CellFuncSet<R>{
        void set(R r, HeaderCellImpl cell);
    }

    private <R> void func(Function<String,R> func, List<HeaderCellImpl> cellList
            , CellFuncSet<R> cellFuncSet, Function<HeaderCellImpl,String> pathGet
            , Predicate<HeaderCellImpl> cellPredicate
    ) {
        for (HeaderCellImpl cell : cellList) {
            if (!cellPredicate.test(cell)) {
                continue;
            }
            R apply = func.apply(pathGet==null?cell.getPath():pathGet.apply(cell));
            cellFuncSet.set(apply,cell);
            if (isNotEmpty(cell.getCellList())) {
                func(func, cell.getCellList(),cellFuncSet,pathGet,cellPredicate);
            }
        }
    }


    public void children(String path, List<String> propertyList) {
        for (String p : propertyList) {
            upset(path,p,null,null);
        }
    }

    @Override
    public Segment.HeaderFeatureOperation children(String path, String property) {
        return children(path,property,null);
    }

    @Override
    public Segment.HeaderFeatureOperation children(String path, String property, Width width) {
        return children(path,property,width,null);
    }

    @Override
    public Segment.HeaderFeatureOperation children(String path, String property, Width width, I18n i18n) {
        return children(path,property,width,i18n,null);
    }

    @Override
    public Segment.HeaderFeatureOperation children(String path, String property, Width width, I18n i18n, CellType cellType) {
        return upset(path,property,width,i18n,cellType);
    }

    @Override
    public Segment.HeaderFeatureOperation upset(String path, String property, Width width) {
        return upset(path,property,width,null);
    }

    @Override
    public Segment.HeaderFeatureOperation upset(String path, String property, Width width, I18n i18n) {
        return upset(path,property,width,i18n,null);
    }

    @Override
    public Segment.HeaderFeatureOperation upset(String path, String property, Width width, I18n i18n,CellType cellType) {
        return upset0(path, property, width, i18n, cellType,c->{},false);
    }

    @Override
    public void upsetAsVirtual(String path, String property) {
        upset0(path, property,a -> 30 , b -> null, c -> CellDataType.NONE,d->{},true);
    }

    @Override
    public void updateAsVirtual(String path) {
        Cell cell = cell(path);
        ((HeaderCellImpl) cell).setVirtual(true);
    }

    private Segment.HeaderFeatureOperation upset0(String path, String property, Width width, I18n i18n
            , CellType cellType, Consumer<HeaderCellImpl> headerCellConsumer, boolean virtual) {
        if (committed) {
            throw new IllegalArgumentException("header was fixed, cannot modify");
        }

        if (isEmpty(path)) {
            HeaderCellImpl cell = new HeaderCellImpl();
            cell.setProperty(property);
            cell.setLevel(0);
            cell.setExcelY(0);
            cell.setPath(property);
            if (width !=null) {
                cell.setWidth(width.width(property));
            }
            if (i18n != null) {
                cell.setName(i18n.i18n(property));
            }
            if (cellType != null) {
                cell.setCellType(cellType.cellType(property));
            }
            cell.setVirtual(virtual);
            headerCellConsumer.accept(cell);
            cellList.add(cell);
            return new HeaderFeatureOperationImpl(cell);
        }

        HeaderCellImpl pathCell = null;
        List<HeaderCellImpl> queryCellList = cellList;
        String[] pl = path.split("\\.");
        for (int i = 0; i < pl.length; i++) {
            String p = pl[i];
            HeaderCellImpl next = null;
            for (HeaderCellImpl cell : queryCellList) {
                if (cell.getProperty().equals(p)) {
                    next = cell;
                    break;
                }
            }

            if (next == null) {
                throw new IllegalArgumentException("sub path: '" + p + "' is missing? " + path);
            }

            queryCellList = next.getCellList();
            pathCell = next;
        }

        if (pathCell == null) {
            throw new IllegalArgumentException("path is missing: " + path);
        }

        List<HeaderCellImpl> cellList =pathCell.getCellList();
        if (cellList==null) {
            cellList=new ArrayList<>();
        }
        HeaderCellImpl cell = new HeaderCellImpl();
        cell.setProperty(property);
        cell.setPath(path + "." + property);
        cell.setLevel(pl.length);
        cell.setExcelY(pl.length);
        if (width !=null) {
            cell.setWidth(width.width(cell.getPath()));
        }
        if (i18n != null) {
            cell.setName(i18n.i18n(cell.getPath()));
        }
        if (cellType != null) {
            cell.setCellType(cellType.cellType(cell.getPath()));
        }
        cell.setVirtual(virtual);
        headerCellConsumer.accept(cell);
        height = Math.max(height, cell.getLevel());
        cellList.add(cell);

        pathCell.setCellList(cellList);

        return new HeaderFeatureOperationImpl(cell);
    }

    @Override
    public void defaultColumnWidth(int width) {
        this.defaultColumnWidth=width;
    }

    @Override
    public int defaultColumnWidth() {
        return defaultColumnWidth;
    }

    @Override
    public void all(Class<? extends Model> clazz) {
        all(clazz,a->null);
    }

    private void extendField(String path,Field field,Map<String, Field> fieldMap,boolean include){
        if(field.getGenericType() instanceof ParameterizedType){
            if (!include) {
                fieldMap.put(path,field);
            }
            Type actualTypeArgument = ((ParameterizedType) field.getGenericType()).getActualTypeArguments()[0];
            Map<String, Field> tmpFieldMap = flatProperty((Class<?>) actualTypeArgument,
                    tmpField -> tmpField.getDeclaredAnnotation(ExcludeInHeader.class)==null
                            && !CellValue.class.isAssignableFrom(tmpField.getDeclaringClass())
                    && !tmpField.isSynthetic() && !tmpField.isEnumConstant()

            );
            for (Map.Entry<String, Field> entry : tmpFieldMap.entrySet()) {
                extendField(path+"."+entry.getKey(), entry.getValue(), fieldMap,false);
            }
        }
        else {
            if (!include) {
                fieldMap.put(path,field);
            }
        }
    }

    @Override
    public void all(Class<? extends Model> clazz, I18n i18n) {
        boolean javaTypeAsDataType=true;
        HeaderDef headerDef = clazz.getDeclaredAnnotation(HeaderDef.class);
        if (headerDef != null) {
            setDefaultColumnWidth(headerDef.defaultColumnWidth()>0?headerDef.defaultColumnWidth():15);
            if (headerDef.fixed()) {
                fixed();
            }
            javaTypeAsDataType = headerDef.javaTypeAsDataType();
        }
        final boolean javaTypeAsDataTypeDef=javaTypeAsDataType;
        Map<String, Field> fieldMap = flatProperty(clazz,
                field -> field.getDeclaredAnnotation(ExcludeInHeader.class)==null
                        && !CellValue.class.isAssignableFrom(field.getDeclaringClass()));
        //flatProperty(...) cannot find generic type ,i.e. List<T> ,
        // so , we need get more if any generic type is here,
        LinkedHashMap<String, Field> tempFieldMap=new LinkedHashMap<>(fieldMap);
        for (Map.Entry<String, Field> entry : fieldMap.entrySet()) {
            extendField(entry.getKey(), entry.getValue(), tempFieldMap,true);
        }
//        LinkedHashMap<String, Field> finalFieldMap=new LinkedHashMap<>();
//        tempFieldMap.entrySet().stream().sorted(Map.Entry.comparingByKey())
//                .forEach(e->finalFieldMap.put(e.getKey(),e.getValue()));
        for (Map.Entry<String, Field> entry : tempFieldMap.entrySet()) {
            String c = entry.getKey();
            String parentPath = c.substring(0, Math.max(c.lastIndexOf("."), 0));
            String subPath=c.substring(Math.max(c.lastIndexOf(".")+1,0));
            Field field = entry.getValue();
            CellDef cellDef = field.getDeclaredAnnotation(CellDef.class);
            CellVirtual cellVirtual = field.getDeclaredAnnotation(CellVirtual.class);
            FontDef fontDef = field.getDeclaredAnnotation(FontDef.class);
            HyperlinkDef hyperlinkDef = field.getDeclaredAnnotation(HyperlinkDef.class);
            upset0(parentPath, subPath, new Width() {
                @Override
                public int width(String path) {
                    return cellDef==null?0:cellDef.width();
                }
            }, null, new CellType() {
                @Override
                public CellDataType cellType(String path) {
                    CellDataType def= cellDef == null ? CellDataType.NONE : cellDef.cellDataType();
                    if (def== CellDataType.NONE && javaTypeAsDataTypeDef) {
                        Class<?> type = field.getType();
                        if (Boolean.class.isAssignableFrom(type) || boolean.class==type) {
                            def=CellDataType.BOOLEAN;
                        }
                        else if (Number.class.isAssignableFrom(type)
                                || int.class==type || long.class==type
                                || short.class==type || byte.class==type
                                || float.class==type || double.class==type
                        ) {
                            def=CellDataType.NUMERIC;
                        }
                        else {
                            def=CellDataType.STRING;
                        }
                    }
                    return def;
                }
            },headerCell -> {
                //tip
                headerCell.setTip(cellDef==null?null:cellDef.tip());
                headerCell.setI18Key(cellDef==null?null:cellDef.i18Key());
                String showName = cellDef == null ? null : cellDef.showName();
                if (isNotEmpty(showName)) {
                    //set as real show name
                    headerCell.setName(showName);
                }

                else if(isNotEmpty(headerCell.getI18Key())){
                    showName = i18n.i18n(headerCell.getI18Key());
                    headerCell.setName(showName);
                }

                else {
                    showName = i18n.i18n(headerCell.getPath());
                    headerCell.setName(showName);
                }
                // font
                if (fontDef != null) {
                    headerCell.getHeaderFeature().setFontFeatureEnabled(true);
                    headerCell.getHeaderFeature().setBold(fontDef.bold());
                    headerCell.getHeaderFeature().setUnderline(fontDef.underline());
                    headerCell.getHeaderFeature().setItalic(fontDef.italic());
                    headerCell.getHeaderFeature().setStrikeout(fontDef.strikeout());
                    if (isNotEmpty(fontDef.color())) {
                        headerCell.getHeaderFeature().setColor(fontDef.color());
                    }
                }
                // hyperlink
                if (hyperlinkDef != null) {
                    headerCell.getHeaderFeature().setHyperlinkFeatureEnabled(true);
                    if (isNotEmpty(hyperlinkDef.type())) {
                        headerCell.getHeaderFeature().setHyperlinkUrl("url".equalsIgnoreCase(hyperlinkDef.type()));
                    }
                }

            }
            ,cellVirtual!=null
            );
        }

    }

    @Override
    public void fixed() {
        this.fixed=true;
    }

    public void commit() {

        if (committed) {
            return;
        }

        committed = true;

        for (HeaderCellImpl cell : cellList) {
            int i = cell.calLeafCount();
            cell.setLeafCount(i);
        }


        for (int i = 0; i < cellList.size(); i++) {
            HeaderCellImpl cell = cellList.get(i);
            if (i==0) {
                cell.calPosition(0);
            }
            else {
                HeaderCellImpl b = cellList.get(i - 1);
                cell.calPosition(b.getPosition()+b.leafCount());
            }
        }

        scanUp2Down(new Scan() {
            @Override
            public void scan(Cell cell) {
                maxHeadIndex =Math.max(maxHeadIndex,cell.excelX());
            }
        });

        // re-calculate x / y
        for (HeaderCellImpl headerCell : cellList) {
            // root
            headerCell.shiftUpY(headerCell.level(),headerCell.isVirtual());
        }

        int l = height();
        for (int i = 1; i <=l; i++) {
            boolean f=true;
            for (HeaderCellImpl headerCell : cellList) {
                if (!(f = headerCell.matchYInAllTopCell(i,l))) {
                    break;
                }
            }
            if (f) {
                //
                deltaY--;
            }
            else {
                break;
            }
        }


    }

    public int height() {
        return height;
    }

    public void scanUp2Down(Scan scan) {
        scanUp2Down0(scan, cellList);
    }

    private void scanUp2Down0(Scan scan, List<HeaderCellImpl> cellList) {
        List<HeaderCellImpl> nextList = new ArrayList<>();
        for (HeaderCellImpl cell : cellList) {
            scan.scan(cell);
            if (isNotEmpty(cell.getCellList())) {
                nextList.addAll(cell.getCellList());
            }
        }
        if (isNotEmpty(nextList)) {
            scanUp2Down0(scan, nextList);
        }
    }

    @Override
    public List<String> properties() {
        List<String> list=new ArrayList<>();
        scanUp2Down(cell -> list.add(cell.property()));
        return list;
    }

    @Override
    public int heightExcludeVirtual() {
        return height+deltaY;
    }

    static class HeaderFeatureOperationImpl implements Segment.HeaderFeatureOperation{

        final private HeaderCellImpl headerCell;

        public HeaderFeatureOperationImpl(HeaderCellImpl headerCell) {
            this.headerCell = headerCell;
        }

        @Override
        public Segment.HeaderFeatureOperation fontBold(boolean bold) {
            headerCell.getHeaderFeature().setFontFeatureEnabled(true);
            headerCell.getHeaderFeature().setBold(bold);
            return this;
        }

        @Override
        public Segment.HeaderFeatureOperation fontUnderline(boolean underline) {
            headerCell.getHeaderFeature().setFontFeatureEnabled(true);
            headerCell.getHeaderFeature().setUnderline(underline);
            return this;
        }

        @Override
        public Segment.HeaderFeatureOperation fontItalic(boolean italic) {
            headerCell.getHeaderFeature().setFontFeatureEnabled(true);
            headerCell.getHeaderFeature().setItalic(italic);
            return this;
        }

        @Override
        public Segment.HeaderFeatureOperation fontStrikeout(boolean strikeout) {
            headerCell.getHeaderFeature().setFontFeatureEnabled(true);
            headerCell.getHeaderFeature().setStrikeout(strikeout);
            return this;
        }

        @Override
        public Segment.HeaderFeatureOperation fontColor(String color) {
            headerCell.getHeaderFeature().setFontFeatureEnabled(true);
            headerCell.getHeaderFeature().setColor(color);
            return this;
        }

        @Override
        public Segment.HeaderFeatureOperation hyperlink(String type) {
            headerCell.getHeaderFeature().setHyperlinkFeatureEnabled(true);
            headerCell.getHeaderFeature().setHyperlinkUrl("url".equalsIgnoreCase(type));
            return this;
        }
    }

}
