package com.ovopark.jobhub.sdk.client.kafka;

import com.ovopark.jobhub.sdk.client.*;
import com.ovopark.jobhub.sdk.model.*;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Model;
import com.ovopark.kernel.shared.ServiceProvider;
import com.ovopark.kernel.shared.ShutdownManager;
import com.ovopark.kernel.shared.kv.KVEngine;
import com.ovopark.kernel.shared.vclient.ClientNode;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.MDC;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.CommandLineRunner;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.listener.AcknowledgingMessageListener;
import org.springframework.kafka.listener.ConcurrentMessageListenerContainer;
import org.springframework.kafka.listener.ConsumerSeekAware;
import org.springframework.kafka.listener.ContainerProperties;
import org.springframework.kafka.support.Acknowledgment;
import org.springframework.kafka.support.TopicPartitionOffset;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicLong;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Component
@Slf4j
public class TaskListenerRunnerViaKafkaProvider implements JobService.TaskListenerRunnerProvider , CommandLineRunner {

    @Qualifier("com.ovopark.jobhub.sdk.client.kafka.KafkaConfig.consumerFactory")
    @Autowired
    ConsumerFactory<String, String> consumerFactory;

    private final Map<String, TopicPartitions> containers = new ConcurrentHashMap<>();

    @Autowired
    private JobService jobService;

    @Autowired
    private Client2ControlTransport client2ControlTransport;

    @Autowired
    private ClientNodeProvider clientNodeProvider;

    final static KVEngine.TtlFunc<String> ttlFunc=KVEngine.newTtl("t-p-heartbeat-ttl");

    final static AtomicLong vcc=new AtomicLong();

    @Override
    public String name() {
        return "kafka";
    }

    final ScheduledExecutorService scheduledExecutorService= Executors.newSingleThreadScheduledExecutor(newThreadFactory("create-kafka-consumer"));

    @Override
    public boolean start(String jobType, String beanUrl, String group, Long minVer
            , JobService.TaskListener taskListener,JobService.TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig) {
        boolean f = start0(jobType, beanUrl, group, minVer, taskListener, taskListenerRunnerProviderConfig);
        if (!f) {
            log.warn("cannot start consumer, schedule a delay task: "+beanUrl+":"+jobType+":"+group);
            scheduledExecutorService.schedule(new Runnable() {
                @Override
                public void run() {
                    try {
                        boolean f = start0(jobType, beanUrl, group, minVer, taskListener, taskListenerRunnerProviderConfig);
                        if (!f) {
                            scheduledExecutorService.schedule(this,10,TimeUnit.SECONDS);
                            log.warn("cannot start consumer, schedule a delay task: "+beanUrl+":"+jobType+":"+group);
                        }
                    }
                    catch (Exception e){
                        log.error(e.getMessage(),e);
                        scheduledExecutorService.schedule(this,10,TimeUnit.SECONDS);
                        log.warn("cannot start consumer, schedule a delay task: "+beanUrl+":"+jobType+":"+group);
                    }
                }
            },10,TimeUnit.SECONDS);
        }
        return f;
    }

    private boolean start0(final String jobType, final String beanUrl, final String group, Long minVer
            , JobService.TaskListener taskListener,JobService.TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig) {
        final String containerId=beanUrl+":"+jobType+":"+group;
        TopicPartitions topicPartitions = lock(containerId, new Callable<TopicPartitions>() {
            @Override
            public TopicPartitions call() throws Exception {
                TopicPartitions tps = containers.get(containerId);
                if (tps != null) {
                    return tps;
                }

                TaskMetaGetRequest taskMetaGetRequest = new TaskMetaGetRequest();
                taskMetaGetRequest.setType(jobType);
                taskMetaGetRequest.setGroup(group);
                TaskMetaGetResponse taskMetaGetResponse = jobService.taskMetaGet(taskMetaGetRequest);
                if (taskMetaGetResponse == null || isEmpty(taskMetaGetResponse.getTopic())) {
                    log.info("cannot find topic " + JSONAccessor.impl().format(taskMetaGetResponse)
                            + ", request: " + JSONAccessor.impl().format(taskMetaGetRequest));
                    return null;
                }
                final String topic = taskMetaGetResponse.getTopic();
                final String consumerGroup = isEmpty(taskListenerRunnerProviderConfig.getConsumerGroup()) ? containerId :
                        taskListenerRunnerProviderConfig.getConsumerGroup();
                // 创建容器
                ContainerProperties containerProps=null;
                String partitions = taskListenerRunnerProviderConfig.getPartitions();
                if (isNotEmpty(partitions)) {
                    List<TopicPartitionOffset> topicPartitionOffsetList=new ArrayList<>();
                    for (String p : partitions.split(",")) {
                        TopicPartitionOffset topicPartitionOffset=new TopicPartitionOffset(topic,Integer.parseInt(p));
                        topicPartitionOffsetList.add(topicPartitionOffset);

                    }
                    containerProps = new ContainerProperties(topicPartitionOffsetList.toArray(new TopicPartitionOffset[]{}));
                }
                else {
                    containerProps = new ContainerProperties(topic);
                }

                containerProps.setGroupId(consumerGroup);
                // 通过 consumer properties 覆盖 offset reset
                if (isNotEmpty(taskListenerRunnerProviderConfig.getOffsetReset())) {
                    containerProps.getKafkaConsumerProperties()
                            .put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, taskListenerRunnerProviderConfig.getOffsetReset()); // e.g., "earliest"
                }

                final boolean autoCommit = taskListenerRunnerProviderConfig.isAutoCommit();
                containerProps.getKafkaConsumerProperties().put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, autoCommit);
                if (!autoCommit) {
                    // 👇 关键：设置为 MANUAL 或 MANUAL_IMMEDIATE
                    containerProps.setAckMode(ContainerProperties.AckMode.MANUAL);
                }
                if (taskListenerRunnerProviderConfig.getPollIntervalTimeSec()>0) {
                    containerProps.getKafkaConsumerProperties().put(ConsumerConfig.MAX_POLL_INTERVAL_MS_CONFIG, taskListenerRunnerProviderConfig.getPollIntervalTimeSec()*1000);
                }

                if (taskListenerRunnerProviderConfig.getMaxPollRecords()>0) {
                    containerProps.getKafkaConsumerProperties().put(ConsumerConfig.MAX_POLL_RECORDS_CONFIG, taskListenerRunnerProviderConfig.getMaxPollRecords());
                }
                containerProps.getKafkaConsumerProperties().put(ConsumerConfig.PARTITION_ASSIGNMENT_STRATEGY_CONFIG
                        , "org.apache.kafka.clients.consumer.CooperativeStickyAssignor");

                Map<String, ScheduledThreadPoolExecutor> heartbeatThread = new ConcurrentHashMap<>();

                containerProps.setMessageListener(
                        new PartitionSeekOffsetMessageListener(jobType,taskListenerRunnerProviderConfig
                                ,containerId,consumerGroup,taskListener,heartbeatThread));

                TopicPartitions topicPartitions = new TopicPartitions();

                ConcurrentMessageListenerContainer<String, String> container =
                        new ConcurrentMessageListenerContainer<>(consumerFactory, containerProps);
                // 可选：设置并发（分区数需支持）
                container.setConcurrency(Math.max(taskListenerRunnerProviderConfig.getConcurrency(), 1));

                // 可选：添加启动/停止监听器
                container.setApplicationEventPublisher(event -> {
                    // 可用于监控
                });

                container.start();

                topicPartitions.setConcurrentMessageListenerContainer(container);
                topicPartitions.setHeartbeatThread(heartbeatThread);

                containers.put(containerId, topicPartitions);
                log.info("Started dynamic consumer for topic: " + topic + ", group: " + consumerGroup + ", concurrency: " + container.getConcurrency());

                return topicPartitions;
            }
        });
        return topicPartitions!=null;
    }


    class PartitionSeekOffsetMessageListener implements AcknowledgingMessageListener<String, String> , ConsumerSeekAware {

        final String jobType;

        final String containerId;

        final private String consumerGroup;

        final JobService.TaskListener taskListener;

        final boolean autoCommit;

        final Map<String, ScheduledThreadPoolExecutor> heartbeatThread;

        final private boolean ignoreTaskIfDone;

        final boolean remoteLogEnabled;

        public PartitionSeekOffsetMessageListener(String jobType
                , JobService.TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig
                , String containerId
                , String consumerGroup
                , JobService.TaskListener taskListener
                ,Map<String, ScheduledThreadPoolExecutor> heartbeatThread
        ) {
            this.jobType = jobType;
            this.containerId = containerId;
            this.consumerGroup = consumerGroup;
            this.taskListener = taskListener;
            this.autoCommit = taskListenerRunnerProviderConfig.isAutoCommit();
            this.heartbeatThread=heartbeatThread;
            this.ignoreTaskIfDone=taskListenerRunnerProviderConfig.isIgnoreTaskIfDone();
            this.remoteLogEnabled=taskListenerRunnerProviderConfig.isRemoteLogEnabled();
        }

        @Override
        public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
            for (TopicPartition partition : partitions) {
                String t = partition.topic();
                int p = partition.partition();
                log.info("partition revoked : " + t + ":" + p);
                String k = TopicPartitionContext.KEY_PREFIX + ":" + containerId + ":" + p;
                ttlFunc.putAndGet(k, (KVEngine.Upset<String, TopicPartitionContext, TopicPartitionContext>) getResult -> {
                    if (!getResult.exists()) {
                        log.info(t+":"+p+" ,  revoked ,partition cache is missing , ignore , not assigned???: "+k);
                        return null;
                    }
                    TopicPartitionContext topicPartitionContext = getResult.value();
                    if (topicPartitionContext.getIoThread()==Thread.currentThread()) {
                        topicPartitionContext.setIoThread(null);
                        //
                        ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = heartbeatThread.get(k);
                        try {
                            if (scheduledThreadPoolExecutor != null) {
                                scheduledThreadPoolExecutor.shutdown();
                            }
                            log.info("partition revoked, close heartbeat thread: " + t + ":" + p);
                        } catch (Exception e) {
                            log.error(e.getMessage(), e);
                        }
                    }
                    return topicPartitionContext;
                });
            }
        }

        @Override
        public void onPartitionsAssigned(Map<TopicPartition, Long> assignments, ConsumerSeekCallback callback) {
            for (TopicPartition partition : assignments.keySet()) {
                String t = partition.topic();
                int p = partition.partition();
                log.info("partition assigned : " + t + ":" + p);
                String k = TopicPartitionContext.KEY_PREFIX + ":" + containerId + ":" + p;
                //put current stat
                ttlFunc.putAndGet(k, (KVEngine.Upset<String, TopicPartitionContext, TopicPartitionContext>) getResult -> {
                    TopicPartitionContext topicPartitionContext =getResult.value();
                    if (!getResult.exists() || getResult.value()==null) {
                        topicPartitionContext = new TopicPartitionContext(t,p,containerId);
                    }
                    Thread beforeIoThread = topicPartitionContext.getIoThread();
                    topicPartitionContext.setIoThread(Thread.currentThread());
                    if (beforeIoThread!=Thread.currentThread()) {
                        //OTHER CONSUMER SUBSCRIBE ???
                        topicPartitionContext.setTempUniqueId(uniqueFirstPart());
                        topicPartitionContext.setProgressStat(new ProgressStat());
                        ScheduledThreadPoolExecutor scheduledThreadPoolExecutor = heartbeatThread.get(k);
                        try {
                            if (scheduledThreadPoolExecutor != null) {
                                scheduledThreadPoolExecutor.shutdown();
                            }
                            log.info("partition assigned, first close heartbeat thread: " + t + ":" + p);
                        } catch (Exception e) {
                            log.error(e.getMessage(), e);
                        }
                        //new
                        log.info("partition assigned, to start heartbeat thread: " + t + ":" + p);
                        ScheduledThreadPoolExecutor taskHeartbeat
                                = new ScheduledThreadPoolExecutor(
                                1, newThreadFactory("tmp"), new ThreadPoolExecutor.CallerRunsPolicy());
                        taskHeartbeat.setKeepAliveTime(60, TimeUnit.SECONDS);
                        taskHeartbeat.allowCoreThreadTimeOut(true);
                        TopicPartitionContext finalTopicPartitionContext = topicPartitionContext;
                        Runnable runnable = catchRunnable(new CatchRunnable() {
                            @Override
                            public void run() throws Exception {
                                //push topic heartbeat
                                ClientNode clientNode = clientNodeProvider.clientNode();
                                KafkaTaskContext kafkaTaskContext = finalTopicPartitionContext.kafkaTaskContext;
                                if (kafkaTaskContext==null) {
                                    return;
                                }
                                HeavyPerTopicPartitionTaskListener.SubProgressStat subProgressStat = kafkaTaskContext.getSubProgressStat();
                                TaskGetResponse.Task task = kafkaTaskContext.task();
                                TaskHeartbeatRequest taskHeartbeatRequest =new TaskHeartbeatRequest();
                                taskHeartbeatRequest.setTaskIdInES(task.getId());
                                taskHeartbeatRequest.setJobIdInES(task.getJobId());
                                taskHeartbeatRequest.setApp(clientNode.app());
                                taskHeartbeatRequest.setNode(ClientNode.UUID_STR);
                                taskHeartbeatRequest.setVcc(vcc.incrementAndGet());
                                //
                                taskHeartbeatRequest.setContainerId(containerId);
                                taskHeartbeatRequest.setPartition(p);
                                taskHeartbeatRequest.setConsumerGroup(consumerGroup);
                                taskHeartbeatRequest.setTempUniqueId(finalTopicPartitionContext.getTempUniqueId());
                                taskHeartbeatRequest.setTaskStartTimeStr(kafkaTaskContext.getTaskStartTimeStr());

                                taskHeartbeatRequest.setTask(task);
                                taskHeartbeatRequest.setHeartbeatData(finalTopicPartitionContext.getHeartbeatData());

                                ProgressStat progressStat=new ProgressStat();
                                BeanUtils.copyProperties(finalTopicPartitionContext.getProgressStat(),progressStat);
                                progressStat.setSubProgressStat(subProgressStat);
                                taskHeartbeatRequest.setProgressStat(progressStat);
                                try {
                                    log.info(k + ", heartbeat, request " + JSONAccessor.impl().format(taskHeartbeatRequest));
                                    TaskHeartbeatResponse taskHeartbeatResponse = client2ControlTransport.heartbeatStat(taskHeartbeatRequest);
                                    if (taskHeartbeatResponse ==null) {
                                        finalTopicPartitionContext.setCancelledTaskIdInES(null);
                                    }
                                    else {
                                        finalTopicPartitionContext.setCancelledTaskIdInES(taskHeartbeatResponse.getCancelledTaskIdInES());
                                    }
                                    log.info(k + ", heartbeat, response " + JSONAccessor.impl().format(taskHeartbeatResponse));
                                } catch (Exception e) {
                                    log.error(e.getMessage(),e);
                                }
                            }
                        });
                        taskHeartbeat.scheduleWithFixedDelay(runnable, 0, 5, TimeUnit.SECONDS);
                        heartbeatThread.put(k, taskHeartbeat);
                        log.info("partition assigned,start heartbeat thread: " + t + ":" + p);
                    }
                    return topicPartitionContext;
                });
            }
        }

        @Override
        public void onMessage(ConsumerRecord<String, String> record, Acknowledgment acknowledgment) {
            String t = record.topic();
            int p = record.partition();
            //put current stat
            final TopicPartitionContext topicPartitionContext = ((TopicPartitionContext) ttlFunc.get(TopicPartitionContext.KEY_PREFIX + ":" + containerId + ":" + p).value());
            try {
                ProgressStat progressStat = topicPartitionContext.getProgressStat();
                if (progressStat.getStartOffset()==0) {
                    progressStat.setStartOffset(record.offset());
                }
                progressStat.setTaskOffset(record.offset());
                //
                String value = record.value();
                TaskGetResponse.Task task = JSONAccessor.impl().read(value, TaskGetResponse.Task.class);
                final String taskKey = "_job4kafka_" + task.getJobId() + "_task_" + task.getId();
                final JobLog jobLog = remoteLogEnabled? new JobLogImpl(client2ControlTransport, taskKey, 0)
                        :JobLog.noopIfNull(null);
                KafkaTaskContext kafkaTaskContext = new KafkaTaskContext();
                kafkaTaskContext.setTask(task);
                kafkaTaskContext.setHeartbeatDataListener(topicPartitionContext::setHeartbeatData);
                kafkaTaskContext.setCancelledSupplier((ServiceProvider<Boolean>) () -> compare2(task.getId(), topicPartitionContext.getCancelledTaskIdInES())==0);
                kafkaTaskContext.setJobLog(jobLog);
                topicPartitionContext.setKafkaTaskContext(kafkaTaskContext);
                //
                if (ignoreTaskIfDone) {
                    TaskIfDoneRequest taskIfDoneRequest =new TaskIfDoneRequest();
                    taskIfDoneRequest.setTaskIdInES(task.getId());
                    taskIfDoneRequest.setConsumerGroup(consumerGroup);
                    TaskIfDoneResponse taskIfDoneResponse = client2ControlTransport.taskIfDone(taskIfDoneRequest);
                    if (taskIfDoneResponse !=null && taskIfDoneResponse.isDone()) {
                        log.info("cancel task: "+JSONAccessor.impl().format(taskIfDoneResponse));
                        return;
                    }
                }


                TaskUpdateRequest taskUpdateRequest = new TaskUpdateRequest();
                taskUpdateRequest.setId(task.getId());
                //
                MDC.put("traceId", taskKey);
                MDC.put("requestId", taskKey);
                long startMs=System.currentTimeMillis();
                JobStatus jobStatus = JobStatus.COMPLETED;
                try {
                    taskListener.on(kafkaTaskContext);
                    jobStatus = convert2Self(kafkaTaskContext.getJobStatus(), JobStatus.COMPLETED);
                }
                catch (Exception e) {
                    log.error(e.getMessage(), e);
                    jobStatus = JobStatus.FAIL;
                    taskUpdateRequest.setCompletedDesc(e.getMessage());
                }
                finally {
                    if (!kafkaTaskContext.isStatusManageManually()) {
                        try {
                            taskUpdateRequest.setId(task.getId());
                            taskUpdateRequest.setDocIndexName(task.getDocIndexName());

                            taskUpdateRequest.setStatus(jobStatus.name());
                            if (isNotEmpty(kafkaTaskContext.getCompletedDesc())) {
                                taskUpdateRequest.setCompletedDesc(kafkaTaskContext.getCompletedDesc());
                            }
                            taskUpdateRequest.setRequestDeviceUrl(kafkaTaskContext.getRequestDeviceUrl());
                            taskUpdateRequest.setRequestDeviceArgs(kafkaTaskContext.getRequestDeviceArgs());
                            taskUpdateRequest.setResponseFromDevice(kafkaTaskContext.getResponseFromDevice());

                            taskUpdateRequest.setRetryCount(kafkaTaskContext.task().getRetryCount());

                            TaskUpdateResponse taskUpdateResponse = jobService.taskUpdate(taskUpdateRequest);
                            log.info("taskUpdate: " + JSONAccessor.impl().format(taskUpdateResponse));

                        } catch (Exception e) {
                            log.error(e.getMessage(),e);
                        }
                    }

                    try {
                        kafkaTaskContext.append0(taskKey+" , cost time: "+costTime(System.currentTimeMillis()-startMs)
                                +", start: "+formatTime(dateTime(startMs)));
                        List<String> contentList = kafkaTaskContext.getContentList();
                        TaskLogPutRequest taskLogPutRequest = new TaskLogPutRequest();
                        taskLogPutRequest.setJobId(task.getJobId());
                        taskLogPutRequest.setTaskId(task.getId());
                        taskLogPutRequest.setType(jobType);
                        taskLogPutRequest.setContentList(contentList);

                        TaskLogPutResponse taskLogPutResponse = jobService.taskLog(taskLogPutRequest);
                        log.info("taskLog: " + JSONAccessor.impl().format(taskLogPutResponse));
                    } catch (Exception e) {
                        log.error(e.getMessage(),e);
                    }

                    try {
                        jobLog.flush(true);
                    } catch (Exception e) {
                        log.error(e.getMessage(),e);
                    }
                    log.info(taskKey+" , cost time: "+costTime(System.currentTimeMillis()-startMs)
                            +", start: "+formatTime(dateTime(startMs)));
                }
            }catch (Exception e) {
                // 处理异常，避免容器崩溃
                log.error("Error processing record from topic: {}", t, e);
            }
            finally {
                // clear current stat
                topicPartitionContext.setKafkaTaskContext(null);
                if (!autoCommit) {
                    acknowledgment.acknowledge();
                }
                MDC.remove("traceId");
                MDC.remove("requestId");
            }
        }
    }

    @Override
    public void close() {

        for (TopicPartitions topicPartitions : containers.values()) {
            try {
                ConcurrentMessageListenerContainer<String, String> container = topicPartitions.getConcurrentMessageListenerContainer();
                container.stop();
            } catch (Exception e) {
                log.error(e.getMessage(),e);
            }
        }

    }


    @Override
    public void run(String... args) throws Exception {
        ShutdownManager.getOrCreate().register(TaskListenerRunnerViaKafkaProvider.class.getName(), new CatchRunnable() {
            @Override
            public void run() throws Exception {
                close();
            }
        });
        log.info("register shutdown hook, jobhub kafka listener");
    }

    TopicPartitionContext get(String containerId, String p){
        String k = TopicPartitionContext.KEY_PREFIX + ":" + containerId + ":" + p;
        KVEngine.GetResult<String, Object> getResult = ttlFunc.get(k);
        if (!getResult.exists()) {
            return null;
        }

        return (TopicPartitionContext) getResult.value();
    }

    @Data
    public static class TopicPartitionContext implements Model{

        static final String KEY_PREFIX="TopicPartitionContext";

        final private String topic;

        final private int partition;

        final private String containerId;

        private final long startTime=System.currentTimeMillis();

        public TopicPartitionContext(String topic, int partition, String containerId) {
            this.topic = topic;
            this.partition = partition;
            this.containerId = containerId;
        }

        private String tempUniqueId;

        // -------------

        transient private Thread ioThread; //important!!!

        transient KafkaTaskContext kafkaTaskContext;

        //-------

        private Object heartbeatData;

        private String cancelledTaskIdInES;

        private ProgressStat progressStat;

    }

    @Data
    public static class KafkaTaskContext extends JobServiceImpl.TaskContextImpl implements JobService.TaskContext{

        final private HeavyPerTopicPartitionTaskListener.SubProgressStat subProgressStat=new HeavyPerTopicPartitionTaskListener.SubProgressStat();

    }


    @Data
    public static class ProgressStat implements Model{

        private long startOffset;

        private long taskOffset;

        private HeavyPerTopicPartitionTaskListener.SubProgressStat subProgressStat;

    }


    @Data
    static class TopicPartitions {

        ConcurrentMessageListenerContainer<String, String> concurrentMessageListenerContainer;

        private Map<String,ScheduledThreadPoolExecutor> heartbeatThread=new ConcurrentHashMap<>();

    }


}
