package com.ovopark.jobhub.sdk.client;

import com.ovopark.jobhub.sdk.model.*;
import com.ovopark.kernel.shared.Config;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.module.shared.BaseResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

import static com.ovopark.kernel.shared.Util.*;

@Slf4j
@Component("com.ovopark.jobhub.sdk.client.JobServiceImpl")
public class JobServiceImpl implements JobService,ApplicationContextAware{

    @Autowired
    private JobHubJobApi jobHubJobApi;

    @Autowired(required = false)
    private SpringBeanUriGetter springBeanUriGetter;

    private List<TaskListenerRunnerProvider> taskListenerRunnerProviderList;

    private ApplicationContext applicationContext;

    final private static String useDefaultProvider = Config.ConfigPriority.option().getString("JOBHUB_TASK_PROVIDER",null);

    @Autowired(required = false)
    private JobListenerKafkaRunnerProvider jobListenerKafkaRunnerProvider;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext=applicationContext;
    }

    @Autowired
    public void setTaskListenerRunnerProviderList(List<TaskListenerRunnerProvider> taskListenerRunnerProviderList) {
        this.taskListenerRunnerProviderList = taskListenerRunnerProviderList;
    }

    @Override
    public JobCreateResponse jobCreate(JobCreateRequest jobCreateRequest) {
        BaseResult<JobCreateResponse> baseResult = jobHubJobApi.jobCreate(jobCreateRequest);
        if (baseResult==null || baseResult.getIsError() || baseResult.getData()==null) {
            return null;
        }
        return baseResult.getData();
    }

    @Override
    public TaskCreateResponse taskCreate(TaskCreateRequest taskCreateRequest) {
        BaseResult<TaskCreateResponse> baseResult = jobHubJobApi.taskCreate(taskCreateRequest);
        if (baseResult==null || baseResult.getIsError() || baseResult.getData()==null) {
            return null;
        }
        return baseResult.getData();
    }

    @Override
    public TaskBulkCreateResponse taskBulkCreate(TaskBulkCreateRequest taskBulkCreateRequest) {
        BaseResult<TaskBulkCreateResponse> baseResult = jobHubJobApi.taskBulkCreate(taskBulkCreateRequest);
        if (baseResult==null || baseResult.getIsError() || baseResult.getData()==null) {
            return null;
        }
        return baseResult.getData();
    }

    @Override
    public TaskUpdateResponse taskUpdate(TaskUpdateRequest taskUpdateRequest) {
        BaseResult<TaskUpdateResponse> baseResult = jobHubJobApi.taskUpdate(taskUpdateRequest);
        if (baseResult==null || baseResult.getIsError() || baseResult.getData()==null) {
            return null;
        }
        return baseResult.getData();
    }

    @Override
    public TaskLogPutResponse taskLog(TaskLogPutRequest taskLogPutRequest) {
        BaseResult<TaskLogPutResponse> baseResult = jobHubJobApi.taskLog(taskLogPutRequest);
        if (baseResult==null || baseResult.getIsError() || baseResult.getData()==null) {
            return null;
        }
        return baseResult.getData();
    }

    @Override
    public TaskGetResponse taskGet(TaskGetRequest taskGetRequest) {
        BaseResult<TaskGetResponse> taskGetResponseBaseResult = jobHubJobApi.taskGet(taskGetRequest);
        if (taskGetResponseBaseResult==null || taskGetResponseBaseResult.getIsError() ||taskGetResponseBaseResult.getData()==null) {
            return null;
        }
        return taskGetResponseBaseResult.getData();
    }

    @Override
    public TaskMetaGetResponse taskMetaGet(TaskMetaGetRequest taskMetaGetRequest) {
        try {
            BaseResult<TaskMetaGetResponse> taskMetaGetResponseBaseResult = jobHubJobApi.taskMetaGet(taskMetaGetRequest);
            if (taskMetaGetResponseBaseResult==null || taskMetaGetResponseBaseResult.getIsError() ||taskMetaGetResponseBaseResult.getData()==null) {
                return null;
            }
            return taskMetaGetResponseBaseResult.getData();
        }
        catch (Exception e){
            log.error(e.getMessage());
            return null;
        }
    }

    @Override
    public void doOnceOnDevice(Integer deviceStatusId, Integer deviceId, Integer userId, Integer groupId
            , String type, String jsonStr, String desc, String jobId, TaskListener taskListener) {
        doOnceOnDevice(deviceStatusId, deviceId, userId, groupId, type, jsonStr, desc, jobId, new TaskCallListener<Void>() {
            @Override
            public Void on(TaskContext taskContext) {
                taskListener.on(taskContext);
                return null;
            }
        });
    }

    @Override
    public <T> T doOnceOnDevice(Integer deviceStatusId, Integer deviceId, Integer userId, Integer groupId, String type, String jsonStr, String desc, String jobId, TaskCallListener<T> taskCallListener) {
        TaskCreateRequest taskCreateRequest=new TaskCreateRequest();
        taskCreateRequest.setDeviceStatusId(deviceStatusId);
        taskCreateRequest.setDeviceId(deviceId);
        taskCreateRequest.setUserId(userId);
        taskCreateRequest.setGroupId(groupId);
        taskCreateRequest.setType(type);
        taskCreateRequest.setJsonStr(jsonStr);
        taskCreateRequest.setDesc(desc);
        taskCreateRequest.setJobId(jobId);

        TaskCreateResponse taskCreateResponse = taskCreate(taskCreateRequest);
        log.info("taskCreate: "+ JSONAccessor.impl().format(taskCreateResponse));
        TaskContextImpl taskContext= new TaskContextImpl();
        JobStatus jobStatus=JobStatus.COMPLETED;
        TaskUpdateRequest taskUpdateRequest=new TaskUpdateRequest();
        try {
            T f = taskCallListener.on(taskContext);
            jobStatus=convert2Self(taskContext.jobStatus,JobStatus.COMPLETED);
            return f;
        }
        catch (Exception e) {
            jobStatus=convert2Self(taskContext.jobStatus,JobStatus.FAIL);
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
        finally {
            if (!taskContext.isStatusManageManually()) {
                taskUpdateRequest.setId(taskCreateResponse.getId());
                taskUpdateRequest.setDocIndexName(taskCreateResponse.getDocIndexName());

                taskUpdateRequest.setStatus(jobStatus.name());
                taskUpdateRequest.setCompletedDesc(taskContext.getCompletedDesc());

                taskUpdateRequest.setRequestDeviceUrl(taskContext.getRequestDeviceUrl());
                taskUpdateRequest.setRequestDeviceArgs(taskContext.getRequestDeviceArgs());
                taskUpdateRequest.setResponseFromDevice(taskContext.getResponseFromDevice());

                TaskUpdateResponse taskUpdateResponse = taskUpdate(taskUpdateRequest);
                log.info("taskUpdate: "+ JSONAccessor.impl().format(taskUpdateResponse));
            }

            List<String> contentList = taskContext.getContentList();
            TaskLogPutRequest taskLogPutRequest =new TaskLogPutRequest();
            taskLogPutRequest.setDeviceStatusId(deviceStatusId);
            taskLogPutRequest.setDeviceId(deviceId);
            taskLogPutRequest.setJobId(jobId);
            taskLogPutRequest.setTaskId(taskCreateResponse.getId());
            taskLogPutRequest.setType(type);
            taskLogPutRequest.setContentList(contentList);

            TaskLogPutResponse taskLogPutResponse = taskLog(taskLogPutRequest);
            log.info("taskLog: "+ JSONAccessor.impl().format(taskLogPutResponse));

        }
    }

    @Override
    public Long cronTaskRegister(String name, String bean, String cron){
        final String uri = springBeanUriGetter.springBean(bean);
        return cronTaskRegister0(name, bean, cron, uri,null,null);
    }

    private Long cronTaskRegister0(String name, String bean,String cron, String uri
            ,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig,String kafkaTopic
    ) {
        try {
            CronTaskSaveRequest cronTaskSaveRequest=new CronTaskSaveRequest();
            cronTaskSaveRequest.setName(name);
            cronTaskSaveRequest.setLockName(true); // important!!!
            cronTaskSaveRequest.setCreateBy(-1);
            cronTaskSaveRequest.setModifyBy(-1);

            cronTaskSaveRequest.setUri(uri);
            cronTaskSaveRequest.setArgs(null);
            cronTaskSaveRequest.setCron(cron);
    //        cronTaskSaveRequest.setCron("0/10 * * * * ?");
            BaseResult<CronTaskSaveResponse> baseResult = jobHubJobApi.saveCronTask(cronTaskSaveRequest);
            log.info("create cron task result: "+ JSONAccessor.impl().format(baseResult));

    //        if (baseResult==null || baseResult.getIsError()
    //                || baseResult.getData()==null || baseResult.getData().getId()==null) {
    //            throw new IllegalStateException("cannot create a new cron task.");
    //        }

            if (baseResult!=null && !baseResult.getIsError() && baseResult.getData()!=null && baseResult.getData().getId()!=null) {
                return baseResult.getData().getId();
            }
            return null;
        }
        catch (Exception e){
            log.error(e.getMessage(),e);
            return null;
        }
        finally {
            // start kafka client
            if (jobListenerRunnerKafkaConfig!=null) {
                jobListenerKafkaRunnerProvider.start(kafkaTopic,bean,jobListenerRunnerKafkaConfig);
            }
        }
    }

    @Override
    public Long cronTaskRegisterViaKafka(String name, String bean,String cron
            , String kafkaTopic, String partitionKey,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig) {
        final String uri = DelayTaskExecutor.kafkaChannel(kafkaTopic,partitionKey,bean);
        return cronTaskRegister0(name, bean, cron, uri,jobListenerRunnerKafkaConfig,kafkaTopic);
    }

    @Override
    public Long delayTaskRegister(String name, String bean, String args, long triggerTimeMs, long endTimeMs
            , int triggerIfMiss, Integer createBy) {
        final String uri = springBeanUriGetter.springBean(bean);
        return delayTaskRegister0(name, args, bean,triggerTimeMs, endTimeMs, triggerIfMiss, createBy, uri,null,null);
    }

    private Long delayTaskRegister0(String name, String bean,String args, long triggerTimeMs, long endTimeMs
            , int triggerIfMiss, Integer createBy, String uri
            ,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig
                                    ,String kafkaTopic
    ) {
        DelayTaskSaveRequest cronTaskSaveRequest=new DelayTaskSaveRequest();
        cronTaskSaveRequest.setName(name);
        cronTaskSaveRequest.setCreateBy(createBy);
        cronTaskSaveRequest.setModifyBy(-1);

        cronTaskSaveRequest.setUri(uri);
        cronTaskSaveRequest.setArgs(args);
        cronTaskSaveRequest.setTriggerIfMiss(triggerIfMiss);
        cronTaskSaveRequest.setTriggerTimeMs(triggerTimeMs);
        cronTaskSaveRequest.setEndTimeMs(endTimeMs);

        try {
    //        cronTaskSaveRequest.setCron("0/10 * * * * ?");
            BaseResult<DelayTaskSaveResponse> baseResult = jobHubJobApi.saveDelayTask(cronTaskSaveRequest);
            log.info("create delay task result: "+ JSONAccessor.impl().format(baseResult));

    //        if (baseResult==null || baseResult.getIsError()
    //                || baseResult.getData()==null || baseResult.getData().getId()==null) {
    //            throw new IllegalStateException("cannot create a new cron task.");
    //        }

            if (baseResult!=null && !baseResult.getIsError() && baseResult.getData()!=null && baseResult.getData().getId()!=null) {
                return baseResult.getData().getId();
            }
            return null;
        }
        catch (Exception e){
            log.error(e.getMessage(),e);
            return null;
        }
        finally {
            // start kafka client
            if (jobListenerRunnerKafkaConfig!=null) {
                jobListenerKafkaRunnerProvider.start(kafkaTopic,bean,jobListenerRunnerKafkaConfig);
            }
        }
    }

    @Override
    public Long delayTaskRegisterViaKafka(String name,String bean, String args
            , long triggerTimeMs, long endTimeMs, int triggerIfMiss, Integer createBy
            , String kafkaTopic, String partitionKey,JobListenerRunnerKafkaConfig jobListenerRunnerKafkaConfig) {
        final String uri = DelayTaskExecutor.kafkaChannel(kafkaTopic,partitionKey,bean);
        return delayTaskRegister0(name, bean,args, triggerTimeMs, endTimeMs, triggerIfMiss, createBy, uri
                ,jobListenerRunnerKafkaConfig,kafkaTopic);
    }

    @Override
    public TaskContext mockTaskContext(TaskGetResponse.Task task) {
        TaskContextImpl taskContext=new TaskContextImpl();
        taskContext.setTask(task);
        return taskContext;
    }

    @Override
    public JobContext mockJobContext(String name, String uri, String args, String jobIdInES) {
        if (isEmpty(jobIdInES)) {
            //create new job
            JobCreateRequest jobCreateRequest=new JobCreateRequest();
            jobCreateRequest.setJsonStr(args);
            jobCreateRequest.setType("mockJobContext");
            jobCreateRequest.setDesc("mockJobContext");
            JobCreateResponse jobCreateResponse = jobCreate(jobCreateRequest);
            jobIdInES=jobCreateResponse.getId();
        }

        String finalJobIdInES = jobIdInES;
        return new JobContext() {
            @Override
            public String name() {
                return name;
            }

            @Override
            public String uri() {
                return uri;
            }

            @Override
            public String args() {
                return args;
            }

            @Override
            public String jobIdInES() {
                return finalJobIdInES;
            }

            @Override
            public JobService jobService() {
                return JobServiceImpl.this;
            }
        };
    }

    @Override
    public boolean register(String jobType, String beanUrl,String group, Long minVer) {
        TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig=new TaskListenerRunnerProviderConfig();
        taskListenerRunnerProviderConfig.setConcurrency(1);
        return register(jobType, beanUrl, group, minVer,taskListenerRunnerProviderConfig);
    }

    @Override
    public boolean register(String jobType, String beanUrl, String group, Long minVer, TaskListenerRunnerProviderConfig taskListenerRunnerProviderConfig) {
        Object object = applicationContext.getBean(beanUrl);
        if (!(object instanceof TaskListener)) {
            throw new RuntimeException("bean must implement "+TaskListener.class.getName());
        }

        String realProvider= isNotEmpty(useDefaultProvider)? useDefaultProvider :taskListenerRunnerProviderConfig.getProvider();

        TaskListenerRunnerProvider found=null;
        for (TaskListenerRunnerProvider taskListenerRunnerProvider : taskListenerRunnerProviderList) {
            if (realProvider.equalsIgnoreCase(taskListenerRunnerProvider.name())) {
                found=taskListenerRunnerProvider;
                break;
            }
        }

        if (found==null) {
            throw new RuntimeException("cannot find provider, missing kafka dependency ???: jobhub-sdk-client-kafka ");
        }

        return found.start(jobType,beanUrl,group,minVer,((TaskListener) object),taskListenerRunnerProviderConfig);
    }

    @Data
    public static class TaskContextImpl implements TaskContext{

        public static final int MAX_CAPACITY = 100;

        JobStatus jobStatus;

        String completedDesc;

        List<String> contentList=new ArrayList<>(MAX_CAPACITY);

        String requestDeviceUrl;

        String requestDeviceArgs;

        String responseFromDevice;

        TaskGetResponse.Task task;

        boolean statusManageManually;

        @Override
        public void status(JobStatus jobStatus) {
            this.jobStatus=jobStatus;
        }

        @Override
        public void completedDesc(String completedDesc) {
            this.completedDesc=completedDesc;
        }

        @Override
        synchronized public void appendLog(String content) {
            if (contentList.size()>MAX_CAPACITY) {
                return;
            }
            contentList.add(content);
        }

        @Override
        public TaskGetResponse.Task task() {
            return task;
        }

        @Override
        public void statusManageManually() {
            statusManageManually=true;
        }

        @Override
        public synchronized void addRetryCount() {
            task.setRetryCount(task().getRetryCount()+1);
        }

        @Override
        public void requestDeviceUrl(String requestUrl) {
            this.requestDeviceUrl=requestUrl;
        }

        @Override
        public void requestDeviceArgs(String requestArgs) {
            this.requestDeviceArgs=requestArgs;
        }

        @Override
        public void responseFromDevice(String response) {
            this.responseFromDevice=response;
        }
    }

}
