package com.ovopark.jobhub.sdk.client;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.ovopark.jobhub.sdk.model.JobCaptureRequest;
import com.ovopark.jobhub.sdk.model.JobCaptureResponse;
import com.ovopark.jobhub.sdk.model.internal.TaskLockRequest;
import com.ovopark.jobhub.sdk.model.internal.TaskLockResponse;
import com.ovopark.jobhub.sdk.model.internal.TaskSubmitRequest;
import com.ovopark.jobhub.sdk.model.internal.TaskSubmitResponse;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.vclient.ClientNode;
import com.ovopark.module.shared.BaseResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Slf4j
@RestController("com.ovopark.jobhub.sdk.client.JobEndpoint")
@RequestMapping("/feign/jobhub-job/processing")
public class JobEndpoint {

    @Autowired
    private DelayTaskExecutor delayTaskExecutor;

    @Autowired
    private Client2ControlTransport client2ControlTransport;

    @Autowired
    private JobService jobService;

    private static final ExecutorService jobExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,64)
                    , Integer.parseInt(System.getProperty("JOBHUB_JOB_IO", "0")))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,1024)
            , Integer.parseInt(System.getProperty("JOBHUB_JOB_IO", "0")))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(1)
            , newThreadFactory("jobhub-job-io")
            , new ThreadPoolExecutor.AbortPolicy());

    @Autowired
    JobInnerContextMgr jobInnerContextMgr;

    @RequestMapping("/submit")
    @ResponseBody
    public BaseResult<TaskSubmitResponse> submit(@RequestBody TaskSubmitRequest taskSubmitRequest){
        final long cronTaskId = taskSubmitRequest.getCronTaskId();
        final long cronTaskHistoryId = taskSubmitRequest.getCronTaskHistoryId();
        final String taskKey = "task_"+jobInnerContextMgr.key(cronTaskId,cronTaskHistoryId);
        MDC.put("traceId", taskKey);
        MDC.put("requestId", taskKey);
        log.info("receive a new task, we can do it???: "+JSONAccessor.impl().format(taskSubmitRequest));
        final TaskSubmitResponse taskSubmitResponse=new TaskSubmitResponse();
        taskSubmitResponse.setSubmitted(false);
        JobLog logBeforeSubmitted=new JobLogImpl(client2ControlTransport,taskKey, cronTaskHistoryId);
        logLink("receive a new task, we can do it???: "+JSONAccessor.impl().format(taskSubmitRequest))
                .log(logBeforeSubmitted::log);
        try {
            lock(taskKey, new Callable<Void>() {
                @Override
                public Void call() throws Exception {

                    if (jobInnerContextMgr.exists(cronTaskId,cronTaskHistoryId)) {
                        taskSubmitResponse.setSubmitted(false);
                        taskSubmitResponse.setDesc("Rejected,the previous task is executing");
                        logLink("Rejected,the previous task is executing")
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw new IllegalStateException("the previous task is executing, reject the task: "
                                +JSONAccessor.impl().format(taskSubmitRequest));
                    }

                    TaskLockRequest taskLockRequest=new TaskLockRequest();
                    taskLockRequest.setCronTaskId(cronTaskId);
                    taskLockRequest.setCronTaskHistoryId(cronTaskHistoryId);
                    taskLockRequest.setNode(ClientNode.UUID_STR);
                    TaskLockResponse taskLockResponse = client2ControlTransport.lockClient(taskLockRequest);
                    if (taskLockResponse == null || !taskLockResponse.isSuccess()) {
                        logLink("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest))
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw new IllegalStateException("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest));
                    }

                    final JobLog jobLog = new JobLogImpl(client2ControlTransport, taskKey, cronTaskHistoryId);
                    JobContextImpl jobContext =new JobContextImpl(taskSubmitRequest, jobService);
                    jobContext.setCronTaskId(cronTaskId);
                    jobContext.setCronTaskHistoryId(cronTaskHistoryId);
                    try {
                        Future<?> future = jobExecutor.submit(catchRunnable(() -> {
                            //OK , start doing real task logic
                            jobContext.setIoThread(Thread.currentThread());
                            try {
                                delayTaskExecutor.execute(jobContext, jobLog);
                            }
                            catch (Exception e) {
                                log.error(e.getMessage(), e);
                                jobLog.flush();
                                errorStackList(e).forEach(jobLog::log);
                                jobLog.flush();
                            }
                            finally {
                                jobInnerContextMgr.remove(cronTaskId,cronTaskHistoryId);
                                try {
                                    logLink(taskKey + " > completed: " + formatTime(LocalDateTime.now()))
                                            .log(log::info).log(jobLog::log);
                                    jobLog.flush();
                                    jobLog.close();
                                } catch (Exception e) {
                                    log.error(e.getMessage(), e);
                                }
                            }
                        }));
                        jobContext.setFuture(future);
                        logLink("OK, the task is scheduled: "+taskKey)
                                .log(log::info).log(logBeforeSubmitted::log);
                        //OK ,  resource is available , executor accepts the task
                        taskSubmitResponse.setSubmitted(true);
                        jobInnerContextMgr.add(cronTaskId,cronTaskHistoryId, jobContext);
                    }
                    catch (RejectedExecutionException e) {
                        // resource is unavailable
                        taskSubmitResponse.setSubmitted(false);
                        taskSubmitResponse.setDesc("Rejected, resource is unavailable ");
                        logLink("Rejected, resource is unavailable ")
                                .log(log::info).log(logBeforeSubmitted::log);
                        throw convert2RuntimeException(e);
                    }
                    return null;
                }
            },15,TimeUnit.SECONDS);

        }
        catch (Exception e) {
            log.error(e.getMessage(),e);
            logBeforeSubmitted.flush();
            errorStackList(e).forEach(logBeforeSubmitted::log);
            logBeforeSubmitted.flush();
        }
        finally {
            try {
                logBeforeSubmitted.close();
            }
            catch (Exception e){
                log.error(e.getMessage(),e);
            }
            finally {
                MDC.remove("traceId");
                MDC.remove("requestId");
            }
        }
        return BaseResult.success(taskSubmitResponse);
    }


    @Data
    public static class JobContextImpl implements JobContext , JobInnerContext{

        final TaskSubmitRequest taskSubmitRequest;

        final JobService jobService;

        private long cronTaskId;

        private long cronTaskHistoryId;

        private Future<?> future;

        private boolean killed;

        @JsonIgnore
        private transient Thread ioThread;

        final private long startTime=System.currentTimeMillis();

        public JobContextImpl(TaskSubmitRequest taskSubmitRequest, JobService jobService) {
            this.taskSubmitRequest = taskSubmitRequest;
            this.jobService = jobService;
        }

        @Override
        public Long jobId() {
            return taskSubmitRequest.getCronTaskId();
        }

        @Override
        public String name() {
            return taskSubmitRequest.getName();
        }

        @Override
        public String uri() {
            return taskSubmitRequest.getUri();
        }

        @Override
        public String args() {
            return taskSubmitRequest.getArgs();
        }

        @Override
        public String jobIdInES() {
            return taskSubmitRequest.getJobIdInES();
        }

        @Override
        public JobService jobService() {
            return jobService;
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Override
        public void heartbeatData(Object data) {

        }

        @Override
        public long taskId() {
            return cronTaskId;
        }

        @Override
        public Future<?> future() {
            return future;
        }

        @Override
        public boolean killed() {
            return killed;
        }

        @Override
        public Thread ioThread() {
            return ioThread;
        }
    }


    @RequestMapping("/capture")
    @ResponseBody
    public JobCaptureResponse capture(@RequestBody JobCaptureRequest jobCaptureRequest){
        JobCaptureResponse jobCaptureResponse =new JobCaptureResponse();
        jobCaptureResponse.setSuccess(true);

        long cronTaskId = jobCaptureRequest.getCronTaskId();
        long cronTaskHistoryId = jobCaptureRequest.getCronTaskHistoryId();
        JobContextImpl jobContext = (JobContextImpl) jobInnerContextMgr.jobInnerContext(cronTaskId, cronTaskHistoryId);
        jobCaptureResponse.setAllTaskCountInNode(jobInnerContextMgr.count());
        jobCaptureResponse.setFound(false);
        if (jobContext==null) {
            return jobCaptureResponse;
        }
        jobCaptureResponse.setFound(true);
        jobCaptureResponse.setCronTaskId(cronTaskId);
        jobCaptureResponse.setCronTaskHistoryId(cronTaskHistoryId);
        jobCaptureResponse.setStartTime(jobContext.getStartTime());
        jobCaptureResponse.setStartTimeStr(formatTime(dateTime(jobContext.getStartTime())));

        if (jobCaptureRequest.isCaptureStack()) {
            Thread ioThread = jobContext.getIoThread();
            StackTraceElement[] stackTraceElements = ioThread.getStackTrace();
            List<String> stackList=new ArrayList<>(stackTraceElements.length);
            for (StackTraceElement stackTraceElement : stackTraceElements) {
                stackList.add(stackTraceElement.toString());
            }

            jobCaptureResponse.setIoThread(ioThread.getName());
            jobCaptureResponse.setIoStackList(stackList);
        }

        return jobCaptureResponse;
    }

}

