package com.ovopark.knife.client.register;
import com.alibaba.fastjson.JSONObject;
import com.ovopark.boot.kit.http.HttpClientKit;
import com.ovopark.boot.kit.lang.StrKit;
import com.ovopark.knife.client.properties.OpenPlatformApiProperties;
import com.ovopark.knife.common.annotation.OpenPlatformApi;
import com.ovopark.knife.common.comm.Const;
import com.ovopark.knife.common.model.RegisterInfo;
import com.ovopark.knife.common.utils.IPKit;
import org.reflections.Reflections;
import org.reflections.scanners.MethodAnnotationsScanner;
import org.reflections.scanners.MethodParameterNamesScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Created by xuxing on 2019/9/10
 * <p>
 * 扫描@OpenPlatformApi注解，向注册中心注册路由
 */
@Configuration
@EnableConfigurationProperties(OpenPlatformApiProperties.class)
public class RegisterHandler {

  Logger log = LoggerFactory.getLogger(RegisterHandler.class);

  final static String OpenMethod = "api.method";

  @Resource
  private OpenPlatformApiProperties openPlatformApiProperties;

  @Resource
  private Environment env;

  @Bean
  @Order(99)
  public OpenPlatformApiProperties register() {
    try {
      if (!openPlatformApiProperties.getEnable()) {
        return openPlatformApiProperties;
      }
      if (StrKit.isEmpty(openPlatformApiProperties.getRegisterUrl())) {
        throw new RuntimeException("register centre url is null");
      }
      String appName = StrKit.isNotEmpty(openPlatformApiProperties.getApplicationName())
          ? openPlatformApiProperties.getApplicationName() : StrKit.isNotEmpty(env.getProperty("spring.application.name"))
          ? env.getProperty("spring.application.name") : env.getProperty("dubbo.application.name");

      if (StrKit.isEmpty(appName)) {
        throw new RuntimeException("application name is null");
      }
      // zk 地址
      String zkaddress = StrKit.isNotEmpty(openPlatformApiProperties.getZkAddress())
          ? openPlatformApiProperties.getZkAddress() : env.getProperty("dubbo.registry.address");

      Reflections reflections = new Reflections(openPlatformApiProperties.getScanPackage(),
          new MethodAnnotationsScanner(), new MethodParameterNamesScanner());

      // 获取注解的方法
      Set<Method> methodsAnnotatedWith = reflections.getMethodsAnnotatedWith(OpenPlatformApi.class);

      // 服务环境
      String environment = env.getProperty("spring.profiles.active");

      Map<String, String> routeUrlMap = new HashMap<>();
      Map<String, String> dubboRouteUrlMap = new HashMap<>();
      for (Method method : methodsAnnotatedWith) {
        //获取方法上的注解
        OpenPlatformApi annotation = method.getAnnotation(OpenPlatformApi.class);
        // 获取接口名
        String interfaceName = method.getDeclaringClass().getName();
        // 方法名
        String methodName = method.getName();

        dubboRouteUrlMap.put(createRouteKey(annotation.mt(), annotation.version().getKey()), interfaceName + "-" + methodName);

        routeUrlMap.put(createRouteKey(annotation.mt(), annotation.version().getKey()), annotation.route());

        // 如果zk 地址不为空，则添加zk 地址、接口信息
        log.info("method name = {} ;route = {} ;mt = {}", methodName, annotation.route(), annotation.mt());
      }

      RegisterInfo registerInfo = new RegisterInfo();
      registerInfo.setApplicationName(appName);
      registerInfo.setRouteUrlMap(routeUrlMap);
      registerInfo.setZkAddress(zkaddress);
      registerInfo.setEnv(environment);
      registerInfo.setRouteDubboMap(dubboRouteUrlMap);
      registerInfo.setIp(IPKit.getV4IP() + "/" + IPKit.getInnetIp() + ":" + env.getProperty("server.port"));

      String result = HttpClientKit.init().doPostJson(openPlatformApiProperties.getRegisterUrl(), JSONObject.toJSON(registerInfo));

      if (!"success".equals(result)) {
        throw new RuntimeException("Registration to " + openPlatformApiProperties.getRegisterUrl() + " failed !!!!");
      }
      log.info("Registration successfully to {} !!!!", openPlatformApiProperties.getRegisterUrl());
    } catch (Exception e) {
      log.error(e.getMessage());
    }
    return openPlatformApiProperties;
  }

  /**
   * @throws
   * @Title: createRouteKey
   * @Description: TODO(创建redis 路由的key)
   * @param: @param method _mt参数-方法
   * @param: @param version _version参数-版本号
   * @param: @param environment
   * @param: @return
   * @return: String
   */
  private static String createRouteKey(String method, String version) {
    return OpenMethod + Const.LINE + version + Const.LINE + method;
  }

}
