package com.ovopark.messagehub.sdk.job;

import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.concurrent.ReleasableLock;
import com.ovopark.messagehub.sdk.internal.MessageHubJobApi;
import com.ovopark.messagehub.sdk.model.internal.job.TaskLogRequest;
import com.ovopark.messagehub.sdk.model.internal.job.TaskLogResponse;
import com.ovopark.module.shared.BaseResult;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.*;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import static com.ovopark.kernel.shared.Util.catchRunnable;
import static com.ovopark.kernel.shared.Util.newThreadFactory;

@Slf4j
public class JobLogImpl implements JobLog{


    private static final ExecutorService jobLogExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,16)
                    , Integer.parseInt(System.getProperty("MESSAGEHUB_JOB_LOG_IO", "0")))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,64)
            , Integer.parseInt(System.getProperty("MESSAGEHUB_JOB_LOG_IO", "0")))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(1)
            , newThreadFactory("messagehub-job-log-io")
            , new ThreadPoolExecutor.CallerRunsPolicy());


    final MessageHubJobApi messageHubJobApi;

    final Queue<String> contentQueue;

    final String traceId;

    private volatile long lastSyncLogTimeMs=System.currentTimeMillis();


    private final ReentrantReadWriteLock dataReadWriteLock =new ReentrantReadWriteLock();
    private final ReleasableLock dataSLock = new ReleasableLock(dataReadWriteLock.readLock());
    private final ReleasableLock dataXLock = new ReleasableLock(dataReadWriteLock.writeLock());

    private volatile boolean closed;


    public JobLogImpl(MessageHubJobApi messageHubJobApi,int size,String traceId) {
        this.messageHubJobApi = messageHubJobApi;
        contentQueue=new ArrayBlockingQueue<>(size);
        this.traceId=traceId;
    }



    @Override
    public void log(String content) {
        if (closed) {
            return;
        }

        final String  finalContent=traceId+" > "+content;

        if (!contentQueue.offer(finalContent)) {
            dataXLock.acquire();
            try {
                if (closed) {
                    return;
                }
                final long now=System.currentTimeMillis();
                if (now-lastSyncLogTimeMs>1_000) {
                    List<String> list=new ArrayList<>(contentQueue.size()+1);
                    list.addAll(contentQueue);
                    list.add(finalContent);
                    TaskLogRequest taskLogRequest=new TaskLogRequest();
                    taskLogRequest.setLogList(list);
                    taskLogRequest.setNode(ClientNode.UUID_STR);
                    // clear
                    contentQueue.clear();
                    lastSyncLogTimeMs=now;

                    //sync log
                    jobLogExecutor.execute(catchRunnable(() -> {
                        BaseResult<TaskLogResponse> baseResult = messageHubJobApi.log(taskLogRequest);
                        log.info("sync log success: "+ JSONAccessor.impl().format(baseResult));
                    }));
                }
                else {
                    contentQueue.poll();
                    contentQueue.offer(finalContent);
                }
            }
            finally {
                dataXLock.unlock();
            }

        }

    }

    @Override
    public void flush() {
        dataXLock.acquire();
        try {
            List<String> list = new ArrayList<>(contentQueue.size());
            list.addAll(contentQueue);
            TaskLogRequest taskLogRequest = new TaskLogRequest();
            taskLogRequest.setLogList(list);
            taskLogRequest.setNode(ClientNode.UUID_STR);
            // clear
            contentQueue.clear();

            //sync log
            jobLogExecutor.execute(catchRunnable(() -> {
                BaseResult<TaskLogResponse> baseResult = messageHubJobApi.log(taskLogRequest);
                log.info("sync log success: " + JSONAccessor.impl().format(baseResult));
            }));
        }
        finally {
            closed=true;
            dataXLock.unlock();
        }
    }
}
