package com.ovopark.messagehub.sdk.job;

import com.ovopark.kernel.shared.CachedExecutors;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.kv.CacheService;
import com.ovopark.kernel.shared.vclient.ClientNode;
import com.ovopark.messagehub.sdk.internal.MessageHubJobApi;
import com.ovopark.messagehub.sdk.model.internal.job.TaskLockRequest;
import com.ovopark.messagehub.sdk.model.internal.job.TaskLockResponse;
import com.ovopark.messagehub.sdk.model.internal.job.TaskSubmitRequest;
import com.ovopark.messagehub.sdk.model.internal.job.TaskSubmitResponse;
import com.ovopark.module.shared.BaseResult;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.Set;
import java.util.concurrent.*;
import java.util.function.Supplier;

import static com.ovopark.kernel.shared.Util.*;

@JobClientActive
@Slf4j
@RestController("com.ovopark.messagehub.sdk.job.JobEndpoint")
@RequestMapping("/feign/messagehub-job/processing")
public class JobEndpoint {

    @Autowired
    private DelayTaskExecutor delayTaskExecutor;

    @Autowired
    private MessageHubJobApi messageHubJobApi;

    private static final ExecutorService jobExecutor = new ThreadPoolExecutor(
            Math.max(Math.min(Runtime.getRuntime().availableProcessors() * 2,64)
                    , Integer.parseInt(System.getProperty("MESSAGEHUB_JOB_IO", "0")))
            , Math.max(Math.max(Runtime.getRuntime().availableProcessors() * 2,1024)
            , Integer.parseInt(System.getProperty("MESSAGEHUB_JOB_IO", "0")))
            , 600, TimeUnit.SECONDS
            //Integer.MAX_VALUE ???
            , new LinkedBlockingQueue(1)
            , newThreadFactory("messagehub-job-io")
            , new ThreadPoolExecutor.AbortPolicy());

    private static final CacheService<String, TaskSubmitRequest> C=new CacheService.MapCacheService<>(true
            , CachedExecutors.impl("messagehub-job-io-c",1,1)
            );

    final private Set<Long> executingCronTaskIdSet = ConcurrentHashMap.newKeySet();

    @RequestMapping("/submit")
    @ResponseBody
    public BaseResult<TaskSubmitResponse> submit(@RequestBody TaskSubmitRequest taskSubmitRequest){
        final String taskKey = "task_" + taskSubmitRequest.getCronTaskId() + "_" + taskSubmitRequest.getCronTaskHistoryId();
        MDC.put("traceId", taskKey);
        MDC.put("requestId", taskKey);
        log.info("receive a new task, we can do it???: "+JSONAccessor.impl().format(taskSubmitRequest));
        final TaskSubmitResponse taskSubmitResponse=new TaskSubmitResponse();
        taskSubmitResponse.setSubmitted(false);
        try {
            lock("task:" + taskSubmitRequest.getCronTaskId(), new Callable<Void>() {
                @Override
                public Void call() throws Exception {
                    // same task is submit in the same node , 10min???
                    // skip the same task ??? task id + task history id
                    C.putIfAbsentAndGet(taskKey, new Supplier<TaskSubmitRequest>() {
                        @Override
                        public TaskSubmitRequest get() {

                            if (executingCronTaskIdSet.contains(taskSubmitRequest.getCronTaskId())) {
                                taskSubmitResponse.setSubmitted(false);
                                taskSubmitResponse.setDesc("Rejected,the previous task is executing");
                                throw new IllegalStateException("the previous task is executing, reject the task: "
                                        +JSONAccessor.impl().format(taskSubmitRequest));
                            }

                            TaskLockRequest taskLockRequest=new TaskLockRequest();
                            taskLockRequest.setCronTaskId(taskSubmitRequest.getCronTaskId());
                            taskLockRequest.setCronTaskHistoryId(taskSubmitRequest.getCronTaskHistoryId());
                            taskLockRequest.setNode(ClientNode.UUID_STR);
                            BaseResult<TaskLockResponse> baseResult = messageHubJobApi.lockClient(taskLockRequest);
                            if (baseResult == null || baseResult.getIsError()) {
                                throw new IllegalStateException("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest));
                            }
                            TaskLockResponse taskLockResponse = baseResult.getData();
                            if (taskLockResponse == null || !taskLockResponse.isSuccess()) {
                                throw new IllegalStateException("cannot get lock: "+ JSONAccessor.impl().format(taskLockRequest));
                            }

                            executingCronTaskIdSet.add(taskSubmitRequest.getCronTaskId());
                            try {
                                jobExecutor.execute(catchRunnable(() -> {
                                    //OK , start doing real task logic
                                    JobLog jobLog=new JobLogImpl(messageHubJobApi,taskKey,taskSubmitRequest.getCronTaskHistoryId());
                                    try {
                                        delayTaskExecutor.execute(taskSubmitRequest.getUri(), taskSubmitRequest.getArgs(),jobLog);
                                    }
                                    finally {
                                        executingCronTaskIdSet.remove(taskLockRequest.getCronTaskId());
                                        try {
                                            jobLog.log(taskKey+" > completed: "+formatTime(LocalDateTime.now()));
                                            jobLog.flush();
                                        }catch (Exception e){
                                            log.error(e.getMessage(),e);
                                        }
                                    }
                                }));
                                log.info("OK, the task is scheduled: "+taskKey);
                                //OK ,  resource is available , executor accepts the task
                                taskSubmitResponse.setSubmitted(true);
                            } catch (RejectedExecutionException e) {
                                // resource is unavailable
                                executingCronTaskIdSet.remove(taskLockRequest.getCronTaskId());
                                taskSubmitResponse.setSubmitted(false);
                                taskSubmitResponse.setDesc("Rejected, resource is unavailable ");
                                throw convert2RuntimeException(e);
                            }
                            return taskSubmitRequest;
                        }
                    }, 600, TimeUnit.SECONDS);

                    return null;
                }
            },10,TimeUnit.SECONDS);

        } catch (Exception e) {
            log.error(e.getMessage(),e);
        }
        finally {
            MDC.remove("traceId");
            MDC.remove("requestId");
        }
        return BaseResult.success(taskSubmitResponse);
    }

}
