package com.ovopark.open.platform.sms.api;

import com.alibaba.fastjson.JSON;
import com.ovopark.open.platform.sms.api.router.DefaultSmsRouter;
import com.ovopark.open.platform.sms.api.router.SmsRouter;
import lombok.Getter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;
import org.springframework.util.DigestUtils;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 发送短信 .
 * @author <a href="mailto:guyue375@outlook.com">Klaus.turbo</a>
 **/
@Getter
@ToString
public class SendMessageAction {
    
    private String nationCode = "+86";
    
    private String phoneNum;
    
    private String templateCode;
    
    private String templateParamJson;
    
    private String encryptionResult;
    
    private Map<String, String> nationCodeRoutingTable = new HashMap<>();
    
    private SmsRouter smsRouter;
    
    private HttpClient httpClient;
    
    public HttpResponse<String> sendMessage() throws IOException, InterruptedException {
        
        String smsEndpoint = this.nationCodeRoutingTable.get(this.nationCode);
        if (StringUtils.isBlank(smsEndpoint)) {
            smsEndpoint = this.smsRouter.apply(this.nationCode);
        }
        String smsAddress = smsEndpoint + ApiConstant.DEFAULT_SMS_URI;
        final HttpRequest request = HttpRequest.newBuilder(URI.create(smsAddress))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers.ofString(JSON.toJSONString(this))).build();
        return httpClient.send(request, HttpResponse.BodyHandlers.ofString());
    }
    
    
    public static SendMessageAction.Builder builder() {
        return new SendMessageAction.Builder();
    }
    
    public static class Builder {
        
        private String nationCode = "+86";
        
        private String phoneNum;
        
        private String templateCode;
        
        private String templateParamJson;
        
        private Map<String, String> nationCodeRoutingTable;
        
        private String salt;
        
        private SmsRouter smsRouter;
        
        private HttpClient httpClient;
        
        
        public SendMessageAction.Builder nationCode(String nationCode) {
            this.nationCode = nationCode;
            return this;
        }
        
        public SendMessageAction.Builder phoneNum(String phoneNum) {
            this.phoneNum = phoneNum;
            return this;
        }
        
        public SendMessageAction.Builder templateCode(String templateCode) {
            this.templateCode = templateCode;
            return this;
        }
        
        public SendMessageAction.Builder templateParamJson(String templateParamJson) {
            this.templateParamJson = templateParamJson;
            return this;
        }
        
        public SendMessageAction.Builder salt(String salt) {
            this.salt = salt;
            return this;
        }
        
        public SendMessageAction.Builder nationCodeRoutingTable(Map<String, String> nationCodeRoutingTable) {
            this.nationCodeRoutingTable = nationCodeRoutingTable;
            return this;
        }
        
        public SendMessageAction.Builder smsRouter(SmsRouter smsRouter) {
            this.smsRouter = smsRouter;
            return this;
        }
        
        public SendMessageAction.Builder httpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }
        
        /**
         * 简单参数加密.
         * @param key 被加密字段
         * @return
         */
        private static String simpleSign(String key, String salt) {
            return DigestUtils.md5DigestAsHex((key + salt).getBytes(StandardCharsets.UTF_8)).toUpperCase();
        }
        
        private String encrypt(String salt) {
            String key = String.join(" ", StringUtils.isBlank(this.nationCode) ? StringUtils.EMPTY : this.nationCode,
                    this.phoneNum, this.templateCode, this.templateParamJson);
            return simpleSign(key, salt);
        }
        
        public SendMessageAction build() {
            Assert.isTrue(StringUtils.isNotBlank(this.nationCode), "区号不能为空");
            Assert.isTrue(StringUtils.isNotBlank(this.phoneNum), "手机号不能为空");
            Assert.isTrue(StringUtils.isNotBlank(this.templateCode), "阿里云短信模版不能为空");
            Assert.isTrue(StringUtils.isNotBlank(this.templateParamJson), "阿里云短信模版参数不能为空");
            Assert.isTrue(StringUtils.isNotBlank(this.salt), "加密salt不能为空");
            SendMessageAction action = new SendMessageAction();
            action.nationCode = this.nationCode;
            action.phoneNum = this.phoneNum;
            action.templateCode = this.templateCode;
            action.templateParamJson = this.templateParamJson;
            action.nationCodeRoutingTable.putAll(this.nationCodeRoutingTable);
            action.encryptionResult = encrypt(this.salt);
            
            if (Objects.isNull(this.smsRouter)) {
                action.smsRouter = new DefaultSmsRouter();
            } else {
                action.smsRouter = this.smsRouter;
            }
            
            if (Objects.isNull(this.httpClient)) {
                action.httpClient = HttpClient.newBuilder().build();
            } else {
                action.httpClient = this.httpClient;
            }
            return action;
        }
    }
    
}
