package com.ovopark.open.platform.sms.api;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.annotation.JSONField;
import com.google.i18n.phonenumbers.NumberParseException;
import com.google.i18n.phonenumbers.PhoneNumberUtil;
import com.google.i18n.phonenumbers.Phonenumber;
import com.ovopark.open.platform.sms.api.router.DefaultSmsRouter;
import com.ovopark.open.platform.sms.api.router.SmsRouter;
import lombok.Getter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;
import org.springframework.util.DigestUtils;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 发送短信 .
 * @author <a href="mailto:guyue375@outlook.com">Klaus.turbo</a>
 **/
@Getter
@ToString
public class SendMessageAction {
    
    private String nationCode = "86";
    
    private String phoneNum;
    
    private String templateCode;
    
    private String templateParamJson;
    
    private String encryptionResult;
    
    @JSONField(serialize = false)
    private Map<String, String> nationCodeRoutingTable = new HashMap<>();
    
    @JSONField(serialize = false)
    private SmsRouter smsRouter;
    
    @JSONField(serialize = false)
    private HttpClient httpClient;
    
    public HttpResponse<String> sendMessage() throws IOException, InterruptedException {
        
        String smsEndpoint = this.nationCodeRoutingTable.get(this.nationCode);
        if (StringUtils.isBlank(smsEndpoint)) {
            smsEndpoint = this.smsRouter.apply(this.nationCode);
        }
        String smsAddress = smsEndpoint + SmsConstant.DEFAULT_SMS_URI;
        final HttpRequest request = HttpRequest.newBuilder(URI.create(smsAddress))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers.ofString(JSON.toJSONString(this))).build();
        return httpClient.send(request, HttpResponse.BodyHandlers.ofString());
    }
    
    
    public static SendMessageAction.Builder builder() {
        return new SendMessageAction.Builder();
    }
    
    public static class Builder {
        
        private String nationCode = "86";
        
        private String phoneNum;
        
        private String templateCode;
        
        private String templateParamJson;
        
        private Map<String, String> nationCodeRoutingTable;
        
        private String salt;
        
        private SmsRouter smsRouter;
        
        private HttpClient httpClient;
        
        
        public SendMessageAction.Builder nationCode(String nationCode) {
            this.nationCode = nationCode;
            return this;
        }
        
        public SendMessageAction.Builder phoneNum(String phoneNum) {
            this.phoneNum = phoneNum;
            return this;
        }
        
        public SendMessageAction.Builder templateCode(String templateCode) {
            this.templateCode = templateCode;
            return this;
        }
        
        public SendMessageAction.Builder templateParamJson(String templateParamJson) {
            this.templateParamJson = templateParamJson;
            return this;
        }
        
        public SendMessageAction.Builder salt(String salt) {
            this.salt = salt;
            return this;
        }
        
        public SendMessageAction.Builder nationCodeRoutingTable(Map<String, String> nationCodeRoutingTable) {
            this.nationCodeRoutingTable = nationCodeRoutingTable;
            return this;
        }
        
        public SendMessageAction.Builder smsRouter(SmsRouter smsRouter) {
            this.smsRouter = smsRouter;
            return this;
        }
        
        public SendMessageAction.Builder httpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }
        
        /**
         * 简单参数加密.
         * @param key 被加密字段
         * @return
         */
        private static String simpleSign(String key, String salt) {
            return DigestUtils.md5DigestAsHex((key + salt).getBytes(StandardCharsets.UTF_8)).toUpperCase();
        }
        
        private String encrypt(String salt) {
            String key = String.join(" ", StringUtils.isBlank(this.nationCode) ? StringUtils.EMPTY : this.nationCode,
                    this.phoneNum, this.templateCode, this.templateParamJson);
            return simpleSign(key, salt);
        }
        
        private static boolean isValidPhoneNumber(String phoneNumber, String countryCode) {
            try {
                
                Phonenumber.PhoneNumber numberProto = new Phonenumber.PhoneNumber();
                //                PhoneNumberUtil.getInstance().parse(phoneNumber,countryCode);
                numberProto.setCountryCode(Integer.parseInt(countryCode));
                numberProto.setNationalNumber(Long.parseLong(phoneNumber));
                return PhoneNumberUtil.getInstance().isValidNumber(numberProto);
            } catch (Exception e) {
                System.err.println("NumberParseException was thrown: " + e.toString());
            }
            return false;
        }
        
        public SendMessageAction build() {
            Assert.isTrue(StringUtils.isNotBlank(this.nationCode), "Nation code cannot be empty");
            Assert.isTrue(StringUtils.isNotBlank(this.phoneNum), "Mobile phone number cannot be empty");
            Assert.isTrue(StringUtils.isNotBlank(this.templateCode), "Alibaba Cloud SMS template cannot be empty");
            Assert.isTrue(StringUtils.isNotBlank(this.templateParamJson),
                    "Alibaba Cloud SMS template parameters cannot be empty");
            Assert.isTrue(StringUtils.isNotBlank(this.salt), "The encryption salt cannot be empty");
            SendMessageAction action = new SendMessageAction();
            action.nationCode = this.nationCode;
            action.phoneNum = this.phoneNum;
            //            Assert.isTrue(isValidPhoneNumber(this.phoneNum, this.nationCode), "This nation code does not support this type of mobile phone number");
            action.templateCode = this.templateCode;
            action.templateParamJson = this.templateParamJson;
            if (Objects.nonNull(this.nationCodeRoutingTable) && !this.nationCodeRoutingTable.isEmpty()) {
                action.nationCodeRoutingTable.putAll(this.nationCodeRoutingTable);
            }
            action.encryptionResult = encrypt(this.salt);
            
            if (Objects.isNull(this.smsRouter)) {
                action.smsRouter = new DefaultSmsRouter();
            } else {
                action.smsRouter = this.smsRouter;
            }
            
            if (Objects.isNull(this.httpClient)) {
                action.httpClient = HttpClient.newBuilder().build();
            } else {
                action.httpClient = this.httpClient;
            }
            return action;
        }
    }
    
    public static void main(String[] args) {
        final SendMessageAction build = SendMessageAction.builder().httpClient(HttpClient.newBuilder().build())
                .nationCode("86").templateCode("SMS_480925178")
                .templateParamJson(JSON.toJSONString(Map.of("code", "123456"))).phoneNum("14803872510")
                .salt("5be1e35042144923b7fb78a9b3cb54ea").build();
    }
}
