package com.ovopark.privilege.aop;

import com.ovopark.privilege.common.exception.CommonException;
import com.ovopark.privilege.common.response.ExceptionEnum;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.common.support.StorageInfo;
import com.ovopark.privilege.common.utils.IpUtils;
import com.ovopark.privilege.common.utils.TokenUtils;
import com.ovopark.privilege.model.entity.Users;
import com.ovopark.privilege.service.UsersService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.assertj.core.util.Lists;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Locale;
import java.util.Objects;

/**
 * @ClassName RequestAspect
 * @Description //
 * @Author yuyinghua@ovopark.com
 * @Date 2019/6/27 20:00
 * @Version 1.0
 **/
@Component
@Aspect
@Order(1)
@Slf4j
public class RequestAspect {

    private static final String GET = "GET";
    private static final String PUT = "PUT";
    private static final String POST = "POST";
    private static final String DELETE = "DELETE";
    private static final String TICKET = "ticket";
    private static final String OVOPARK_TOKEN = "authenticator";
    private static final String NEW_OVOPARK_TOKEN = "Ovo-Authorization";

    @Value("${server.servlet.context-path}")
    private String contextPath;

    @Resource
    private UsersService usersService;

    @Pointcut("execution(* com.ovopark.privilege.controller.*Controller.*(..))")
    public void allControllerRequestCut() {
    }


    @Around("allControllerRequestCut()")
    public Object allControllerRequestBeforePoint(ProceedingJoinPoint pjp) throws Throwable {
        LocalDateTime now = LocalDateTime.now();
        LocalStorageUtils.remove();
        long startTime = System.currentTimeMillis();
        HttpServletRequest request = ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getRequest();
        String remoteIp = IpUtils.getRemoteIp(request);
        StorageInfo storageInfo = LocalStorageUtils.getStorageInfo();
        storageInfo
                .setRemoteIp(remoteIp);
        String platform = getPlatform(request);
        LocalStorageUtils.getStorageInfo().setUserPlatform(platform);
        checkToken(request);
        Object result = null;
        result = pjp.proceed();
        return result;
    }

    private void checkToken(HttpServletRequest request) {
        String requestURI = request.getRequestURI().replace(contextPath, "");
        String newOvoToken = request.getHeader(NEW_OVOPARK_TOKEN);
        Integer userId = null;
        Users users = null;
        String ovoToken = null;
        for (String u : noCheckUri){
            if (u.equals(requestURI)){
                return;
            }
        }
        if (!StringUtils.isEmpty(newOvoToken)){
            // 新token方式解析
            ovoToken = newOvoToken;
            userId = TokenUtils.getNewUserId(ovoToken);
            if (userId == null){
                throw new CommonException(ExceptionEnum.INVALID_TOKEN);
            }
        } else {
            ovoToken = request.getHeader(OVOPARK_TOKEN);
            if (!StringUtils.isEmpty(ovoToken)){
                String userIdStr = TokenUtils.getUserId(ovoToken);
                if (StringUtils.isEmpty(userIdStr)){
                    throw new CommonException(ExceptionEnum.INVALID_TOKEN);
                }
                userId = Integer.valueOf(userIdStr);
            }
        }
        users = this.usersService.getUserById(userId);
        if (users!=null){
            StorageInfo mo = LocalStorageUtils.getStorageInfo();
            ovoToken = setStorageInfo(ovoToken, mo);
            mo.setUserId(users.getId());
            mo.setGroupId(users.getGroupId());
            LocalStorageUtils.setStorageInfo(mo);
        }
    }

    private String setStorageInfo(String ovoToken, StorageInfo mo) {
        String[] auths = ovoToken.split(" ");
        if (auths.length > 0 && StringUtils.isNotEmpty(ovoToken)) {
            ovoToken = auths[0];
            mo.setToken(ovoToken);
        }
        if (auths.length > 1) {
            mo.setClient(auths[1]);
        }
        if (auths.length > 2) {
            mo.setVersion(auths[2]);
        }
        if (auths.length > 3) {
            mo.setLang(auths[3]);
            if ("ENGLISH".equalsIgnoreCase(auths[3])) {//英文
                mo.setLocale(Locale.US);
            } else if ("TRADITIONAL_CHINESE".equalsIgnoreCase(auths[3])) {//繁体中文
                mo.setLocale(Locale.TRADITIONAL_CHINESE);
            }
        }
        if (auths.length > 4) {
            mo.setTimeZone(auths[4]);
        }
        mo.setToken(ovoToken);
        return ovoToken;
    }

    /**
     * 不需要权限过滤的路径
     */
    private static final List<String> noCheckUri = Lists.newArrayList(
            "/user/reFreshToken",
            "/user/login",
            "/user/checkUserLogin",
            "/user/getCheckUserLoginVo",
            "/user/loginByOldToken",
            "/user/parseToken"
    );


    public static String getPlatform(HttpServletRequest request) {
        String userAgent = request.getHeader("User-Agent");
//		System.out.println("UA:===========" + userAgent);
        String platform = null;
        if (userAgent != null) {
            // 安卓使用okhttp框架
            if (userAgent.contains("Android") || userAgent.contains("okhttp")) {
                platform = "Android";
            } else if (userAgent.contains("iOS")) {
                platform = "iOS";
            } else {
                // 获取客户端浏览器
                platform = "web";
				/*StringTokenizer st = new StringTokenizer(userAgent, ";");
				st.nextToken();
				platform = st.nextToken();*/
            }
        } else {
            // TODO 没有默认UA，是自定义客户端传来的，需定义各个客户端的标记
            platform = "unknown";
        }
        return platform;
    }
}
