package com.ovopark.privilege.common.utils;

import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.Formatter;
import java.util.Locale;

/**
 * 日期工具类
 *
 * @author wxb
 */
@Component
public class DateUtils {

  public static ZoneId GMT8 = ZoneId.of("GMT+08:00");

  /**
   * 默认格式化格式 yyyy-MM-dd HH:mm:ss
   */
  private static final DateTimeFormatter DEFAULT_DATETIME_FORMAT = Format.LONG_DATE_FORMAT_LINE.formatter;

  private static final DateTimeFormatter SHORT_DATE_FORMAT_LINE = Format.LONG_DATE_FORMAT_LINE.formatter;

  /**
   * <p>
   * String -> date
   * </p>
   * <p>
   * 默认时间格式
   * </p>
   */
  public static LocalDateTime parseTime(String timeStr) {
    return LocalDateTime.parse(timeStr, DEFAULT_DATETIME_FORMAT);
  }

  public static LocalDate parseDate(String timeStr) {
    return LocalDate.parse(timeStr, SHORT_DATE_FORMAT_LINE);
  }

  /**
   * <p>
   * String -> date
   * </p>
   * <p>
   * 自定义格式的转换
   * </p>
   *
   * @param format 时间格式
   */
  public static LocalDateTime parseTime(String timeStr, Format format) {
    return LocalDateTime.parse(timeStr, format.formatter);
  }

  /**
   * <p>
   * time -> date
   * </p>
   * <p>
   * 自定义格式 yyyy-MM-dd HH:mm:ss
   * </p>
   */
  public static String parseTime(LocalDateTime time) {
    return DEFAULT_DATETIME_FORMAT.format(time);
  }

  /**
   * <p>
   * time -> date
   * </p>
   * <p>
   * 自定义时间格式
   * </p>
   */
  public static String parseTime(LocalDateTime time, Format format) {
    return format.formatter.format(time);
  }

  /**
   * <p>
   * date -> string
   * </p>
   * <p>
   * 自定义时间格式
   * </p>
   */
  public static String parseTime(Date time, Format format) {
    LocalDateTime localDateTime = dateToLocalDateTime(time);
    return format.formatter.format(localDateTime);
  }

  /**
   * <p>
   * string -> date
   * </p>
   * <p>
   * 自定义时间格式
   * </p>
   */
  public static Date strToDate(String timeStr, Format format) {
    LocalDateTime dateTime = LocalDateTime.from(format.formatter.parse(timeStr));
    Date date = ldtToDate(dateTime);
    return date;
  }

  /**
   * <p>
   * 获取当前时间 字符串
   * </p>
   * <p>
   * 自定义格式
   * </p>
   */
  public static String getCurrentTime(Format format) {
    return format.formatter.format(LocalDateTime.now());
  }

  /**
   * <p>
   * 获取当前时间
   * </p>
   * <p>
   * 默认格式
   * </p>
   */
  public static String getCurrentTime() {
    return DEFAULT_DATETIME_FORMAT.format(LocalDateTime.now());
  }

  /**
   * DATE -> LOCALDATETIME
   */
  public static LocalDateTime dateToLocalDateTime(Date date) {
    Instant instant = date.toInstant();
    ZoneId zoneId = ZoneId.systemDefault();
    LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, zoneId);
    return localDateTime;
  }

  /**
   * LOCALTIMEDATE -> DATE
   */
  public static Date ldtToDate(LocalDateTime localDateTime) {
    ZoneId zoneId = ZoneId.systemDefault();
    return Date.from(localDateTime.atZone(zoneId).toInstant());
  }

  /**
   * 计算两个日期相差的秒数
   */
  public static Long betweenSeconds(LocalDateTime start, LocalDateTime end) {
    Duration between = Duration.between(start, end);
    return between.getSeconds();
  }

  /**
   * 当前日期的毫秒数
   *
   * @return
   */
  public static long getNowInMillis() {
    return LocalDateTime.now().toInstant(ZoneOffset.of("+8")).toEpochMilli();
  }

  /**
   * 当前日期的秒数
   *
   * @return
   */
  public static long getNowInSecond() {
    return LocalDateTime.now().toEpochSecond(ZoneOffset.of("+8"));
  }

  /**
   * 获取指定日期的秒数
   * @return
   */
  public static long getTimeInSecond(String date) {
    return parseTime(date).toEpochSecond(ZoneOffset.of("+8"));
  }

  /**
   * 获取指定日期的毫秒数
   * @return
   */
  public static long getTimeInMillis(String date) {
    return parseTime(date).toInstant(ZoneOffset.of("+8")).toEpochMilli();
  }

  public static String formatDateTime(long time) {
    Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis(time);
    Formatter ft=new Formatter(Locale.CHINA);
    return ft.format("%1$tY-%1$tm-%1$td %1$tT", cal).toString();
  }

  /**
   * 获取当天的前后日期
   * @param date
   * @param num
   * @return
   */
  public static Date addDay(Date date,int num){
    Calendar cal = Calendar.getInstance();
    cal.setTime(date);
    cal.add(Calendar.DATE, num);
    return cal.getTime();
  }

  /**
   * 获取当天结束时间
   * @param date
   * @return
   */
  public static Date getEndTime(Date date) {
    Calendar dateEnd = Calendar.getInstance();
    dateEnd.setTime(date);
    dateEnd.set(Calendar.HOUR_OF_DAY, 23);
    dateEnd.set(Calendar.MINUTE, 59);
    dateEnd.set(Calendar.SECOND, 59);
    return dateEnd.getTime();
  }

    public enum Format {

    /**
     * 短时间格式化 只有年月日
     */
    SHORT_DATE_FORMAT_LINE("yyyy-MM-dd"),
    SHORT_DATE_FORMAT_SLASH("yyyy/MM/dd"),
    SHORT_DATE_FORMAT_DOUBLE_SLASH("yyyy\\MM\\dd"),
    SHORT_DATE_FORMAT_NONE("yyyyMMdd"),
    SHORT_DATE_FORMAT_NONEDAY("yyyyMM"),
    SHORT_DATE_FORMAT_LINE_NONEDAY("yyyy-MM"),

    /**
     * 长时间格式化 年月日时分秒
     */
    LONG_DATE_FORMAT_LINE("yyyy-MM-dd HH:mm:ss"),
    LONG_DATA_FORMAT_LINE_NOSS("yyyy-MM-dd HH:mm"),
    LONG_DATE_FORMAT_LINE_ZERO("yyyy-MM-dd 00:00:00"),
    LONG_DATE_FORMAT_SLASH("yyyy/MM/dd HH:mm:ss"),
    LONG_DATA_FORMAT_SLASH_NOSS("yyyy/MM/dd HH:mm"),
    LONG_DATE_FORMAT_DOUBLE_SLASH("yyyy\\MM\\dd HH:mm:ss"),
    LONG_DATE_FORMAT_NONE("yyyyMMdd HH:mm:ss"),
    LONG_DATE_FORMAT_NONE_CHAR("yyyyMMddHHmmss"),
    LONG_DATA_FORMAT_MIN_NONE_CHAR("yyyyMMddHHmm"),
    LONG_DATE_FORMAT_LINE_HH_MM_SS("yyyy_MM_dd_HH_mm_ss"),

    /**
     * 长时间格式 带毫秒
     */
    LONG_DATE_FORMAT_WITH_MILSEC_LINE("yyyy-MM-dd HH:mm:ss.SSS"),
    LONG_DATE_FORMAT_WITH_MILSEC_SLASH("yyyy/MM/dd HH:mm:ss.SSS"),
    LONG_DATE_FORMAT_WITH_MILSEC_DOUBLE_SLASH("yyyy/MM/dd HH:mm:ss.SSS"),
    LONG_DATE_FORMAT_WITH_MILSEC_NONE("yyyyMMdd HH:mm:ss.SSS"),
    LONG_DATE_FORMAT_WITH_MILSEC_NONE_CHAR("yyyyMMddHHmmssSSS"),

    /**
     * 短时间格式 仅时分
     */
    SHORT_DATE_FORMAT_WITH_HOURMIN("HH:mm");

    private transient DateTimeFormatter formatter;

    Format(String pattern) {
      formatter = DateTimeFormatter.ofPattern(pattern);
    }

  }

  public enum TimeType {
    YEAR,
    MONTH,
    DAY,
    HOUR,
    MIN,
    SECOND,
    MILL
  }

}
