package com.ovopark.privilege.controller;

import com.ovopark.privilege.common.response.BaseResult;
import com.ovopark.privilege.common.response.ExceptionEnum;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.common.utils.CollectionUtil;
import com.ovopark.privilege.model.mo.InsterUserPrivilegesMo;
import com.ovopark.privilege.model.mo.SaveUserPrivilegesMo;
import com.ovopark.privilege.model.mo.UpdateRolePrivilegeMo;
import com.ovopark.privilege.model.pojo.PrivilegesPojo;
import com.ovopark.privilege.model.vo.*;
import com.ovopark.privilege.pojo.UserPrivilegesPojo;
import com.ovopark.privilege.pojo.UsersPojo;
import com.ovopark.privilege.service.PrivilegesService;
import com.ovopark.privilege.service.ProjectsService;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 权限信息表 前端控制器
 * </p>
 *
 * @author wxb
 * @since 2020-09-01
 */
@RestController
@RequestMapping("/privilege")
public class PrivilegesController {

    @Resource
    private PrivilegesService privilegesService;

    @Resource
    private ProjectsService projectsService;

    /**
     * @description: 获取菜单-首页
     * @param: [type]1web2app
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>>
     * @author wxb
     * @date: 2020/9/17 15:00
     */
    @GetMapping("/getMenus")
    public BaseResult<List<MenusPrivilegesVo>> getMenus() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取用户groupId
        Integer groupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getMenus(userId, groupId));
    }

    /**
     * @description: 获取权限-首页
     * @param: [type]1web2app
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>>
     * @author wxb
     * @date: 2020/9/17 15:00
     */
    @GetMapping("/getMenusPrivileges")
    public BaseResult<List<PrivilegesPojo>> getMenusPrivileges() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取用户groupId
        Integer groupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getWebPrivileges(userId, groupId));
    }

    /**
     * @description: 获取用户订阅产品
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>>
     * @author wxb
     * @date: 2020/10/29 10:30
     */
    @GetMapping("/getUserRecentProduct")
    public BaseResult<List<UserRecentProductVo>> getUserRecentProduct() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getUserRecentProduct(userId));
    }

    /** 
     * @description: 保存订阅的产品
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.UserRecentProductVo>>
     * @author wxb
     * @date: 2020/10/29 10:52
     */ 
    @PostMapping("/saveUserRecentProduct")
    public BaseResult<Boolean> saveUserRecentProduct(@RequestBody Map<String, Object> parameter) {
        String productIds = (String) parameter.get("productIds");
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (productIds == null){
            return BaseResult.paramError();
        }
        Boolean success = privilegesService.saveUserRecentProduct(userId, productIds);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 保存企业权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult
     * @author wxb
     * @date: 2020/9/17 15:18
     */
    @PostMapping("/saveEnterprisePrivilege")
    public BaseResult<Boolean> saveEnterprisePrivilege (@RequestBody Map<String, Object> parameter){
        // 企业id
        Integer enterpriseId = (Integer) parameter.get("enterpriseId");
        // 权限
        String privilegeIds = (String) parameter.get("privilegeIds");
        Boolean success = privilegesService.saveEnterprisePrivilege(enterpriseId, privilegeIds);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 同步保存新企业权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/11/10 17:23
     */ 
    @PostMapping("/synSaveEnterprisePrivilege")
    public BaseResult<Boolean> synSaveEnterprisePrivilege (@RequestBody Map<String, Object> parameter){
        // 企业id
        Integer enterpriseId = (Integer) parameter.get("enterpriseId");
        // 权限
        String privilegeIds = (String) parameter.get("privilegeIds");
        Boolean success = privilegesService.synSaveEnterprisePrivilege(enterpriseId, privilegeIds);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 获取企业的开发菜单
     * @param: [enterpriseId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>>
     * @author wxb
     * @date: 2020/11/6 10:13
     */ 
    @GetMapping("/getEnterpriseOpenMenus")
    public BaseResult<List<MenusPrivilegesVo>> getEnterpriseOpenMenus (Integer enterpriseId){
        return BaseResult.success(privilegesService.getEnterpriseOpenMenus(enterpriseId));
    }

    /**
     * @description: 获取开放菜单
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.MenusVo>>
     * @author wxb
     * @date: 2020/9/17 15:18
     */
    @GetMapping("/getOpenMenus")
        public BaseResult<List<MenusPrivilegesVo>> getOpenMenus(){
        return BaseResult.success(privilegesService.getOpenMenus());
    }

    /**
     * @description: 获取方案列表
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.entity.Projects>>
     * @author wxb
     * @date: 2020/10/13 19:08
     */
    @GetMapping("/getPlans")
    public BaseResult<List<ProjectVo>> getPlans(){
        return BaseResult.success(projectsService.getPlans());
    }

    /**
     * @description: 获取方案对应的权限列表
     * @param: [planId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>>
     * @author wxb
     * @date: 2020/10/13 19:47
     */
    @GetMapping("/getPlanPrivileges")
    public BaseResult<List<MenusPrivilegesVo>> getPlanPrivileges(@RequestParam(defaultValue = "1",required = false) Integer planId){
        return BaseResult.success(privilegesService.getPlanPrivileges(planId));
    }

    /** 
     * @description: 获取菜单和权限列表-用户管理
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>>
     * @author wxb
     * @date: 2020/9/18 15:02
     */ 
    @GetMapping("/getMenuPrivileges")
    public BaseResult<List<MenusPrivilegesVo>> getMenuPrivileges(Integer enterpriseId){
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getMenuPrivileges(enterpriseId));
    }

    /** 
     * @description: 获取角色权限
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>>
     * @author wxb
     * @date: 2020/10/12 15:11
     */ 
    @GetMapping("/getRolePrivilege")
    public BaseResult<RolePrivilegeVo> getRolePrivilege(Integer userId, Integer groupId){
        if (userId == null || groupId == null){
            return BaseResult.paramError();
        }
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (currGroupId != null){
            groupId = currGroupId;
        }
        return BaseResult.success(privilegesService.getRolePrivilege(userId, groupId));
    }

    /**
     * @description: 更新 用户角色权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<com.ovopark.privilege.server.model.vo.RolePrivilegeVo>
     * @author wxb
     * @date: 2020/10/23 15:02
     */
    @PostMapping("/updateRolePrivilege")
    public BaseResult<Boolean> updateRolePrivilege(@RequestBody UpdateRolePrivilegeMo parameter){
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (parameter.getUser() == null){
            return BaseResult.paramError();
        }
        Integer userId = parameter.getUser().getUserId();
        Integer groupId = parameter.getUser().getGroupId();
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (groupId == null){
            BaseResult.paramError();
        }
        Boolean success = privilegesService.updateRolePrivilege(userId, groupId,parameter.getRoleList(),parameter.getRoleDepStr(),parameter.getPrivilegeIds());
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 同步更新 用户角色权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/11/10 19:55
     */
    @PostMapping("/synUpdateRolePrivilege")
    public BaseResult<Boolean> synUpdateRolePrivilege(@RequestBody UpdateRolePrivilegeMo parameter){
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (parameter.getUser() == null){
            return BaseResult.paramError();
        }
        Integer userId = parameter.getUser().getUserId();
        Integer groupId = parameter.getUser().getGroupId();
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (groupId == null){
            BaseResult.paramError();
        }
        Boolean success = privilegesService.synUpdateRolePrivilege(userId, groupId,parameter.getPrivilegeIds());
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 获取模块-APP
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>>
     * @author wxb
     * @date: 2020/10/26 11:16
     */ 
    @GetMapping("/getModules")
    public BaseResult<ModulesVo> getModules() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取用户groupId
        Integer groupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getModules(userId, groupId));
    }

    /**
     * @description: 获取权限列表-APP
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>>
     * @author wxb
     * @date: 2020/10/27 17:06
     */
    @GetMapping("/getPrivileges")
    public BaseResult<List<PrivilegesVo>> getPrivileges() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取用户groupId
        Integer groupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getPrivileges(userId, groupId));
    }

    /**
     * @description: 首页-获取用户常用模块
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>>
     * @author wxb
     * @date: 2020/11/3 23:40
     */
    @GetMapping("/getUserRecentModules")
    public BaseResult<List<PrivilegesVo>> getUserRecentModules() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取用户企业id
        Integer groupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        return BaseResult.success(privilegesService.getUserRecentModules(userId, groupId));
    }

    /**
     * @description: 首页-保存用户常用模块-APP
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>>
     * @author wxb
     * @date: 2020/11/4 2:05
     */
    @PostMapping("/saveUserRecentModules")
    public BaseResult<Boolean> saveUserRecentModules(@RequestBody Map<String, Object> parameter) {
        String moduleIds = (String) parameter.get("moduleIds");
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (moduleIds == null){
            return BaseResult.paramError();
        }
        Boolean success = privilegesService.saveUserRecentModules(userId, moduleIds);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 管理-获取用户订阅模块
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>>
     * @author wxb
     * @date: 2020/11/3 23:40
     */
    @GetMapping("/getUserSubscribeModules")
    public BaseResult<List<PrivilegesPojo>> getUserSubscribeModules() {
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        // 获取用户groupId
        Integer groupId = LocalStorageUtils.getStorageInfo().getGroupId();
        return BaseResult.success(privilegesService.getUserSubscribeModules(userId,groupId));
    }

    /**
     * @description: 管理-保存用户订阅模块-APP
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>>
     * @author wxb
     * @date: 2020/11/4 2:05
     */
    @PostMapping("/saveUserSubscribeModules")
    public BaseResult<Boolean> saveUserSubscribeModules(@RequestBody Map<String, Object> parameter) {
        String moduleIds = (String) parameter.get("moduleIds");
        // 获取用户ID
        Integer userId = LocalStorageUtils.getStorageInfo().getUserId();
        if (userId == null){
            BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (moduleIds == null){
            return BaseResult.paramError();
        }
        Boolean success = privilegesService.saveUserSubscribeModules(userId, moduleIds);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 用户是否有某个权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/11/13 14:10
     */ 
    @PostMapping("/isUserPrivilegeByUserIdAndPrivilegeId")
    public BaseResult<Boolean> isUserPrivilegeByUserIdAndPrivilegeId(@RequestBody Map<String, Object> parameter) {
        // 用户id
        Integer userId = (Integer) parameter.get("userId");
        // 权限id
        Integer privilegeId = (Integer) parameter.get("privilegeId");
        if (userId == null || privilegeId == null){
            BaseResult.paramError();
        }
        Boolean success = privilegesService.isUserPrivilegeByUserIdAndPrivilegeId(userId, privilegeId);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 用户是否有某个权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/11/13 14:19
     */ 
    @PostMapping("/isUserPrivilegeByUserIdAndPrivilegeName")
    public BaseResult<Boolean> isUserPrivilegeByUserIdAndPrivilegeName(@RequestBody Map<String, Object> parameter) {
        // 用户id
        Integer userId = (Integer) parameter.get("userId");
        // 权限名
        String privilegeName = (String) parameter.get("privilegeName");
        if (userId == null || privilegeName == null){
            BaseResult.paramError();
        }
        Boolean success = privilegesService.isUserPrivilegeByUserIdAndPrivilegeName(userId, privilegeName);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 获取用户的指定权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/14 14:06
     */ 
    @PostMapping("/getPrivilegeByUserIdAndPrivilegeId")
    public BaseResult<PrivilegesPojo> getPrivilegeByUserIdAndPrivilegeId(@RequestBody Map<String, Object> parameter) {
        // 用户id
        Integer userId = (Integer) parameter.get("userId");
        // 权限id
        Integer privilegeId = (Integer) parameter.get("privilegeId");
        if (userId == null || privilegeId == null){
            BaseResult.paramError();
        }
        return BaseResult.success(privilegesService.getPrivilegeByUserIdAndPrivilegeId(userId, privilegeId));
    }

    /** 
     * @description: 获取用户的权限菜单
     * @param: [user]
     * @return: com.ovopark.privilege.common.response.BaseResult<java.util.List<com.ovopark.privilege.model.pojo.PrivilegesPojo>>
     * @author wxb
     * @date: 2020/11/14 14:28
     */ 
    @PostMapping("/getUserPrivileges")
    public BaseResult<List<PrivilegesVo>> getUserPrivileges(UsersPojo user) {
        // 用户id
        Integer userId = user.getId();
        // 企业id
        Integer groupId = user.getGroupId();
        if (userId == null || groupId == null){
            BaseResult.paramError();
        }
        return BaseResult.success(privilegesService.getUserPrivileges(user));
    }

    /** 
     * @description: 新增用户权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/14 14:12
     */ 
    @PostMapping("/insterUserPrivileges")
    public BaseResult<Boolean> insterUserPrivileges(@RequestBody Map<String, Object> parameter) {
        // 用户权限对象
        UserPrivilegesPojo userPrivilege = (UserPrivilegesPojo) parameter.get("userPrivilege");
        if (userPrivilege == null ){
            BaseResult.paramError();
        }
        Boolean success = privilegesService.insterUserPrivileges(userPrivilege);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 批量新增用户权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/14 14:40
     */ 
    @PostMapping("/batchInsterUserPrivileges")
    public BaseResult<Boolean> batchInsterUserPrivileges(@RequestBody InsterUserPrivilegesMo parameter) {
        // 用户权限对象
        List<UserPrivilegesPojo> userPrivileges = parameter.getUserPrivileges();
        if (CollectionUtil.isEmpty(userPrivileges)){
            BaseResult.paramError();
        }
        Boolean success = privilegesService.batchInsterUserPrivileges(userPrivileges);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 保存用户的权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/17 19:19
     */
    @PostMapping("/saveUserPrivileges")
    public BaseResult saveUserPrivileges(@RequestBody SaveUserPrivilegesMo parameter) {
        // 用户id
        Integer userId = parameter.getUserId();
        // 权限id集合
        List<Integer> privilegeIds = parameter.getPrivilegeIds();
        // 组织架构id
        Integer enterpriseId = parameter.getEnterpriseId();

        Boolean underEnterprise = parameter.getUnderEnterprise();
        privilegesService.saveUserPrivileges(userId, privilegeIds, enterpriseId, underEnterprise);
        return BaseResult.success();
    }

    /**
     * @description: 删除用户的所有权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/17 19:20
     */
    @PostMapping("/deleteUserPrivilegesByUserId")
    public BaseResult<Boolean> deleteUserPrivilegesByUserId(@RequestBody Map<String, Object> parameter) {
        // 用户id
        Integer userId = (Integer) parameter.get("userId");
        if (userId == null){
            return BaseResult.paramError();
        }
        Boolean success = privilegesService.deleteUserPrivilegesByUserId(userId);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 删除用户的指定权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/17 19:20
     */
    @PostMapping("/deleteUserPrivilegesByUserIdAndPrivilegeId")
    public BaseResult<Boolean> deleteUserPrivilegesByUserIdAndPrivilegeId(@RequestBody Map<String, Object> parameter) {
        // 用户id
        Integer userId = (Integer) parameter.get("userId");
        // 权限id
        Integer privilegeId = (Integer) parameter.get("privilegeId");
        if (userId == null || privilegeId == null){
            return BaseResult.paramError();
        }
        Boolean success = privilegesService.deleteUserPrivilegesByUserIdAndPrivilegeId(userId, privilegeId);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 更新用户权限
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/17 19:46
     */ 
    @PostMapping("/updateUserPrivileges")
    public BaseResult<Boolean> updateUserPrivileges(@RequestBody Map<String, Object> parameter) {
        // 用户id
        UserPrivilegesPojo userPrivilege = (UserPrivilegesPojo) parameter.get("userPrivilege");
        if (userPrivilege == null){
            return BaseResult.paramError();
        }
        Boolean success = privilegesService.updateUserPrivileges(userPrivilege);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 根据企业id获取权限列表
     * @param: [parameter]
     * @return: com.ovopark.privilege.common.response.BaseResult<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/18 9:35
     */
    @PostMapping("/getPrivilegesByGroupId")
    public BaseResult<List<PrivilegesPojo>> getPrivilegesByGroupId(@RequestBody Map<String, Object> parameter) {
        // 企业id
        Integer groupId = (Integer) parameter.get("groupId");
        if (groupId == null){
            return BaseResult.paramError();
        }
        return BaseResult.success(privilegesService.getPrivilegesByGroupId(groupId));
    }

}

