package com.ovopark.privilege.controller;

import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.ovopark.privilege.common.response.BaseResult;
import com.ovopark.privilege.common.response.ExceptionEnum;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.model.vo.CheckUserLoginVo;
import com.ovopark.privilege.model.vo.LoginUser;
import com.ovopark.privilege.model.vo.MobileLoginUser;
import com.ovopark.privilege.model.vo.TokenValue;
import com.ovopark.privilege.model.mo.LoginMo;
import com.ovopark.privilege.model.mo.MobileLoginMo;
import com.ovopark.privilege.service.LoginService;
import com.ovopark.privilege.service.UserRoleService;
import com.ovopark.privilege.service.UsersService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;

/**
 * <p>
 * 用户登录 前端控制器
 * </p>
 *
 * @author wxb
 * @since 2020-09-10
 */
@Slf4j
@RestController
@RequestMapping("/user")
public class UserController {

    @Resource
    private LoginService loginService;

    @Resource
    private UsersService usersService;

    @Resource
    private UserRoleService userRoleService;

    /**
     * @description: 登录
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<com.ovopark.privilege.server.model.vo.LoginUser>
     * @author wxb
     * @date: 2020/9/17 15:17
     */
    @PostMapping("/login")
    public BaseResult<LoginUser> login(@RequestBody LoginMo parameter){
        if (parameter.getLoginType() == null){
            return BaseResult.paramError();
        }
        LoginUser data = loginService.login(parameter.getUserName(),parameter.getPassword(),parameter.getGroupId(),parameter.getLoginType(),parameter.getPhone(),parameter.getMsgCheckCode());
        if (data == null){
            return BaseResult.error();
        }
        return BaseResult.success(data);
    }

    /** 
     * @description: 移动端登录
     * @param: [parameter]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<com.ovopark.privilege.server.model.vo.MobileLoginUser>
     * @author wxb
     * @date: 2020/11/5 16:47
     */ 
    @PostMapping("/mobileLogin")
    public BaseResult<MobileLoginUser> mobileLogin(@RequestBody MobileLoginMo parameter){
        if (parameter.getLoginType() == null){
            return BaseResult.paramError();
        }
        MobileLoginUser data = loginService.mobileLogin(parameter.getUserName(),parameter.getPassword(),parameter.getGroupId(),parameter.getLoginType(),parameter.getPhone(),parameter.getMsgCheckCode());
        if (data == null){
            return BaseResult.error();
        }
        return BaseResult.success(data);
    }

    /**
     * @description: 刷新用户token
     * @param: [token] 用户token
     * @return: com.ovopark.privilege.server.common.response.BaseResult<com.ovopark.privilege.server.model.vo.TokenValue>
     * @author wxb
     * @date: 2020/9/17 15:17
     */
    @GetMapping("reFreshToken")
    public BaseResult<TokenValue> reFreshToken(String token){
        if (StringUtils.isBlank(token)){
            BaseResult.paramError();
        }
        return BaseResult.success(loginService.reFreshToken(token));
    }

    /** 
     * @description: 退出登录
     * @param: []
     * @return: com.ovopark.privilege.server.common.response.BaseResult
     * @author wxb
     * @date: 2020/9/27 17:24
     */ 
    @GetMapping("logout")
    public BaseResult logout(){
        // 获取用户token
        String token = LocalStorageUtils.getStorageInfo().getToken();
        log.info("用户退出，token为：" + token);
        if (loginService.logout(token)){
            return BaseResult.success();
        }
        return BaseResult.error();
    }

    /**
     * @description: 校验用户
     * @param: [userId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult
     * @author wxb
     * @date: 2020/9/28 15:23
     */
    @GetMapping("checkUserLogin")
    public BaseResult<Boolean> checkUserLogin(Integer userId){
        if (userId == null){
            return BaseResult.paramError();
        }
        return BaseResult.success(loginService.checkUserLogin(userId));
    }

    /**
     * 获取用户信息
     * @param userId
     * @return
     */
    @GetMapping("getCheckUserLoginVo")
    public BaseResult<CheckUserLoginVo> checkUserLoginVo(Integer userId){
        if (userId == null){
            return BaseResult.paramError();
        }
        return BaseResult.success(loginService.getCheckUserLoginVo(userId));
    }

    /**
     * @description: 解析token 获取过期时间
     * @param: [token]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<com.ovopark.privilege.server.model.vo.CheckUserLoginVo>
     * @author wxb
     * @date: 2020/10/13 14:21
     */
    @GetMapping("parseToken")
    public BaseResult<TokenValue> parseToken(String token){
        // 获取用户token
        return BaseResult.success(loginService.getTokenValue(token));
    }

    /**
     * 老token 登录
     * @param oldToken
     * @return
     */
    @GetMapping("loginByOldToken")
    public BaseResult<TokenValue> loginByOldToken(String oldToken){
        return BaseResult.success(loginService.getUserTokenValueByOldToken(oldToken));
    }

    /** 
     * @description: 删除用户
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult
     * @author wxb
     * @date: 2020/10/28 16:56
     */ 
    @GetMapping("deleteUser")
    public BaseResult<Boolean> deleteUser(Integer userId, Integer groupId){
        // 获取用户token
        String token = LocalStorageUtils.getStorageInfo().getToken();
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (userId == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = usersService.deleteUser(userId, groupId,token);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 冻结用户
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/10/28 17:25
     */
    @GetMapping("userFreeze")
    public BaseResult<Boolean> userFreeze(Integer userId, Integer groupId){
        // 获取用户token
        String token = LocalStorageUtils.getStorageInfo().getToken();
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (userId == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = usersService.userFreeze(userId, groupId,token);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /** 
     * @description: 解冻用户
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/10/28 17:26
     */ 
    @GetMapping("userUnfreeze")
    public BaseResult<Boolean> userUnfreeze(Integer userId, Integer groupId){
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (userId == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = usersService.userUnfreeze(userId, groupId);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 删除用户
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult
     * @author wxb
     * @date: 2020/10/28 16:56
     */
    @GetMapping("deleteBatchUser")
    public BaseResult<Boolean> deleteBatchUser(String userIds, Integer groupId){
        // 获取用户token
        String token = LocalStorageUtils.getStorageInfo().getToken();
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (userIds == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = usersService.deleteBatchUser(userIds, groupId,token);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 冻结用户
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.server.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/10/28 17:25
     */
    @GetMapping("userBatchFreeze")
    public BaseResult<Boolean> userBatchFreeze(String userIds, Integer groupId){
        // 获取用户token
        String token = LocalStorageUtils.getStorageInfo().getToken();
        // 获取当前登录用户ID
        Integer currUserId = LocalStorageUtils.getStorageInfo().getUserId();
        // 获取当前用户groupId
        Integer currGroupId = LocalStorageUtils.getStorageInfo().getGroupId();
        if (currUserId == null){
            return BaseResult.error(ExceptionEnum.INVALID_TOKEN);
        }
        if (currGroupId != null){
            groupId = currGroupId;
        }
        if (userIds == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = usersService.userBatchFreeze(userIds, groupId,token);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 是否是超级管理员
     * @param: [userIds, groupId]
     * @return: com.ovopark.privilege.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/11/17 19:02
     */
    @PostMapping("isSuperAdmin")
    public BaseResult<Boolean> isSuperAdmin(Integer userId, Integer groupId){
        if (userId == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = userRoleService.isSuperAdmin(userId, groupId);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }

    /**
     * @description: 是否是系统管理员
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.common.response.BaseResult<java.lang.Boolean>
     * @author wxb
     * @date: 2020/11/17 19:04
     */
    @PostMapping("isSystemAdmin")
    public BaseResult<Boolean> isSystemAdmin(Integer userId, Integer groupId){
        if (userId == null || groupId == null){
            BaseResult.paramError();
        }
        Boolean success = userRoleService.isSystemAdmin(userId, groupId);
        if (!success) {
            return BaseResult.error();
        }
        return BaseResult.success();
    }
}
