package com.ovopark.privilege.service.impl;

import com.ovopark.privilege.common.constant.Constant;
import com.ovopark.privilege.common.constant.ResultConstant;
import com.ovopark.privilege.common.constant.SignUsersConstant;
import com.ovopark.privilege.common.exception.CommonException;
import com.ovopark.privilege.common.response.ExceptionEnum;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.common.utils.*;
import com.ovopark.privilege.model.entity.SignUsers;
import com.ovopark.privilege.model.entity.Users;
import com.ovopark.privilege.model.vo.CheckUserLoginVo;
import com.ovopark.privilege.model.vo.LoginUser;
import com.ovopark.privilege.model.vo.MobileLoginUser;
import com.ovopark.privilege.model.vo.TokenValue;
import com.ovopark.privilege.service.LoginService;
import com.ovopark.privilege.service.SignUsersService;
import com.ovopark.privilege.service.UsersService;
import com.ovopark.shopweb.service.ImService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * <p>
 * 用户登录 服务实现类
 * </p>
 *
 * @author wxb
 * @since 2020-09-10
 */
@Slf4j
@Service
public class LoginServiceImpl implements LoginService {

    @Resource
    private UsersService usersService;

    @Resource
    private SignUsersService signUsersService;

    @Resource
    private MessageSource messageSource;

    @Resource
    private ImService imService;

    /**
     * @description: 用户登录
     * @param: [userName, password, groupId, loginType]
     * @return: com.ovopark.privilege.server.model.vo.LoginUser
     * @author wxb
     * @date: 2020/9/17 15:10
     */
    @Override
    public LoginUser login(String userName, String password, Integer groupId, Integer loginType,String phone,String msgCheckCode){
        log.info("用户:" + userName + ",密码：" + password + "登录");
        LoginUser responMsg = new LoginUser();
        Locale locale = Locale.SIMPLIFIED_CHINESE;
        LocalDateTime start = LocalDateTime.now();
        /*********************** 验证用户登录用户名及密码(调用数据库) *****************************/
        try {
        // 查找数据库中是否有此用户
        Users user = null;
        SignUsers signUser = null;
        // 判断是否是企业用户
        if (groupId != null){
            // 企业用户，根据用户名和企业id查询用户
            user = usersService.getUserByUserNameAndGroupId(userName, groupId);
            if (user == null) {
                signUser = signUsersService.getSignUserByUserNameAndGroupId(userName, groupId);
            }
        }else {
            // 非企业用户，根据用户名查询用户
            List<Users> users = usersService.getUserByUserName(userName);
            for (Users usr : users) {
                // TODO 来伊份特殊处理
                // if (usr.getGroup_id() == null || usr.getGroup_id() != 83)
                // {
                user = usr;
                break;
                // }
            }
            if (user == null) {
                signUser = signUsersService.getSignUserByUserName(userName);
            }
        }
        // 用户不存在通过邮箱查找
        if (user == null && signUser == null) {
            user = usersService.getUserByUserEmail(userName);
            if (user == null) {
                signUser = signUsersService.getSignUserByEmail(userName);
            }
        }
        // 用户不存在通过手机号查找
        if (user == null && signUser == null) {
            user = usersService.getUserByUserPhone(userName);
            if (user == null) {
                signUser = signUsersService.getSignUserByPhone(userName);
            }
        }
        LocalDateTime end = LocalDateTime.now();
        log.info("查询用户耗时－－－－－－－－－－－－－:" + DateUtils.betweenSeconds(start,end) + "秒");
        responMsg.setVersion(Constant.map.get("version"));
        // 用户不存在
        if (user == null && signUser == null) {
            responMsg.setResult(ResultConstant.FALSE);
            responMsg.setErrordescription(messageSource.getMessage("login.usernameIsNotExsits", null, locale));
            return responMsg;
        } else {
            // 0正常1被冻结
            if (user != null && user.getIsFrozen() == 1) {
                responMsg.setResult(ResultConstant.FALSE);
                responMsg.setErrordescription(messageSource.getMessage("login.userFrozen", null, locale));
                return responMsg;
            }
            // 密码不正确
            if ((user != null && !(user.getPassword().equals(password))) || (signUser != null && !(signUser.getPassword().equals(password)))) {
                responMsg.setResult(ResultConstant.FALSE);
                responMsg.setErrordescription(messageSource.getMessage("login.pwdIsNotRight", null, locale));
                return responMsg;
            }
            // 激活码不正确
            if (user != null && StringUtils.isNotEmpty(user.getActivateCode()) && !"1".equals(user.getActivateCode())) {
                responMsg.setResult(ResultConstant.FALSE);
                responMsg.setErrordescription(messageSource.getMessage("login.userNameCanNotUse", null, locale));
                return responMsg;
            }
            LocalDate today = LocalDate.now();
            // 用户过期判断，过期七天以内都算正常，允许登陆
            if (user != null && user.getValidateDate() != null && user.getValidateDate().toLocalDate().isBefore(today.plusDays(7))) {
                responMsg.setResult(ResultConstant.FALSE);
                responMsg.setErrordescription(messageSource.getMessage("login.userNameisdelay", null, locale));
                return responMsg;
            }
        }
        if (user == null) {
            if (signUser.getState() != null) {
                // 正在审核中
                if (SignUsersConstant.UNCHECK.equals(signUser.getState())) {
                    responMsg.setResult(ResultConstant.FALSE);
                    responMsg.setErrordescription(messageSource.getMessage("login.signUserUncheck", null, locale));
                    return responMsg;
                    // 审核不通过
                } else if (SignUsersConstant.UNPASS.equals(signUser.getState())) {
                    // 将审核用户id返回
                    responMsg.setSignUsersId(signUser.getId().toString());
                    responMsg.setResult(ResultConstant.FALSE);
                    responMsg.setErrordescription(messageSource.getMessage("login.signUserUnpass", null, locale));
                    return responMsg;
                }
            }
        }
        if (user != null){
            // 是否需要手机验证码登录
            if (user.getNeedSms() != null && user.getNeedSms().equals(1)) {
                if (phone != null) {
                    phone = phone.trim();
                    phone= new String(AesUtil.decrypt(phone, "kedacom.com"));
                }
                if (msgCheckCode != null) {
                    msgCheckCode = msgCheckCode.trim();
                }
                // 手机号或验证码为空
                if (StringUtils.isEmpty(phone) || StringUtils.isEmpty(msgCheckCode)) {
                    // 配置了多因子登录无法用户名密码直接登录
                    responMsg.setResult(ResultConstant.FALSE);
                    responMsg.setErrordescription(messageSource.getMessage("login.needSmsLogin", null, locale));
                    return responMsg;
                }
                String msgCode = SessionManager.getSession(phone);
                if (!msgCheckCode.equals(msgCode)) {
                    // 验证码错误
                    responMsg.setResult(ResultConstant.FALSE);
                    responMsg.setErrordescription(messageSource.getMessage("login.PhoneOrmsgCheckCodeError", null, locale));
                    return responMsg;
                }
            }
            // 生成token
            TokenValue token = TokenUtils.getToken(loginType,user.getId());
            if (user.getLocalelan() != null && !"".equals(user.getLocalelan())) {
                if (user.getLocalelan().equals("zh")) {
                    LocalStorageUtils.getStorageInfo().setLocale(Locale.SIMPLIFIED_CHINESE);
                } else if (user.getLocalelan().equals("en")) {
                    LocalStorageUtils.getStorageInfo().setLocale(Locale.US);
                }
            } else {
                LocalStorageUtils.getStorageInfo().setLocale(Locale.SIMPLIFIED_CHINESE);
            }
            log.info("登录成功");
            // 登录成功
            responMsg.setResult(ResultConstant.TRUE);
            // token
            responMsg.setToken(token.getToken());
            // token过期时间戳
            responMsg.setTokenExpirationTimestamp(token.getTokenExpirationTimestamp());
            // token刷新过期时间戳
            responMsg.setRefreshExpirationTimestamp(token.getRefreshExpirationTimestamp());
            responMsg.setRegistFlag(user.getIsRegisterUser());
            responMsg.setGroupId(groupId);
            return responMsg;
        }
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        return null;
    }

    /**
     * @description: 移动端用户登录
     * @param: [userName, password, groupId, loginType, phone, msgCheckCode]
     * @return: com.ovopark.privilege.server.model.vo.MobileLoginUser
     * @author wxb
     * @date: 2020/11/5 14:21
     */
    @Override
    public MobileLoginUser mobileLogin(String userName, String password, Integer groupId, Integer loginType,String phone,String msgCheckCode) {
        log.info("用户:" + userName + ",密码：" + password + "登录");
        MobileLoginUser responMsg = new MobileLoginUser();
        Locale locale = Locale.SIMPLIFIED_CHINESE;
        LocalDateTime start = LocalDateTime.now();
        /*********************** 验证用户登录用户名及密码(调用数据库) *****************************/
        try {
            // 查找数据库中是否有此用户
            Users user = null;
            SignUsers signUser = null;
            // 判断是否是企业用户
            if (groupId != null){
                // 企业用户，根据用户名和企业id查询用户
                user = usersService.getUserByUserNameAndGroupId(userName, groupId);
                if (user == null) {
                    signUser = signUsersService.getSignUserByUserNameAndGroupId(userName, groupId);
                }
            }else {
                // 非企业用户，根据用户名查询用户
                List<Users> users = usersService.getUserByUserName(userName);
                for (Users usr : users) {
                    // TODO 来伊份特殊处理
                    // if (usr.getGroup_id() == null || usr.getGroup_id() != 83)
                    // {
                    user = usr;
                    break;
                    // }
                }
                if (user == null) {
                    signUser = signUsersService.getSignUserByUserName(userName);
                }
            }
            imService.insertNewMessage(groupId, user.getId());
            // 用户不存在通过邮箱查找
            if (user == null && signUser == null) {
                user = usersService.getUserByUserEmail(userName);
                if (user == null) {
                    signUser = signUsersService.getSignUserByEmail(userName);
                }
            }
            // 用户不存在通过手机号查找
            if (user == null && signUser == null) {
                user = usersService.getUserByUserPhone(userName);
                if (user == null) {
                    signUser = signUsersService.getSignUserByPhone(userName);
                }
            }
            LocalDateTime end = LocalDateTime.now();
            log.info("查询用户耗时－－－－－－－－－－－－－:" + DateUtils.betweenSeconds(start,end) + "秒");
            responMsg.setVersion(Constant.map.get("version"));
            // 用户不存在
            if (user == null && signUser == null) {
                responMsg.setResult(ResultConstant.LOGIN_FAILED);
                responMsg.setErrordescription(messageSource.getMessage("login.usernameIsNotExsits", null, locale));
                return responMsg;
            } else {
                // 验证密码
                password = Md5Encoder.toMd5(password);
                // 0正常1被冻结
                if (user != null && user.getIsFrozen() == 1) {
                    responMsg.setResult(ResultConstant.RESULT_USER_FROZEN);
                    responMsg.setErrordescription(messageSource.getMessage("login.userFrozen", null, locale));
                    return responMsg;
                }
                // 密码不正确
                if ((user != null && !(user.getPassword().equals(password))) || (signUser != null && !(signUser.getPassword().equals(password)))) {
                    responMsg.setResult(ResultConstant.LOGIN_FAILED);
                    responMsg.setErrordescription(messageSource.getMessage("login.pwdIsNotRight", null, locale));
                    return responMsg;
                }
                // 激活码不正确
                if (user != null && StringUtils.isNotEmpty(user.getActivateCode()) && !"1".equals(user.getActivateCode())) {
                    responMsg.setResult(ResultConstant.ACTIVATECODE_ERROR);
                    responMsg.setErrordescription(messageSource.getMessage("login.userNameCanNotUse", null, locale));
                    return responMsg;
                }
                LocalDate today = LocalDate.now();
                // 用户过期判断，过期七天以内都算正常，允许登陆
                if (user != null && user.getValidateDate() != null && user.getValidateDate().toLocalDate().isBefore(today.plusDays(7))) {
                    responMsg.setResult(ResultConstant.RESULT_USER_EXPIRED);
                    responMsg.setErrordescription(messageSource.getMessage("login.userNameisdelay", null, locale));
                    return responMsg;
                }
            }
            if (user == null) {
                if (signUser.getState() != null) {
                    // 正在审核中
                    if (SignUsersConstant.UNCHECK.equals(signUser.getState())) {
                        responMsg.setResult(ResultConstant.USER_UNCHECK);
                        responMsg.setErrordescription(messageSource.getMessage("login.signUserUncheck", null, locale));
                        return responMsg;
                        // 审核不通过
                    } else if (SignUsersConstant.UNPASS.equals(signUser.getState())) {
                        // 将审核用户id返回
                        responMsg.setSignUsersId(signUser.getId().toString());
                        responMsg.setResult(ResultConstant.USER_UNPASS);
                        responMsg.setErrordescription(messageSource.getMessage("login.signUserUnpass", null, locale));
                        return responMsg;
                    }
                }
            }
            if (user != null){
                // 是否需要手机验证码登录
                if (user.getNeedSms() != null && user.getNeedSms().equals(1)) {
                    if (phone != null) {
                        phone = phone.trim();
                        phone= new String(AesUtil.decrypt(phone, "kedacom.com"));
                    }
                    if (msgCheckCode != null) {
                        msgCheckCode = msgCheckCode.trim();
                    }
                    // 手机号或验证码为空
                    if (StringUtils.isEmpty(phone) || StringUtils.isEmpty(msgCheckCode)) {
                        // 配置了多因子登录无法用户名密码直接登录
                        responMsg.setResult(ResultConstant.RESULT_NEED_SMS_LOGIN);
                        responMsg.setErrordescription(messageSource.getMessage("login.needSmsLogin", null, locale));
                        return responMsg;
                    }
                    String msgCode = SessionManager.getSession(phone);
                    if (!msgCheckCode.equals(msgCode)) {
                        // 验证码错误
                        responMsg.setResult(ResultConstant.VALIDATE_CODE_ERROR);
                        responMsg.setErrordescription(messageSource.getMessage("login.PhoneOrmsgCheckCodeError", null, locale));
                        return responMsg;
                    }
                }
                // 生成token
                TokenValue token = TokenUtils.getToken(loginType,user.getId());
                if (user.getLocalelan() != null && !"".equals(user.getLocalelan())) {
                    if (user.getLocalelan().equals("zh")) {
                        LocalStorageUtils.getStorageInfo().setLocale(Locale.SIMPLIFIED_CHINESE);
                    } else if (user.getLocalelan().equals("en")) {
                        LocalStorageUtils.getStorageInfo().setLocale(Locale.US);
                    }
                } else {
                    LocalStorageUtils.getStorageInfo().setLocale(Locale.SIMPLIFIED_CHINESE);
                }
                log.info("登录成功");
                // 登录成功
                responMsg.setResult(ResultConstant.TRUE);
                // TODO:根据用户表的最后登陆时间，判断用户是否首次登陆，首次登陆，则新增IM消息到messages表中。
                Date loginTime = user.getLoginTime();
                // 首次登陆
                if (loginTime == null) {
                    imService.insertNewMessage(groupId, user.getId());
                }
                // token
                responMsg.setToken(token.getToken());
                // token过期时间戳
                responMsg.setTokenExpirationTimestamp(token.getTokenExpirationTimestamp());
                // token刷新过期时间戳
                responMsg.setRefreshExpirationTimestamp(token.getRefreshExpirationTimestamp());
                responMsg.setRegistFlag(user.getIsRegisterUser());
                responMsg.setGroupId(groupId);
                return responMsg;
            }
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        return null;
    }

    /**
     * @description: 刷新用户token
     * @param: [token]
     * @return: com.ovopark.privilege.server.model.vo.TokenValue
     * @author wxb
     * @date: 2020/9/17 15:10
     */
    @Override
    public TokenValue reFreshToken(String token) {
        TokenValue tokenValue = new TokenValue();
        // 刷新token
        tokenValue = TokenUtils.reFreshToken(token);
        return tokenValue;
    }

    /** 
     * @description: 退出
     * @param: []
     * @return: void
     * @author wxb
     * @date: 2020/9/27 17:10
     */ 
    @Override
    public boolean logout(String token) {
        return TokenUtils.logoutByToken(token);
    }

    /**
     * @description: 校验用户
     * @param: [userId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/9/28 15:26
     */
    @Override
    public Boolean checkUserLogin(Integer userId) {
        // 获取用户
        CheckUserLoginVo checkUserLoginVo = this.getCheckUserLoginVo(userId);
        return checkUserLoginVo.getCanLoginFlag();
    }

    /**
     * 判断用户是否可以登录 并返回一些用户字段
     *
     * @param userId 用户id
     * @return CheckUserLoginVo
     */
    @Override
    public CheckUserLoginVo getCheckUserLoginVo(Integer userId) {
        CheckUserLoginVo result = new CheckUserLoginVo();
        result.setCanLoginFlag(false);
        Users users = usersService.getUserById(userId);
        if (users != null){
            // 0正常 1删除  0正常1被冻结
            // 用户过期判断，过期七天以内都算正常，允许登陆
            if (users.getIsDel() == 0 && users.getIsFrozen() == 0 && (users.getValidateDate()==null || users.getValidateDate().toLocalDate().isAfter(LocalDate.now().plusDays(7)))){
                result.setCanLoginFlag(true);
                result.setUserId(users.getId());
                result.setGroupId(users.getGroupId());
                result.setUserName(users.getUserName());
            }
        }
        return result;
    }

    /** 
     * @description: 校验token 获取过期时间
     * @param: []
     * @return: com.ovopark.privilege.server.model.vo.CheckTokenVo
     * @author wxb
     * @date: 2020/10/13 14:25
     */ 
    @Override
    public TokenValue getTokenValue(String token) {
        // 解析token
        return TokenUtils.parseToken(token);
    }

    /**
     * 老token换新token
     * @param oldToken 新token
     * @return TokenValue
     */
    @Override
    public TokenValue getUserTokenValueByOldToken(String oldToken){
        if (StringUtils.isEmpty(oldToken)){
            throw new CommonException(ExceptionEnum.PARAM_ERROR);
        }
        String userIdStr = TokenUtils.getUserId(oldToken);
        if (StringUtils.isNotEmpty(userIdStr)){
            return TokenUtils.getToken(1,Integer.parseInt(userIdStr));
        }else {
            throw new CommonException(ExceptionEnum.PARAM_ERROR);
        }
    }


    public static void main(String[] args){
        Users users = new Users();
        users.setIsDel(0);
        users.setIsFrozen(0);
        if (users.getIsDel() == 0 && users.getIsFrozen() == 0 && (users.getValidateDate()==null || users.getValidateDate().isAfter(LocalDateTime.now()))){
            System.out.println(users);
        }
    }
}
