package com.ovopark.privilege.service.impl;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.common.constant.ApplyToDeptConstant;
import com.ovopark.privilege.common.constant.PrivilegeTypeConstant;
import com.ovopark.privilege.common.constant.UserConstant;
import com.ovopark.privilege.common.exception.CommonException;
import com.ovopark.privilege.common.response.ExceptionEnum;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.common.utils.*;
import com.ovopark.privilege.config.MyConfig;
import com.ovopark.privilege.mapper.PrivilegesMapper;
import com.ovopark.privilege.model.entity.*;
import com.ovopark.privilege.model.mo.RoleIdAndDepIdsMo;
import com.ovopark.privilege.model.pojo.PrivilegesPojo;
import com.ovopark.privilege.model.pojo.ProductPojo;
import com.ovopark.privilege.model.pojo.RolesPojo;
import com.ovopark.privilege.model.vo.*;
import com.ovopark.privilege.pojo.UserPrivilegesPojo;
import com.ovopark.privilege.pojo.UsersPojo;
import com.ovopark.privilege.service.*;
import com.ovopark.shopweb.model.ModuleIcon;
import com.ovopark.shopweb.service.ModuleIconService;
import com.ovopark.shopweb.utils.LogUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;

/**
 * <p>
 * 权限表 服务实现类
 * </p>
 *
 * @author wxb
 * @since 2020-09-01
 */
@Slf4j
@Service
public class PrivilegesServiceImpl extends ServiceImpl<PrivilegesMapper, Privileges> implements PrivilegesService {

    @Resource
    private PrivilegesMapper privilegesMapper;

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private MessageSource messageSource;

    @Resource
    private EnterprisePrivilegesService enterprisePrivilegesService;

    @Resource
    private ProductService productService;

    @Resource
    private UsersService usersService;

    @Resource
    private UserPrivilegesService userPrivilegesService;

    @Resource
    private ProjectPrivilegesService projectPrivilegesService;

    @Resource
    private RolesService rolesService;

    @Resource
    private UserDepartmentService userDepartmentService;

    @Resource
    private DepartmentService departmentService;

    @Resource
    private LogsService logsService;

    @Resource
    private UserRecentProductService userRecentProductService;

    @Resource
    private UserRecentModuleService userRecentModuleService;

    @Resource
    private UserSubscribeModuleService userSubscribeModuleService;

    @Resource
    private OldPrivilegesService oldPrivilegesService;

    @Resource
    private ModuleIconService moduleIconService;

    private static final String CLIENT_WEB = "web";

    private static final String TYPE_MSG_RECEIVE = "1000";

    private static final String MESSAGE_CONTENT = "您的系统应用权限已变更，请重新登录获取！";

    /**
     * @description: 获取菜单树
     * @param: [privileges]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/22 20:23
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesTree (List<PrivilegesPojo> privileges){
        List<PrivilegesPojo> privilegesTree = new ArrayList<>();
        for (PrivilegesPojo parent :  privileges){
            if (parent.getParentId() == null){
                privilegesTree.add(this.getChildrenPrivileges(parent, privileges));
            }
        }
        return privilegesTree;
    }

    /**
     * @description: 获取子菜单
     * @param: [parent, privileges]
     * @return: com.ovopark.privilege.server.model.pojo.PrivilegesPojo
     * @author wxb
     * @date: 2020/9/22 20:23
     */
    private PrivilegesPojo getChildrenPrivileges (PrivilegesPojo parent, List<PrivilegesPojo> privileges){
        for (PrivilegesPojo privilege : privileges){
            if (parent.getId().equals(privilege.getParentId())){
                if (parent.getChildren() == null){
                    parent.setChildren(new ArrayList<>());
                }
                parent.getChildren().add(this.getChildrenPrivileges(privilege,privileges));
            }
        }
        return parent;
    }

    /**
     * @description: 设置权限的所属产品
     * @param: [privileges]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusVo>
     * @author wxb
     * @date: 2020/9/29 11:01
     */
    @Override
    public List<MenusPrivilegesVo> setProduct (List<PrivilegesPojo> privileges){
        // 开放菜单
        List<MenusPrivilegesVo> menus= new ArrayList<>();
        // 查询所有的产品
        List<ProductPojo> products = productService.getAllProduct();
        // 遍历产品
        if (CollectionUtils.isNotEmpty(products)){
            for (ProductPojo product : products){
                List<PrivilegesPojo> privilegesPojos = new ArrayList<>();
                if (CollectionUtils.isNotEmpty(privileges)){
                    for (PrivilegesPojo privilege : privileges){
                        if (privilege.getProductId() != null && privilege.getProductId().equals(product.getId())){
                            privilegesPojos.add(privilege);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(privilegesPojos)){
                    MenusPrivilegesVo menu = new MenusPrivilegesVo();
                    menu.setMenus(privilegesPojos);
                    menu.setId(product.getId());
                    menu.setProductName(product.getProductName());
                    menu.setProductDesc(product.getProductDesc());
                    menu.setCreateTime(product.getCreateTime());
                    menus.add(menu);
                }
            }
        }
        return menus;
    }

    /**
     * @description: 根据id获取权限
     * @param: [privilegeId]
     * @return: com.ovopark.privilege.server.model.pojo.PrivilegesPojo
     * @author wxb
     * @date: 2020/10/13 19:57
     */
    @Override
    public PrivilegesPojo getPrivilegesPojoById(Integer privilegeId) {
        Privileges privilege = this.baseMapper.selectById(privilegeId);
        PrivilegesPojo privilegesPojo = new PrivilegesPojo();
        if (privilege != null){
            BeanUtils.copyProperties(privilege,privilegesPojo);
            return privilegesPojo;
        }
        return null;
    }

    /**
     * @description: 获取所有的权限
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:10
     */
    @Override
    public List<PrivilegesPojo> getAll() {
        List<PrivilegesPojo> privilegeList = new ArrayList<>();
        List<Privileges> privileges = new ArrayList<>();
        privileges = this.baseMapper.selectList(new LambdaQueryWrapper<>());
        if (CollectionUtils.isNotEmpty(privileges)){
            for (Privileges privilege : privileges){
                if (privilege != null){
                    PrivilegesPojo privilegesPojo = new PrivilegesPojo();
                    BeanUtils.copyProperties(privilege,privilegesPojo);
                    privilegeList.add(privilegesPojo);
                }
            }
        }
        return privilegeList;
    }

    /**
     * @description: 获取企业的所有权限
     * @param: [groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesByEnterpriseId(Integer enterpriseId) {
        return privilegesMapper.getPrivilegesByEnterpriseId(enterpriseId);
    }

    /**
     * @description: 获取企业菜单 子模块和页面菜单
     * @param: [groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getOpenMenusByEnterpriseId(Integer enterpriseId) {
        return privilegesMapper.getOpenMenusByEnterpriseId(enterpriseId);
    }

    /**
     * @description: 根据用户id和用户所在的企业id获取用户权限列表
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesByUserIdAndGroupId(Integer userId, Integer groupId) {
        return privilegesMapper.getPrivilegesByUserIdAndGroupId(userId, groupId);
    }

    /**
     * @description: 获取用户的菜单 子模块和页面菜单
     * @param: [userId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getOpenMenusByUserId(Integer userId) {
        return privilegesMapper.getOpenMenusByUserId(userId);
    }

    /**
     * @description: 获取用户权限
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesByUser(Integer userId, Integer groupId) {
        List<PrivilegesPojo> privileges = new ArrayList<>();
        if (userId != null){
            // 是否是超级管理员
            if (userRoleService.isSuperAdmin(userId, groupId)) {
                if (groupId == null){
                    // 系统管理员 root账户 获取所有权限
                    privileges = this.getAll();
                }else {
                    // 企业管理员 获取企业的所有权限
                    privileges = this.getPrivilegesByEnterpriseId(groupId);
                }
                // 普通用户
            } else {
                privileges = this.getPrivilegesByUserIdAndGroupId(userId, groupId);
            }
        }
        return privileges;
    }

    /**
     * @description: 获取web首页菜单权限
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>
     * @author wxb
     * @date: 2020/10/29 19:34
     */
    @Override
    public List<MenusPrivilegesVo> getMenus(Integer userId, Integer groupId) {
        // 获取用户权限列表
        List<PrivilegesPojo> privilegesPojos = this.getPrivilegesByUser(userId, groupId);
        if (CollectionUtils.isNotEmpty(privilegesPojos)){
            // 只显示web
            privilegesPojos.removeIf(privilegesPojo -> privilegesPojo.getIsWeb() == null);
        }
        if (CollectionUtils.isNotEmpty(privilegesPojos)){
            // 不显示操作权限
            privilegesPojos.removeIf(privilegesPojo -> privilegesPojo.getType().equals(PrivilegeTypeConstant.PRIVILEGE));
        }
        // 获取子菜单
        privilegesPojos = this.getPrivilegesTree(privilegesPojos);
        // 设置菜单所属产品
        List<MenusPrivilegesVo> menus = this.setProduct(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(menus)) {
            for (MenusPrivilegesVo product : menus) {
                // 产品名称
                product.setProductDesc(messageSource.getMessage("product." + product.getProductName(), null, locale));
                List<PrivilegesPojo> modules = new ArrayList<>();
                // 模块菜单
                modules = product.getMenus();
                for (PrivilegesPojo module : modules) {
                    module.setPrivilegeDesc(messageSource.getMessage("privileges." + module.getPrivilegeName(), null, locale));
                    List<PrivilegesPojo> privileges = new ArrayList<>();
                    // 页面菜单
                    privileges = module.getChildren();
                    if (CollectionUtils.isNotEmpty(privileges)) {
                        for (PrivilegesPojo privilege : privileges) {
                            privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
                        }
                    }
                }
            }
        }
        return menus;
    }

    /**
     * @description: 获取权限 WEB
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/30 16:18
     */
    @Override
    public List<PrivilegesPojo> getWebPrivileges(Integer userId, Integer groupId) {
        // 获取用户权限列表
        List<PrivilegesPojo> privilegesPojos = this.getPrivilegesByUser(userId, groupId);
        if (CollectionUtil.isNotEmpty(privilegesPojos)){
            // 只显示web 不显示操作权限
            privilegesPojos.removeIf(privilegesPojo -> privilegesPojo != null && privilegesPojo.getType() != null && (privilegesPojo.getType().equals(PrivilegeTypeConstant.MENU) || privilegesPojo.getIsWeb() == null));
        }
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        for (PrivilegesPojo privilege : privilegesPojos) {
            privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
        }
        return privilegesPojos;
    }

    /**
     * @description: 获取用户所属权限菜单
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<MenusPrivilegesVo> getMenusPrivileges(Integer userId, Integer groupId) {
        // 获取用户权限列表
        List<PrivilegesPojo> privilegesPojos = this.getPrivilegesByUser(userId, groupId);
        // 获取子菜单
        privilegesPojos = this.getPrivilegesTree(privilegesPojos);
        // 设置菜单所属产品
        List<MenusPrivilegesVo> menus = this.setProduct(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(menus)) {
            for (MenusPrivilegesVo product : menus) {
                // 产品名称
                product.setProductDesc(messageSource.getMessage("product." + product.getProductName(), null, locale));
                List<PrivilegesPojo> modules = new ArrayList<>();
                // 模块菜单
                modules = product.getMenus();
                for (PrivilegesPojo module : modules) {
                    module.setPrivilegeDesc(messageSource.getMessage("privileges." + module.getPrivilegeName(), null, locale));
                    List<PrivilegesPojo> privileges = new ArrayList<>();
                    // 页面菜单
                    privileges = module.getChildren();
                    if (CollectionUtils.isNotEmpty(privileges)) {
                        for (PrivilegesPojo privilege : privileges) {
                            privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
                        }
                    }
                }
            }
        }
        return menus;
    }

    /** 
     * @description: 获取用户订阅的产品
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.UserRecentModuleVo>
     * @author wxb
     * @date: 2020/10/29 10:43
     */ 
    @Override
    public List<UserRecentProductVo> getUserRecentProduct(Integer userId) {
        List<UserRecentProductVo> userRecentProductVoList = new ArrayList<>();
        // 获取用户的订阅产品
        List<UserRecentProduct> userRecentProductList = userRecentProductService.getUserRecentProductList(userId);
        if (CollectionUtils.isNotEmpty(userRecentProductList)){
            for (UserRecentProduct userRecentProduct : userRecentProductList){
                // 用户订阅的产品
                UserRecentProductVo userRecentProductVo = new UserRecentProductVo();
                userRecentProductVo.setProductId(userRecentProduct.getProductId());
                userRecentProductVo.setCreateTime(userRecentProduct.getCreateTime());
                // 用户订阅的产品集
                userRecentProductVoList.add(userRecentProductVo);
            }
            return userRecentProductVoList;
        }
        return null;
    }

    /** 
     * @description: 保存用户订阅的产品
     * @param: [userId, moduleIds]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.UserRecentModuleVo>
     * @author wxb
     * @date: 2020/10/29 10:54
     */ 
    @Override
    public Boolean saveUserRecentProduct(Integer userId, String productIds) {
        try {
            // 获取用户的订阅
            List<UserRecentProduct> userRecentProductList = userRecentProductService.getUserRecentProductList(userId);
            List<Integer> productIdList = new ArrayList<>();
            if (productIds != null){
                productIdList = StringUtils.stringToList(productIds);
                for (int i = 0 ; i < productIdList.size() ; i++){
                    Integer productId = productIdList.get(i);
                    boolean exsists = false;
                    if (CollectionUtil.isNotEmpty(userRecentProductList)){
                        for (UserRecentProduct userRecentProduct : userRecentProductList){
                            if (userRecentProduct.getProductId().equals(productId)){
                                if (userRecentProduct.getIsOrder() != i) {
                                    // 根据产品顺利排序
                                    userRecentProduct.setIsOrder(i);
                                    userRecentProductService.updateUserRecentProduct(userRecentProduct);
                                }
                                exsists = true;
                                break;
                            }
                        }
                    }
                    // 新订阅模块
                    if (!exsists) {
                        UserRecentProduct userRecentProduct = new UserRecentProduct();
                        userRecentProduct.setUserId(userId);
                        userRecentProduct.setProductId(productId);
                        userRecentProduct.setIsOrder(i);
                        userRecentProductService.insterUserRecentProduct(userRecentProduct);
                    }
                }
            }
            // 取消订阅
            for (UserRecentProduct userRecentProduct : userRecentProductList) {
                if (!productIdList.contains(userRecentProduct.getProductId())) {
                    userRecentProductService.deleteUserRecentProduct(userRecentProduct);
                }
            }
            return true;
        } catch (Exception e){
            log.error("配置订阅产品异常", e);
            return false;
        }
    }

    /**
     * @description: 查询子级权限列表
     * @param: [parentId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesByParentId(Integer parentId) {
        List<PrivilegesPojo> privilegeList = new ArrayList<>();
        List<Privileges> privileges = new ArrayList<>();
        LambdaQueryWrapper<Privileges> queryWrapper = new LambdaQueryWrapper<>();
        // 只获取操作权限
        queryWrapper.eq(Privileges::getParentId,parentId).eq(Privileges::getType,PrivilegeTypeConstant.PRIVILEGE);
        privileges = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(privileges)){
            for (Privileges privilege : privileges){
                if (privilege != null){
                    PrivilegesPojo privilegesPojo = new PrivilegesPojo();
                    BeanUtils.copyProperties(privilege,privilegesPojo);
                    privilegeList.add(privilegesPojo);
                }
            }
        }
        return privilegeList;
    }

    /**
     * @description: 保存企业权限
     * @param: [enterpriseId, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public Boolean saveEnterprisePrivilege(Integer enterpriseId, String privilegeIds) {
        // 删除企业的权限
        enterprisePrivilegesService.deleteEnterprisePrivileges(enterpriseId);
        List<EnterprisePrivileges> enterprisePrivileges = new ArrayList<>();
        if (StringUtils.isNotBlank(privilegeIds)){
            // 权限字符串转list
            List<Integer> privilegeIdList = new ArrayList<>();
            privilegeIdList = StringUtils.createList(privilegeIds);
            for (Integer privilegeId : privilegeIdList){
                // 企业权限
                EnterprisePrivileges enterprisePrivilege = new EnterprisePrivileges();
                enterprisePrivilege.setEnterpriseId(enterpriseId);
                enterprisePrivilege.setPrivilegeId(privilegeId);
                enterprisePrivileges.add(enterprisePrivilege);
                // 如果是页面权限 要获取下面的操作权限
                PrivilegesPojo privilegesPojo = this.getPrivilegesPojoById(privilegeId);
                if (privilegesPojo.getType().equals(PrivilegeTypeConstant.MODULES)){
                    // 获取操作权限
                    List<PrivilegesPojo> privileges = this.getPrivilegesByParentId(privilegeId);
                    for (PrivilegesPojo privilege : privileges){
                        // 子级权限
                        EnterprisePrivileges child = new EnterprisePrivileges();
                        child.setPrivilegeId(privilege.getId());
                        child.setEnterpriseId(enterpriseId);
                        enterprisePrivileges.add(child);
                    }
                }
            }
        }
        // 批量保存企业权限
        return enterprisePrivilegesService.batchInsertEnterprisePrivileges(enterprisePrivileges);
    }

    /**
     * @description: 同步保存企业权限
     * @param: [enterpriseId, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/10 17:22
     */
    @Override
    public Boolean synSaveEnterprisePrivilege(Integer enterpriseId, String privilegeIds) {
        List<EnterprisePrivileges> enterprisePrivileges = new ArrayList<>();
        // 权限id列表
        List<Integer> privilegeIdList = StringUtils.stringToList(privilegeIds);
        // 查询新权限列表
        List<EnterprisePrivileges> newEnterprisePrivileges = enterprisePrivilegesService.getEnterprisePrivilegesByEnterpriseId(enterpriseId);
        // 查询权限列表
        List<OldPrivileges> oldPrivilegeList = oldPrivilegesService.getOldPrivilegeListByIds(privilegeIdList);
        if (oldPrivilegeList != null){
            // 删除新权限
            enterprisePrivilegesService.batchDeleteEnterprisePrivileges(newEnterprisePrivileges);
            if (CollectionUtil.isNotEmpty(privilegeIdList)){
                for (OldPrivileges oldPrivilege : oldPrivilegeList){
                    // 根据权限名获取新权限
//                Privileges newPrivileges = this.getPrivilegesByPrivilegeName(oldPrivilege.getPrivilegeName());
//                if (newPrivileges != null){
                    EnterprisePrivileges enterprisePrivilege = new EnterprisePrivileges();
                    enterprisePrivilege.setEnterpriseId(enterpriseId);
                    enterprisePrivilege.setPrivilegeId(oldPrivilege.getId());
                    enterprisePrivileges.add(enterprisePrivilege);
                    // 页面权限
                    List<Privileges> pagePrivileges = this.getPrivilegeListByParentId(oldPrivilege.getId());
                    for (Privileges pagePrivilege : pagePrivileges){
                        EnterprisePrivileges child = new EnterprisePrivileges();
                        child.setPrivilegeId(pagePrivilege.getId());
                        child.setEnterpriseId(enterpriseId);
                        enterprisePrivileges.add(child);
                        // 功能权限
                        List<Privileges> funPrivileges = this.getPrivilegeListByParentId(pagePrivilege.getId());
                        for (Privileges funPrivilege : funPrivileges){
                            EnterprisePrivileges childs = new EnterprisePrivileges();
                            childs.setPrivilegeId(funPrivilege.getId());
                            childs.setEnterpriseId(enterpriseId);
                            enterprisePrivileges.add(childs);
                        }
                    }
//                }
                }
            }
        }
        if (CollectionUtil.isNotEmpty(privilegeIdList)){
            for (OldPrivileges oldPrivilege : oldPrivilegeList){
                // 根据权限名获取新权限
//                Privileges newPrivileges = this.getPrivilegesByPrivilegeName(oldPrivilege.getPrivilegeName());
//                if (newPrivileges != null){
                    EnterprisePrivileges enterprisePrivilege = new EnterprisePrivileges();
                    enterprisePrivilege.setEnterpriseId(enterpriseId);
                    enterprisePrivilege.setPrivilegeId(oldPrivilege.getId());
                    enterprisePrivileges.add(enterprisePrivilege);
                    // 页面权限
                    List<Privileges> pagePrivileges = this.getPrivilegeListByParentId(oldPrivilege.getId());
                    for (Privileges pagePrivilege : pagePrivileges){
                        EnterprisePrivileges child = new EnterprisePrivileges();
                        child.setPrivilegeId(pagePrivilege.getId());
                        child.setEnterpriseId(enterpriseId);
                        enterprisePrivileges.add(child);
                        // 功能权限
                        List<Privileges> funPrivileges = this.getPrivilegeListByParentId(pagePrivilege.getId());
                        for (Privileges funPrivilege : funPrivileges){
                            EnterprisePrivileges childs = new EnterprisePrivileges();
                            childs.setPrivilegeId(funPrivilege.getId());
                            childs.setEnterpriseId(enterpriseId);
                            enterprisePrivileges.add(childs);
                        }
                    }
//                }
            }
        }
        // 批量保存企业权限
        return enterprisePrivilegesService.batchInsertEnterprisePrivileges(enterprisePrivileges);
    }

    /**
     * @description: 根据权限名获取新权限
     * @param: [privilegeName]
     * @return: com.ovopark.privilege.server.model.entity.Privileges
     * @author wxb
     * @date: 2020/11/10 18:39
     */
    @Override
    public Privileges getPrivilegesByPrivilegeName(String privilegeName) {
        if (privilegeName != null){
            LambdaQueryWrapper<Privileges> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(Privileges::getPrivilegeName,privilegeName);
            return this.baseMapper.selectOne(queryWrapper);
        }
        return null;
    }

    /**
     * @description: 获取子权限
     * @param: [parentId]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.Privileges>
     * @author wxb
     * @date: 2020/11/10 18:53
     */
    @Override
    public List<Privileges> getPrivilegeListByParentId(Integer parentId) {
        if (parentId != null){
            LambdaQueryWrapper<Privileges> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(Privileges::getParentId,parentId);
            return this.baseMapper.selectList(queryWrapper);
        }
        return null;
    }

    /**
     * @description: 获取企业的开发菜单
     * @param: [enterpriseId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>
     * @author wxb
     * @date: 2020/11/6 10:00
     */
    @Override
    public List<MenusPrivilegesVo> getEnterpriseOpenMenus(Integer enterpriseId) {
        // 获取企业权限
        List<PrivilegesPojo> privilegesPojos = this.getOpenMenusByEnterpriseId(enterpriseId);
        // 获取菜单列表
        this.getPrivilegesTree(privilegesPojos);
        // 设置菜单所属产品
        List<MenusPrivilegesVo> menus = this.setProduct(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(menus)){
            // 开放菜单遍历
            for (MenusPrivilegesVo product : menus){
                // 产品名称
                product.setProductDesc(messageSource.getMessage("product."+product.getProductName(), null, locale));
                List<PrivilegesPojo> modules = new ArrayList<>();
                // 模块菜单
                modules = product.getMenus();
                if (CollectionUtils.isNotEmpty(modules)){
                    for (PrivilegesPojo module : modules){
                        // 模块名称
                        module.setPrivilegeDesc(messageSource.getMessage("privileges."+module.getPrivilegeName(), null, locale));
                        List<PrivilegesPojo> privileges = new ArrayList<>();
                        // 模块下的页面菜单
                        privileges = module.getChildren();
                        if (CollectionUtils.isNotEmpty(privileges)){
                            for (PrivilegesPojo privilege : privileges){
                                // 页面菜单名称
                                privilege.setPrivilegeDesc(messageSource.getMessage("privileges."+privilege.getPrivilegeName(), null, locale));
                            }
                        }
                    }
                }
            }
        }
        return menus;
    }

    /**
     * @description: 获取所有的菜单 开放菜单
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getAllMenus() {
        List<Privileges> privileges = new ArrayList<>();
        List<PrivilegesPojo> privilegesPojoList = new ArrayList<>();
        LambdaQueryWrapper<Privileges> queryWrapper = new LambdaQueryWrapper<>();
        // 只获取菜单 0子模块1页面权限2操作权限
        queryWrapper.ne(Privileges::getType,PrivilegeTypeConstant.PRIVILEGE).isNull(Privileges::getIsRoot);
        privileges = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(privileges)){
            for (Privileges privilege : privileges){
                if (privilege != null){
                    PrivilegesPojo privilegesPojo = new PrivilegesPojo();
                    BeanUtils.copyProperties(privilege,privilegesPojo);
                    privilegesPojoList.add(privilegesPojo);
                }
            }
        }
        return privilegesPojoList;
    }

    /**
     * @description: 获取开放菜单
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/29 10:15
     */
    @Override
    public List<MenusPrivilegesVo> getOpenMenus() {
        // 开放菜单列表
        List<PrivilegesPojo> privilegesPojos = this.getUserOpenMenus();
        // 设置菜单所属产品
        List<MenusPrivilegesVo> menus = this.setProduct(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(menus)){
            // 开放菜单遍历
            for (MenusPrivilegesVo product : menus){
                // 产品名称
                product.setProductDesc(messageSource.getMessage("product."+product.getProductName(), null, locale));
                List<PrivilegesPojo> modules = new ArrayList<>();
                // 模块菜单
                modules = product.getMenus();
                if (CollectionUtils.isNotEmpty(modules)){
                    for (PrivilegesPojo module : modules){
                        // 模块名称
                        module.setPrivilegeDesc(messageSource.getMessage("privileges."+module.getPrivilegeName(), null, locale));
                        List<PrivilegesPojo> privileges = new ArrayList<>();
                        // 模块下的页面菜单
                        privileges = module.getChildren();
                        if (CollectionUtils.isNotEmpty(privileges)){
                            for (PrivilegesPojo privilege : privileges){
                                // 页面菜单名称
                                privilege.setPrivilegeDesc(messageSource.getMessage("privileges."+privilege.getPrivilegeName(), null, locale));
                            }
                        }
                    }
                }
            }
        }
        return menus;
    }

    /**
     * @description: 获取方案对应的权限列表
     * @param: [planId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>
     * @author wxb
     * @date: 2020/10/13 19:53
     */
    @Override
    public List<MenusPrivilegesVo> getPlanPrivileges(Integer planId) {
        // 方案权限列表
        List<PrivilegesPojo> privilegesPojos = this.getPrivilegesByPlanId(planId);
        // 设置菜单所属产品
        List<MenusPrivilegesVo> menus = this.setProduct(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(menus)){
            // 开放菜单遍历
            for (MenusPrivilegesVo product : menus){
                // 产品名称
                product.setProductDesc(messageSource.getMessage("product."+product.getProductName(), null, locale));
                List<PrivilegesPojo> modules = new ArrayList<>();
                // 模块菜单
                modules = product.getMenus();
                if (CollectionUtils.isNotEmpty(modules)){
                    for (PrivilegesPojo module : modules){
                        // 模块名称
                        module.setPrivilegeDesc(messageSource.getMessage("privileges."+module.getPrivilegeName(), null, locale));
                        List<PrivilegesPojo> privileges = new ArrayList<>();
                        // 模块下的页面菜单
                        privileges = module.getChildren();
                        if (CollectionUtils.isNotEmpty(privileges)){
                            for (PrivilegesPojo privilege : privileges){
                                // 页面菜单名称
                                privilege.setPrivilegeDesc(messageSource.getMessage("privileges."+privilege.getPrivilegeName(), null, locale));
                            }
                        }
                    }
                }
            }
        }
        return menus;
    }

    /**
     * @description: 根据方案id获取权限列表
     * @param: [planId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/15 13:25
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesByPlanId(Integer planId) {
        List<PrivilegesPojo> privilegesPojos = new ArrayList<>();
        List<ProjectPrivileges> projectPrivilegesList = projectPrivilegesService.getPlanPrivileges(planId);
        if (CollectionUtils.isNotEmpty(projectPrivilegesList)){
            for (ProjectPrivileges projectPrivilege : projectPrivilegesList){
                // 获取权限
                PrivilegesPojo privilegesPojo = this.getPrivilegesPojoById(projectPrivilege.getPrivilegeId());
                privilegesPojos.add(privilegesPojo);
            }
        }
        // 获取权限树结构
        return this.getPrivilegesTree(privilegesPojos);
    }

    /**
     * @description: 获取开放菜单
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getUserOpenMenus() {
        // 最终菜单
        List<MenusPrivilegesVo> menus = new ArrayList<>();
        // 子模块和页面菜单
        List<PrivilegesPojo> privileges = new ArrayList<>();
        // 获取所有的菜单
        privileges = this.getAllMenus();
        // 获取菜单列表
        return this.getPrivilegesTree(privileges);
    }

    /**
     * @description: 获取用户有权限的菜单
     * @param: [userId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>
     * @author wxb
     * @date: 2020/9/17 15:11
     */
    @Override
    public List<PrivilegesPojo> getUserOpenMenus(Integer userId) {
        // 最终菜单
        List<MenusPrivilegesVo> menus = new ArrayList<>();
        // 子模块和页面菜单
        List<PrivilegesPojo> privileges = new ArrayList<>();
        if (userId != null){
            privileges = this.getOpenMenusByUserId(userId);
        }
        // 获取菜单
        return this.getPrivilegesTree(privileges);
    }

    /**
     * @description: 获取菜单和权限列表
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>
     * @author wxb
     * @date: 2020/9/18 15:26
     */
    @Override
    public List<MenusPrivilegesVo> getMenuPrivileges(Integer enterpriseId) {
        List<PrivilegesPojo> privilegesPojos = new ArrayList<>();
        privilegesPojos = this.getEnterpriseMenusPrivileges(enterpriseId);
        // 设置菜单所属产品
        List<MenusPrivilegesVo> menus = this.setProduct(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(menus)) {
            for (MenusPrivilegesVo product : menus) {
                // 产品名称
                product.setProductDesc(messageSource.getMessage("product." + product.getProductName(), null, locale));
                List<PrivilegesPojo> modules = new ArrayList<>();
                // 模块菜单
                modules = product.getMenus();
                for (PrivilegesPojo module : modules) {
                    module.setPrivilegeDesc(messageSource.getMessage("privileges." + module.getPrivilegeName(), null, locale));
                    List<PrivilegesPojo> privileges = new ArrayList<>();
                    // 页面菜单
                    privileges = module.getChildren();
                    if (CollectionUtils.isNotEmpty(privileges)) {
                        for (PrivilegesPojo privilege : privileges) {
                            privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
                            List<PrivilegesPojo> childrens = new ArrayList<>();
                            // 页面操作权限
                            childrens = module.getChildren();
                            for (PrivilegesPojo children : childrens) {
                                children.setPrivilegeDesc(messageSource.getMessage("privileges." + children.getPrivilegeName(), null, locale));
                            }
                        }
                    }
                }
            }
        }
        return menus;
    }

    /**
     * @description: 获取用户的菜单和对应权限列表
     * @param: [userId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/12 15:47
     */
    @Override
    public List<PrivilegesPojo> getUserMenusPrivileges(Integer userId) {
        List<PrivilegesPojo> privileges = new ArrayList<>();
        if (userId != null){
            privileges = this.getPrivilegesByUserIdAndGroupId(userId,null);
        }
        return this.getPrivilegesTree(privileges);
    }

    /**
     * @description: 获取企业的菜单和权限
     * @param: [groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/22 20:04
     */
    @Override
    public List<PrivilegesPojo> getEnterpriseMenusPrivileges(Integer groupId) {
        List<PrivilegesPojo> privileges = new ArrayList<>();
        if (groupId == null){
            // 获取所有的菜单
            privileges = this.getAllMenusPrivileges();
        } else {
            // 有权限的菜单
            privileges = this.getMenusPrivilegesByEnterpriseId(groupId);
        }
        return this.getPrivilegesTree(privileges);
    }

    /**
     * @description: 获取所有的菜单和权限
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/22 16:01
     */
    @Override
    public List<PrivilegesPojo> getAllMenusPrivileges() {
        List<Privileges> privileges = new ArrayList<>();
        List<PrivilegesPojo> privilegesPojoList = new ArrayList<>();
        LambdaQueryWrapper<Privileges> queryWrapper = new LambdaQueryWrapper<>();
        privileges = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(privileges)){
            for (Privileges privilege : privileges){
                if (privilege != null){
                    PrivilegesPojo privilegesPojo = new PrivilegesPojo();
                    BeanUtils.copyProperties(privilege,privilegesPojo);
                    privilegesPojoList.add(privilegesPojo);
                }
            }
        }
        return privilegesPojoList;
    }

    /**
     * @description: 获取企业的菜单和权限
     * @param: [groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/9/22 18:36
     */
    @Override
    public List<PrivilegesPojo> getMenusPrivilegesByEnterpriseId(Integer enterpriseId) {
        return privilegesMapper.getMenusPrivilegesByEnterpriseId(enterpriseId);
    }

    /**
     * @description: 获取用户的菜单和权限
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/12 14:43
     */
    @Override
    public List<PrivilegesPojo> getMenuPrivilegesByUser(Integer userId, Integer groupId) {
        List<PrivilegesPojo> privilegesPojos = new ArrayList<>();
        // 判断是否是超管
        if (userRoleService.isSuperAdmin(userId, groupId)){
            if (groupId == null){
                // 系统管理员 root账户 获取所有的菜单
                privilegesPojos = this.getEnterpriseMenusPrivileges(null);
            } else {
                // 企业管理员 获取企业有权限的菜单
                privilegesPojos = this.getEnterpriseMenusPrivileges(groupId);
            }
            // 普通用户
        } else {
            // 获取用户有权限的菜单
            privilegesPojos = this.getUserMenusPrivileges(userId);
        }
        return privilegesPojos;
    }

    /**
     * @description: 保存用户权限
     * @param: [userId, privilegeIds, enterpriseId, underEnterprise]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/9/24 14:50
     */
    @Override
    public void saveUserPrivileges(Integer userId, List<Integer> privilegeIds, Integer enterpriseId, boolean underEnterprise) {
        // 有企业id
        if (enterpriseId != null) {
            // 删除用户的权限
            userPrivilegesService.deleteUserPrivileges(userId, enterpriseId);
        }
        if(CollectionUtils.isNotEmpty(privilegeIds)){
            if (underEnterprise) {
                List<Integer> enterprisePrivilegeIds = enterprisePrivilegesService.getPrivilegeIdsByEnterpriseId(enterpriseId);
                privilegeIds = ListUtils.intersectionWithoutDup(privilegeIds, enterprisePrivilegeIds);
            }
            for(Integer privilegeId : privilegeIds){
                UserPrivileges up = new UserPrivileges();
                up.setPrivilegeId(privilegeId);
                up.setUserId(userId);
                up.setGroupId(enterpriseId);
                userPrivilegesService.saveUserPrivileges(up);
            }
        }
    }

    /**
     * @description: 同步保存用户权限
     * @param: [userId, privilegeIds, enterpriseId, underEnterprise]
     * @return: void
     * @author wxb
     * @date: 2020/11/10 19:59
     */
    @Override
    public void synSaveUserPrivilege(Integer userId, List<Integer> privilegeIds, Integer enterpriseId, boolean underEnterprise) {
        if(CollectionUtils.isNotEmpty(privilegeIds)){
            if (underEnterprise) {
                List<Integer> enterprisePrivilegeIds = enterprisePrivilegesService.getPrivilegeIdsByEnterpriseId(enterpriseId);
                privilegeIds = ListUtils.intersectionWithoutDup(privilegeIds, enterprisePrivilegeIds);
            }
            for(Integer privilegeId : privilegeIds){
                // 获取权限
                OldPrivileges oldPrivilege = oldPrivilegesService.getOldPrivilegesById(privilegeId);
                // 根据权限名获取新新权限
                Privileges newPrivileges = this.getPrivilegesByPrivilegeName(oldPrivilege.getPrivilegeName());
                if (newPrivileges != null){
                    UserPrivileges up = new UserPrivileges();
                    up.setPrivilegeId(newPrivileges.getId());
                    up.setUserId(userId);
                    up.setGroupId(enterpriseId);
                    userPrivilegesService.saveUserPrivileges(up);
                    List<Privileges> privilegesList = this.getPrivilegeListByParentId(newPrivileges.getId());
                    if (CollectionUtil.isNotEmpty(privilegesList)){
                        for (Privileges privilege : privilegesList){
                            if (privilege.getType() != 1){
                                privilegesList.remove(privilege);
                            }else {
                                UserPrivileges userPrivileges = new UserPrivileges();
                                userPrivileges.setPrivilegeId(privilege.getId());
                                userPrivileges.setUserId(userId);
                                userPrivileges.setGroupId(enterpriseId);
                                userPrivilegesService.saveUserPrivileges(userPrivileges);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * @description: 获取角色权限
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/12 15:11
     */
    @Override
    public RolePrivilegeVo getRolePrivilege(Integer userId, Integer groupId) {
        RolePrivilegeVo rolePrivilegeVo = new RolePrivilegeVo();
        // 获取用户
        Users user = usersService.getUserById(userId);
        if (user == null){
            throw new CommonException(ExceptionEnum.PARAM_ERROR);
        }
        // 角色
        List<RolesPojo> roles = rolesService.getRoleListByUserIdAndGroupId(userId,groupId);
        if (CollectionUtils.isNotEmpty(roles)) {
            List<RolesPojo> rolesPojoList = new ArrayList<>();
            // Map<角色ID,Map<门店Id,门店名称>>
            Map<Integer, Map<Integer, String>> roleDepMap = new HashMap<>();
            // 门店角色人员
            List<UserDepartment> userDepartments = userDepartmentService.getUserDepartmentsByUser(userId, groupId);
            if (CollectionUtils.isNotEmpty(userDepartments)) {
                // 门店id
                List<Integer> depIdList = new ArrayList<>();
                for (UserDepartment userDepartment : userDepartments) {
                    depIdList.add(userDepartment.getDepartmentId());
                }
                // Map<门店ID，门店名称>
                Map<Integer, String> depMap = departmentService.getDepMapByIds(depIdList);
                for (UserDepartment userDepartment : userDepartments) {
                    if (roleDepMap.containsKey(userDepartment.getRoleId())) {
                        Map<Integer, String> map = roleDepMap.get(userDepartment.getRoleId());
                        String depName = depMap.get(userDepartment.getDepartmentId());
                        map.put(userDepartment.getDepartmentId(), depName);
                    } else {
                        String depName = depMap.get(userDepartment.getDepartmentId());
                        if (StringUtils.isNotBlank(depName)) {
                            Map<Integer, String> depIdAndName = new HashMap<>();
                            depIdAndName.put(userDepartment.getDepartmentId(), depName);
                            roleDepMap.put(userDepartment.getRoleId(), depIdAndName);
                        }
                    }
                }
            }
            // 国际化
            Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
            for (RolesPojo rolesPojo : roles) {
                if (rolesPojo.getId() <= 2) {
                    rolesPojo.setRoleName((messageSource.getMessage("roles.id." + rolesPojo.getId(), null, locale)));
                }
                Map<Integer, String> map = roleDepMap.get(rolesPojo.getId());
                if (map != null) {
                    List<DepIdAndNameVo> list = new ArrayList<>();
                    for (Map.Entry entry : map.entrySet()) {
                        DepIdAndNameVo depIdAndNameVo = new DepIdAndNameVo();
                        depIdAndNameVo.setDepId(Integer.valueOf(entry.getKey().toString()));
                        depIdAndNameVo.setDepName(entry.getValue().toString());
                        list.add(depIdAndNameVo);
                    }
                    rolesPojo.setDepIdAndNameList(list);
                }
                rolesPojoList.add(rolesPojo);
            }
            rolePrivilegeVo.setRoles(rolesPojoList);
        }
        // 菜单权限
        List<MenusPrivilegesVo> privileges = this.getMenusPrivileges(userId, groupId);
        rolePrivilegeVo.setPrivileges(privileges);
        return rolePrivilegeVo;
    }

    /**
     * @description: 批量获取权限列表
     * @param: [privilegeIdList]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/21 14:16
     */
    @Override
    public List<PrivilegesPojo> batchGetPrivilegeListByIds(List<Integer> privilegeIdList) {
        List<PrivilegesPojo> privilegeList = new ArrayList<>();
        List<Privileges> privileges = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(privilegeIdList)){
            LambdaQueryWrapper<Privileges> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(Privileges::getId,privilegeIdList);
            privileges = this.baseMapper.selectList(queryWrapper);
            if (CollectionUtils.isNotEmpty(privileges)){
                for (Privileges privilege : privileges){
                    if (privilege != null){
                        PrivilegesPojo privilegesPojo = new PrivilegesPojo();
                        BeanUtils.copyProperties(privilege,privilegesPojo);
                        privilegeList.add(privilegesPojo);
                    }
                }
            }
        }
        return privilegeList;
    }

    /**
     * @description: 更新 用户角色权限
     * @param: [userId, groupId, roleList, roleDepStr, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/23 14:54
     */
    @Override
    public Boolean updateRolePrivilege(Integer userId, Integer groupId, List<Integer> roleList, List<RoleIdAndDepIdsMo> roleDepStr, List<Integer> privilegeIds) {
        try {
            // 保存用户角色
            userRoleService.saveUserRoles(roleList, userId, groupId);
            // 设置 用户角色 关联的门店
            if (CollectionUtil.isNotEmpty(roleDepStr)) {
                for (RoleIdAndDepIdsMo temp : roleDepStr) {
                    // 角色id
                    Integer roleId = temp.getRoleId();
                    // 门店id集
                    String depIds = temp.getDepIds();
                    List<Integer> depList = StringUtils.stringToList(depIds);
                    if (CollectionUtils.isNotEmpty(depList)){
                        // 保存用户门店角色关系
                        userDepartmentService.saveUserDepartment(userId, groupId, roleId, depList);
                        // 获取角色
                        RolesPojo rolePojo = rolesService.getRoleById(roleId);
                        // 应用到门店
                        rolePojo.setApplyToDept(ApplyToDeptConstant.APPLY);
                        rolesService.updateRolesById(rolePojo);
                    }
                }
            }
            // 获取用户保存前权限
            List<PrivilegesPojo> beforePrivileges = this.getPrivilegesByUser(userId, groupId);
            // 更新用户权限
            this.saveUserPrivileges(userId, privilegeIds, groupId, true);
            // 更新后的权限
            List<PrivilegesPojo> afterPrivileges = this.getPrivilegesByUser(userId, groupId);
            // 判断是否有更新
            boolean flag1 = beforePrivileges.containsAll(afterPrivileges);
            boolean flag2 = afterPrivileges.containsAll(beforePrivileges);
            if (!flag1 || !flag2){
                // 有更新 下发通知
                Map<String, String> map = new HashMap<>();
                map.put("userId", userId.toString());
                map.put("clients", CLIENT_WEB);
                map.put("message", MESSAGE_CONTENT);
                map.put("messageType", TYPE_MSG_RECEIVE);
                // 推送
                String result = HttpUtils.sendJsonToOtherServerDefault(MyConfig.getWebsocketUrl() + "/websocket/sendMessageToUser", map);
                log.info("message: [{}]sendMessageToUser result:{}", map, result);
            }
            Users user = usersService.getUserById(userId);
            if (user.getIsAssign() == null || user.getIsAssign() ==  UserConstant.ISASSIGN_NO){
                user.setIsAssign(UserConstant.ISASSIGN_YES);
                usersService.updateUserById(user);
            }
            // 填写日志
            Logs logs = new Logs();
            logs.setUserId(user.getId());
            // 方法名/action名
            logs.setOperation("updateRolePrivilege");
            logs.setUserName(user.getUserName());
            logs.setParameters(JSON.toJSONString(roleList));
            logs.setRemoteIp(LogUtils.getRemoteIp(Objects.requireNonNull(SessionUtil.getRequest())));
            logs.setPlatform(LogUtils.getPlatform(Objects.requireNonNull(SessionUtil.getRequest())));
            logs.setCreateTime(LocalDateTime.now());
            logs.setLogLevel(0);
            logs.setDescription("id："+user.getId()+",roleList："+JSON.toJSONString(roleList));
            if(user.getGroupId() != null){
                logs.setEnterpriseId(user.getGroupId());
            }
            logsService.insterLogs(logs);
            return true;
        } catch (Exception e) {
            log.error("更新角色权限异常", e);
            return false;
        }
    }

    /**
     * @description: 同步更新用户权限
     * @param: [userId, groupId, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/10 19:56
     */
    @Override
    public Boolean synUpdateRolePrivilege(Integer userId, Integer groupId, List<Integer> privilegeIds) {
        try {
            // 获取新的权限id
            List<Integer> newPrivilegeIds = this.getNewPrivilegeIds(privilegeIds);
            // 更新用户权限
            this.saveUserPrivileges(userId, newPrivilegeIds, groupId, true);
            return true;
        }catch (Exception e){
            log.error("同步更新角色权限异常", e);
            return false;
        }
    }

    /**
     * @description: 获取新的权限id
     * @param: [oldPrivilegeIds]
     * @return: java.util.List<java.lang.Integer>
     * @author wxb
     * @date: 2020/11/11 15:54
     */
    @Override
    public List<Integer> getNewPrivilegeIds(List<Integer> oldPrivilegeIds) {
        List<Integer> newPrivilegeIds = new ArrayList<>();
        for (Integer privilegeId : oldPrivilegeIds){
            // 获取权限
            OldPrivileges oldPrivilege = oldPrivilegesService.getOldPrivilegesById(privilegeId);
            // 根据权限名获取新权限
            Privileges newPrivilege = this.getPrivilegesByPrivilegeName(oldPrivilege.getPrivilegeName());
            // 如果是模块 直接添加
            if (newPrivilege != null){
                if (newPrivilege.getType().equals(PrivilegeTypeConstant.PRIVILEGE)){
                    // 获取操作权限的页面菜单
                    PrivilegesPojo parentPrivilege = this.getPrivilegesPojoById(newPrivilege.getParentId());
                    newPrivilegeIds.add(parentPrivilege.getId());
                }
                newPrivilegeIds.add(newPrivilege.getId());
            }
        }
        // LIST去重
        Set set = new HashSet<>();
        List privilegeIds = new ArrayList<>();
        for (Iterator iter = newPrivilegeIds.iterator(); iter.hasNext();) {
            Object element = iter.next();
            if (set.add(element))
                privilegeIds.add(element);
        }
        newPrivilegeIds.clear();
        newPrivilegeIds.addAll(privilegeIds);
        return newPrivilegeIds;
    }

    /** 
     * @description: 获取模块
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/26 10:23
     */ 
    @Override
    public ModulesVo getModules(Integer userId, Integer groupId) {
        ModulesVo modulesVo = new ModulesVo();
        // 获取用户的模块
        List<PrivilegesPojo> userPrivileges = this.getPrivilegesByUser(userId, groupId);
        // 配置的模块
        if (CollectionUtils.isNotEmpty(userPrivileges)){
            userPrivileges.removeIf(privilegesPojo -> privilegesPojo.getType() != null && (privilegesPojo.getType().equals(PrivilegeTypeConstant.PRIVILEGE) || privilegesPojo.getIsApp() == null));
        }
        // 获取用户订阅模块
        List<PrivilegesPojo> subscribeModulePrivileges = userSubscribeModuleService.getUserSubscribeModulePrivileges(userId);
        if (CollectionUtil.isNotEmpty(subscribeModulePrivileges)){
            // 排序用户模块
            userPrivileges = this.sortModules(userPrivileges, subscribeModulePrivileges);
        }
        // 设置模块分类
        List<PrivilegesPojo> unconfig = this.setManageClass(userPrivileges);
        // 获取用户订阅模块
        List<PrivilegesPojo> config = this.getUserSubscribeModules(userId, groupId);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        for (PrivilegesPojo category : unconfig) {
            category.setPrivilegeDesc(messageSource.getMessage("product." + category.getPrivilegeName(), null, locale));
            List<PrivilegesPojo> modules = new ArrayList<>();
            modules = category.getChildren();
            if (CollectionUtils.isNotEmpty(modules)) {
                for (PrivilegesPojo privilege : modules) {
                    privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
                }
            }
        }
        for (PrivilegesPojo category : config) {
            category.setPrivilegeDesc(messageSource.getMessage("product." + category.getPrivilegeName(), null, locale));
            List<PrivilegesPojo> modules = new ArrayList<>();
            modules = category.getChildren();
            if (CollectionUtils.isNotEmpty(modules)) {
                for (PrivilegesPojo privilege : modules) {
                    privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
                }
            }
        }
        modulesVo.setConfig(config);
        modulesVo.setUnconfig(unconfig);
        return modulesVo;
    }

    /**
     * @description: 设置管理页分类
     * @param: [privileges]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.MenusPrivilegesVo>
     * @author wxb
     * @date: 2020/11/11 10:15
     */
    private List<PrivilegesPojo> setManageClass (List<PrivilegesPojo> privileges){
        List<PrivilegesPojo> menus= new ArrayList<>();
        // 查询所有的产品
        List<ProductPojo> products = productService.getAllProduct();
        // 遍历产品
        if (CollectionUtils.isNotEmpty(products)){
            for (ProductPojo product : products){
                List<PrivilegesPojo> privilegesPojos = new ArrayList<>();
                if (CollectionUtils.isNotEmpty(privileges)){
                    for (PrivilegesPojo privilege : privileges){
                        if (privilege.getProductId() == null){
                            // 获取父模块
                            PrivilegesPojo parentPrivilege = this.getPrivilegesPojoById(privilege.getParentId());
                            if (parentPrivilege != null && parentPrivilege.getProductId() != null && parentPrivilege.getProductId().equals(product.getId())){
                                // 获取模块下的菜单 属于APP模块
                                privilege.setParentId(product.getId());
                                privilegesPojos.add(privilege);
                            }
                        }else if (privilege.getProductId() != null && privilege.getProductId().equals(product.getId())){
                            // 获取模块下的菜单 属于APP模块
                            privilege.setParentId(product.getId());
                            privilegesPojos.add(privilege);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(privilegesPojos)){
                    PrivilegesPojo privilegesPojo = new PrivilegesPojo();
                    privilegesPojo.setChildren(privilegesPojos);
                    privilegesPojo.setId(product.getId());
                    privilegesPojo.setPrivilegeName(product.getProductName());
                    privilegesPojo.setPrivilegeDesc(product.getProductDesc());
                    privilegesPojo.setCreateTime(product.getCreateTime());
                    menus.add(privilegesPojo);
                }
            }
        }
        return menus;
    }

    /** 
     * @description: 模块排序
     * @param: [privilegesPojos, subscribeModules]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/9 14:46
     */ 
    private List<PrivilegesPojo> sortModules (List<PrivilegesPojo> privilegesPojos, List<PrivilegesPojo> subscribeModules){
        List<PrivilegesPojo> sortPrivileges = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(subscribeModules)){
            for (PrivilegesPojo subscribeModule : subscribeModules){
                for (PrivilegesPojo privilegesPojo : privilegesPojos){
                    if (subscribeModule.getId().equals(privilegesPojo.getId())){
                        // 排序的模块
                        sortPrivileges.add(privilegesPojo);
                    }
                }
            }
        }
        // 未包含的模块 并集
        privilegesPojos.removeAll(sortPrivileges);
        sortPrivileges.addAll(privilegesPojos);
        return sortPrivileges;
    }

    /**
     * @description: 获取权限 APP
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/27 16:59
     */
    @Override
    public List<PrivilegesVo> getPrivileges(Integer userId, Integer groupId) {
        // 获取用户权限列表
        List<PrivilegesPojo> privilegesPojos = this.getPrivilegesByUser(userId, groupId);
        // 操作权限列表
        List<PrivilegesVo> privileges = new ArrayList<>();
        // 只获取操作权限
        if (CollectionUtils.isNotEmpty(privilegesPojos)){
            for (PrivilegesPojo privilegesPojo : privilegesPojos){
                if (PrivilegeTypeConstant.PRIVILEGE.equals(privilegesPojo.getType()) && privilegesPojo.getIsApp() != null && privilegesPojo.getIsApp() == 1){
                    PrivilegesVo privilegesVo = new PrivilegesVo();
                    privilegesVo.setId(privilegesPojo.getId());
                    privilegesVo.setPrivilegeDesc(privilegesPojo.getPrivilegeDesc());
                    privilegesVo.setPrivilegeName(privilegesPojo.getPrivilegeName());
                    privileges.add(privilegesVo);
                }
            }
        }
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(privileges)) {
            for (PrivilegesVo privilege : privileges) {
                privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
            }
        }
        return privileges;
    }

    /** 
     * @description: 首页-获取用户常用模块-APP
     * @param: [userId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>
     * @author wxb
     * @date: 2020/11/3 23:02
     */ 
    @Override
    public List<PrivilegesVo> getUserRecentModules(Integer userId, Integer groupId) {
        List<PrivilegesPojo> privilegesPojos = userRecentModuleService.getUserRecentModulePrivileges(userId);
        List<PrivilegesPojo> recentModules = new ArrayList<>();
        // 未设置常用模块
        if (CollectionUtil.isNotEmpty(privilegesPojos)){
            int size = 8;
            if (privilegesPojos.size() < 8){
                size = privilegesPojos.size();
            }
            // 获取用户的前8个模块
            for (int i = 0 ; i < size ; i++){
                recentModules.add(privilegesPojos.get(i));
            }
        }
        // 填充企业模块图标
        ModuleIcon icons = moduleIconService.selectByGroupId(groupId, 2);
        if (icons != null) {
            List<PrivilegesPojo> iconModules = new ArrayList<>();
            JSONArray array = JSONArray.parseArray(icons.getJsonArray());
            for (int i = 0; i < array.size(); i++) {
                JSONObject obj = array.getJSONObject(i);
                if (recentModules.size() > i) {
                    recentModules.get(i).setPicUrl(obj.getString("url"));
                    iconModules.add(recentModules.get(i));
                } else {
                    PrivilegesPojo iconModule = new PrivilegesPojo();
                    iconModule.setPicUrl(obj.getString("url"));
                    iconModules.add(iconModule);
                }
            }
            recentModules = iconModules;
        }
        List<PrivilegesVo> privileges = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(recentModules)){
            for (PrivilegesPojo recentModule : recentModules){
                PrivilegesVo privilegesVo = new PrivilegesVo();
                privilegesVo.setId(recentModule.getId());
                privilegesVo.setParentId(recentModule.getParentId());
                privilegesVo.setPrivilegeName(recentModule.getPrivilegeName());
                privilegesVo.setPrivilegeDesc(recentModule.getPrivilegeDesc());
                privilegesVo.setPicUrl(recentModule.getPicUrl());
                privileges.add(privilegesVo);
            }
        }
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(privileges)) {
            for (PrivilegesVo module : privileges) {
                if (module.getPrivilegeName() != null){
                    module.setPrivilegeDesc(messageSource.getMessage("privileges." + module.getPrivilegeName(), null, locale));
                }
            }
        }
        return privileges;
    }

    /**
     * @description: 首页-保存用户常用模块-APP
     * @param: [userId, moduleIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/3 23:45
     */
    @Override
    public Boolean saveUserRecentModules(Integer userId, String moduleIds) {
        try {
            // 获取用户的常用模块
            List<UserRecentModule> userRecentModuleList = userRecentModuleService.getUserRecentModules(userId);
            List<Integer> moduleIdList = new ArrayList<>();
            if (moduleIds != null){
                moduleIdList = StringUtils.stringToList(moduleIds);
                for (int i = 0 ; i < moduleIdList.size() ; i++){
                    Integer moduleId = moduleIdList.get(i);
                    boolean exsists = false;
                    if (CollectionUtil.isNotEmpty(userRecentModuleList)){
                        for (UserRecentModule userRecentModule : userRecentModuleList){
                            if (userRecentModule.getModuleId().equals(moduleId)){
                                if (userRecentModule.getIsOrder() != i) {
                                    // 根据模块顺序进行排序
                                    userRecentModule.setIsOrder(i);
                                    userRecentModuleService.updateUserRecentModule(userRecentModule);
                                }
                                exsists = true;
                                break;
                            }
                        }
                    }
                    // 新常用模块
                    if (!exsists) {
                        UserRecentModule userRecentModule = new UserRecentModule();
                        userRecentModule.setUserId(userId);
                        userRecentModule.setModuleId(moduleId);
                        userRecentModule.setIsOrder(i);
                        userRecentModuleService.insterUserRecentModule(userRecentModule);
                    }
                }
            }
            // 取消常用
            for (UserRecentModule userRecentModule : userRecentModuleList) {
                if (!moduleIdList.contains(userRecentModule.getModuleId())) {
                    userRecentModuleService.deleteUserRecentModule(userRecentModule);
                }
            }
            return true;
        } catch (Exception e){
            log.error("配置常用模块异常", e);
            return false;
        }
    }

    /**
     * @description: 管理-获取用户订阅模块-APP
     * @param: [userId]
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.PrivilegesVo>
     * @author wxb
     * @date: 2020/11/7 14:04
     */
    @Override
    public List<PrivilegesPojo> getUserSubscribeModules(Integer userId, Integer groupId) {
        List<PrivilegesPojo> privilegesPojos = new ArrayList<>();
        privilegesPojos = userSubscribeModuleService.getUserSubscribeModulePrivileges(userId);
        // 无订阅模块 返回全部
        if (CollectionUtil.isEmpty(privilegesPojos)){
            // 获取用户的模块
            privilegesPojos = this.getPrivilegesByUser(userId, groupId);
            // 配置的模块
            if (CollectionUtils.isNotEmpty(privilegesPojos)){
                privilegesPojos.removeIf(privilegesPojo -> privilegesPojo.getType() != null && (privilegesPojo.getType().equals(PrivilegeTypeConstant.PRIVILEGE) || privilegesPojo.getIsApp() == null));
            }
        }
        // 设置模块分类
        List<PrivilegesPojo> privileges = this.setManageClass(privilegesPojos);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        for (PrivilegesPojo category : privileges) {
            category.setPrivilegeDesc(messageSource.getMessage("product." + category.getPrivilegeName(), null, locale));
            List<PrivilegesPojo> modules = new ArrayList<>();
            // 管理页模块
            modules = category.getChildren();
            if (CollectionUtils.isNotEmpty(modules)) {
                for (PrivilegesPojo module : modules) {
                    module.setPrivilegeDesc(messageSource.getMessage("privileges." + module.getPrivilegeName(), null, locale));
                }
            }
        }
        return privileges;
    }

    /**
     * @description: 管理-保存用户订阅模块-APP
     * @param: [userId, moduleIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/7 14:05
     */
    @Override
    public Boolean saveUserSubscribeModules(Integer userId, String moduleIds) {
        try {
            // 获取用户的订阅模块
            List<UserSubscribeModule> userSubscribeModuleList = userSubscribeModuleService.getUserSubscribeModules(userId);
            List<Integer> moduleIdList = new ArrayList<>();
            if (moduleIds != null){
                moduleIdList = StringUtils.stringToList(moduleIds);
                for (int i = 0 ; i < moduleIdList.size() ; i++){
                    Integer moduleId = moduleIdList.get(i);
                    boolean exsists = false;
                    if (CollectionUtil.isNotEmpty(userSubscribeModuleList)){
                        for (UserSubscribeModule userSubscribeModule : userSubscribeModuleList){
                            if (userSubscribeModule.getModuleId().equals(moduleId)){
                                if (userSubscribeModule.getIsOrder() != i) {
                                    userSubscribeModule.setIsOrder(i);
                                    userSubscribeModuleService.updateUserSubscribeModule(userSubscribeModule);
                                }
                                exsists = true;
                                break;
                            }
                        }
                    }
                    // 新订阅模块
                    if (!exsists) {
                        UserSubscribeModule userSubscribeModule = new UserSubscribeModule();
                        userSubscribeModule.setUserId(userId);
                        userSubscribeModule.setModuleId(moduleId);
                        userSubscribeModule.setIsOrder(i);
                        userSubscribeModuleService.insterUserSubscribeModule(userSubscribeModule);
                    }
                }
            }
            // 取消订阅
            for (UserSubscribeModule userSubscribeModule : userSubscribeModuleList) {
                if (!moduleIdList.contains(userSubscribeModule.getModuleId())) {
                    userSubscribeModuleService.deleteUserSubscribeModule(userSubscribeModule);
                }
            }
            return true;
        } catch (Exception e){
            log.error("配置订阅模块异常", e);
            return false;
        }
    }

    /**
     * @description: 用户是否有某个权限
     * @param: [userId, privilegeId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/13 13:59
     */
    @Override
    public Boolean isUserPrivilegeByUserIdAndPrivilegeId(Integer userId, Integer privilegeId) {
        // 获取用户的权限
        UserPrivileges userPrivileges = userPrivilegesService.getUserPrivilegesByUserIdAndPrivilegeId(userId, privilegeId);
        return userPrivileges != null;
    }

    /**
     * @description: 用户是否有某个权限
     * @param: [userId, privilegeName]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/13 13:59
     */
    @Override
    public Boolean isUserPrivilegeByUserIdAndPrivilegeName(Integer userId, String privilegeName) {
        // 获取用户的权限
        UserPrivileges userPrivileges = userPrivilegesService.getUserPrivilegesByUserIdAndPrivilegeName(userId, privilegeName);
        return userPrivileges != null;
    }

    /** 
     * @description: 获取用户的指定权限
     * @param: [userId, privilegeId]
     * @return: com.ovopark.privilege.model.pojo.PrivilegesPojo
     * @author wxb
     * @date: 2020/11/14 13:55
     */ 
    @Override
    public PrivilegesPojo getPrivilegeByUserIdAndPrivilegeId(Integer userId, Integer privilegeId) {
        return privilegesMapper.getPrivilegeByUserIdAndPrivilegeId(userId, privilegeId);
    }

    /**
     * @description: 获取用户的权限菜单
     * @param: [user]
     * @return: java.util.List<com.ovopark.privilege.model.vo.PrivilegesVo>
     * @author wxb
     * @date: 2020/11/18 9:35
     */
    @Override
    public List<PrivilegesVo> getUserPrivileges(UsersPojo user) {
        List<PrivilegesPojo> privileges = this.getPrivilegesByUser(user.getId(), user.getGroupId());
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(privileges)) {
            for (PrivilegesPojo privilege : privileges){
                privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
            }
        }
        return this.getPrivileges(user.getId(), user.getGroupId());
    }

    /** 
     * @description: 新增用户权限
     * @param: [userPrivilege]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/14 14:32
     */ 
    @Override
    public Boolean insterUserPrivileges(UserPrivilegesPojo userPrivilege) {
        try {
            UserPrivileges up = new UserPrivileges();
            if (userPrivilege != null){
                BeanUtils.copyProperties(userPrivilege, up);
            }
            userPrivilegesService.insterUserPrivileges(up);
            return true;
        }catch (Exception e){
            log.error("新增用户权限异常", e);
            return false;
        }
    }

    /**
     * @description: 批量添加用户权限
     * @param: [userPrivileges]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/17 19:06
     */
    @Override
    public Boolean batchInsterUserPrivileges(List<UserPrivilegesPojo> userPrivileges) {
        try {
            if (CollectionUtil.isNotEmpty(userPrivileges)){
                for (UserPrivilegesPojo userPrivilegesPojo : userPrivileges){
                    this.insterUserPrivileges(userPrivilegesPojo);
                }
                return true;
            }
        }catch (Exception e){
            log.error("新增用户权限异常", e);
            return false;
        }
        return false;
    }

    /**
     * @description: 删除用户的所有权限
     * @param: [userId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/17 19:21
     */
    @Override
    public Boolean deleteUserPrivilegesByUserId(Integer userId) {
        return userPrivilegesService.deleteUserPrivilegesByUserId(userId);
    }

    /** 
     * @description: 删除用户的指定权限
     * @param: [userId, privilegeId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/17 19:45
     */ 
    @Override
    public Boolean deleteUserPrivilegesByUserIdAndPrivilegeId(Integer userId, Integer privilegeId) {
        return userPrivilegesService.deleteUserPrivilegesByUserIdAndPrivilegeId(userId, privilegeId);
    }

    /** 
     * @description: 更新用户的权限
     * @param: [userPrivilege]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/17 19:46
     */ 
    @Override
    public Boolean updateUserPrivileges(UserPrivilegesPojo userPrivilege) {
        UserPrivileges userPrivileges = new UserPrivileges();
        if (userPrivilege != null){
            BeanUtils.copyProperties(userPrivilege, userPrivileges);
            return userPrivilegesService.updateUserPrivileges(userPrivileges);
        }
        return false;
    }

    /**
     * @description: 根据企业id获取权限列表
     * @param: [groupId]
     * @return: java.util.List<com.ovopark.privilege.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/11/18 9:36
     */
    @Override
    public List<PrivilegesPojo> getPrivilegesByGroupId(Integer groupId) {
        List<PrivilegesPojo> privileges = this.getEnterpriseMenusPrivileges(groupId);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if (CollectionUtils.isNotEmpty(privileges)) {
            for (PrivilegesPojo privilege : privileges){
                privilege.setPrivilegeDesc(messageSource.getMessage("privileges." + privilege.getPrivilegeName(), null, locale));
            }
        }
        return privileges;
    }
}
