package com.ovopark.privilege.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.common.utils.StringUtils;
import com.ovopark.privilege.mapper.ProjectsMapper;
import com.ovopark.privilege.model.entity.ProjectPrivileges;
import com.ovopark.privilege.model.entity.Projects;
import com.ovopark.privilege.model.vo.ProjectVo;
import com.ovopark.privilege.service.ProjectPrivilegesService;
import com.ovopark.privilege.service.ProjectsService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @description: 方案
 * @param:
 * @return:
 * @author wxb
 * @date: 2020/10/13 15:17
 */
@Service
@Slf4j
public class ProjectsServiceImpl extends ServiceImpl<ProjectsMapper, Projects> implements ProjectsService {

    @Resource
    private ProjectPrivilegesService projectPrivilegesService;

    /**
     * @description: 根据id获取方案
     * @param: [id]
     * @return: com.ovopark.privilege.server.model.entity.Projects
     * @author wxb
     * @date: 2020/10/13 15:51
     */
    @Override
    public Projects getProjectById(Integer id) {
        return this.baseMapper.selectById(id);
    }

    /**
     * @description: 新增方案
     * @param: [projects]
     * @return: void
     * @author wxb
     * @date: 2020/10/13 15:38
     */
    @Override
    public Projects insertProject(Projects projects) {
        if (this.baseMapper.insert(projects) > 0){
            return projects;
        }
        return null;
    }

    /**
     * @description: 更新方案
     * @param: [projects]
     * @return: void
     * @author wxb
     * @date: 2020/10/13 15:38
     */
    @Override
    public void updateProject(Projects projects) {
        if (projects.getId() != null){
            this.baseMapper.updateById(projects);
        }
    }

    /**
     * @description: 保存方案
     * @param: [userId, planName, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/20 14:46
     */
    @Override
    public Boolean saveProject(Integer userId, String planName, String privilegeIds) {
        try {
            // 新增方案
            Projects projects = new Projects();
            projects.setName(planName);
            projects.setCreateId(userId);
            projects.setCreateTime(new Date());
            projects = this.insertProject(projects);
            // 新增的方案id
            Integer projectId = projects.getId();
            // 新增方案权限
            List<Integer> privilegeIdList = StringUtils.createList(privilegeIds);
            if (CollectionUtils.isNotEmpty(privilegeIdList)){
                for (Integer privilegeId : privilegeIdList){
                    ProjectPrivileges projectPrivileges = new ProjectPrivileges();
                    projectPrivileges.setPrivilegeId(privilegeId);
                    projectPrivileges.setProjectId(projectId);
                    // 新增
                    projectPrivilegesService.insertProjectPrivileges(projectPrivileges);
                }
            }
            return true;
        }catch (Exception e){
            log.error("保存方案异常");
            return false;
        }
    }

    /**
     * @description: 覆盖方案
     * @param: [userId, planId, planName, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/20 14:46
     */
    @Override
    public Boolean updateProject(Integer userId, Integer planId, String planName, String privilegeIds) {
        try {
            // 更新方案
            Projects projects = this.baseMapper.selectById(planId);
            if (projects != null) {
                projects.setName(planName);
                projects.setCreateId(userId);
                projects.setUpdateTime(new Date());
                this.updateProject(projects);
                // 删除原方案权限
                projectPrivilegesService.deleteProjectPrivilegesByProjectId(planId);
                // 新增方案权限
                List<Integer> privilegeIdList = StringUtils.createList(privilegeIds);
                if (CollectionUtils.isNotEmpty(privilegeIdList)) {
                    for (Integer privilegeId : privilegeIdList) {
                        ProjectPrivileges projectPrivileges = new ProjectPrivileges();
                        projectPrivileges.setPrivilegeId(privilegeId);
                        projectPrivileges.setProjectId(planId);
                        // 新增
                        projectPrivilegesService.insertProjectPrivileges(projectPrivileges);
                    }
                }
            }
            return true;
        }catch (Exception e){
            log.error("覆盖方案异常");
            return false;
        }
    }

    /**
     * @description: 获取方案列表
     * @param: []
     * @return: java.util.List<com.ovopark.privilege.server.model.vo.ProjectVo>
     * @author wxb
     * @date: 2020/10/13 18:59
     */
    @Override
    public List<ProjectVo> getPlans() {
        List<Projects> projectsList = new ArrayList<>();
        List<ProjectVo> projects = new ArrayList<>();
        LambdaQueryWrapper<Projects> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Projects::getIsDelete,0).orderByDesc(Projects::getIsDefault).orderByAsc(Projects::getCreateTime);
        projectsList = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(projectsList)){
            for (Projects project : projectsList){
                if (project != null){
                    ProjectVo projectVo = new ProjectVo();
                    BeanUtils.copyProperties(project,projectVo);
                    projects.add(projectVo);
                }
            }
        }
        return projects;
    }

    /** 
     * @description: 根据id删除方案
     * @param: [planId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/20 15:55
     */ 
    @Override
    public void deletePlanById(Integer planId) {
        this.baseMapper.delete(new LambdaQueryWrapper<Projects>().eq(Projects::getId,planId));
    }

    /**
     * @description: 删除方案
     * @param: [planId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/20 15:58
     */
    @Override
    public Boolean deletePlan(Integer planId) {
        try{
            // 根据方案id删除方案权限表
            projectPrivilegesService.deleteProjectPrivilegesByProjectId(planId);
            // 删除方案表
            this.deletePlanById(planId);
            return true;
        }catch (Exception e){
            log.error("删除方案异常",e);
            return false;
        }
    }
}
