package com.ovopark.privilege.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.common.exception.CommonException;
import com.ovopark.privilege.common.response.ExceptionEnum;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.common.utils.CollectionUtil;
import com.ovopark.privilege.mapper.RoleMapper;
import com.ovopark.privilege.model.entity.RolePrivileges;
import com.ovopark.privilege.model.entity.Roles;
import com.ovopark.privilege.model.entity.UserPrivileges;
import com.ovopark.privilege.model.entity.Users;
import com.ovopark.privilege.model.mo.RoleMo;
import com.ovopark.privilege.model.pojo.PrivilegesPojo;
import com.ovopark.privilege.model.pojo.RolesPojo;
import com.ovopark.privilege.model.vo.MenusPrivilegesVo;
import com.ovopark.privilege.service.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * @author wxb
 */
@Service
public class RolesServiceImpl extends ServiceImpl<RoleMapper, Roles> implements RolesService {

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private MessageSource messageSource;

    @Resource
    private RolePrivilegesService rolePrivilegesService;

    @Resource
    private UsersService usersService;

    @Resource
    private PrivilegesService privilegesService;

    @Resource
    private EnterprisePrivilegesService enterprisePrivilegesService;

    @Resource
    private UserPrivilegesService userPrivilegesService;

    /**
     * @description: 获取角色根据id
     * @param: [id]
     * @return: com.ovopark.privilege.server.model.entity.Roles
     * @author wxb
     * @date: 2020/10/28 13:11
     */
    @Override
    public Roles getById(Integer id) {
        return this.baseMapper.selectById(id);
    }

    /**
     * @description: 根据id获取角色
     * @param: [id]
     * @return: com.ovopark.privilege.server.model.pojo.RolesPojo
     * @author wxb
     * @date: 2020/9/22 9:33
     */
    @Override
    public RolesPojo getRoleById(Integer id) {
        RolesPojo rolesPojo = new RolesPojo();
        Roles role = new Roles();
        role = this.baseMapper.selectById(id);
        if (role != null){
            BeanUtils.copyProperties(role,rolesPojo);
        }
        return rolesPojo;
    }

    /**
     * @description: 删除角色
     * @param: [roles]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 14:52
     */
    @Override
    public void deleteById(Roles roles) {
        if (roles != null && roles.getId() != null){
            this.baseMapper.deleteById(roles);
        }
    }

    /**
     * @description: 根据groupId获取角色列表
     * @param: [groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.Role>
     * @author wxb
     * @date: 2020/9/21 16:06
     */
    @Override
    public List<RolesPojo> getRoleListByGroupId(Integer groupId) {
        List<RolesPojo> rolesPojoList = new ArrayList<>();
        List<Roles> roles = new ArrayList<>();
        LambdaQueryWrapper<Roles> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Roles::getIsGrades,0).and(Wrapper -> Wrapper.eq(Roles::getGroupId,groupId).or().isNull(Roles::getGroupId)).orderByAsc(Roles::getId);
        roles = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(roles)){
            for (Roles role : roles){
                RolesPojo rolesPojo = new RolesPojo();
                BeanUtils.copyProperties(role,rolesPojo);
                rolesPojoList.add(rolesPojo);
            }
        }
        return rolesPojoList;
    }

    /**
     * @description: 获取用户角色列表
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.RolesPojo>
     * @author wxb
     * @date: 2020/9/22 9:30
     */
    @Override
    public List<RolesPojo> getRoleListByUser(Integer userId, Integer groupId) {
        List<RolesPojo> roles = new ArrayList<>();
        // 判断用户是否是超管
        if (userRoleService.isSystemAdmin(userId, groupId)){
            // 超管不返回任何角色
            return roles;
        }else {
            roles = this.getRoleListByGroupId(groupId);
        }
        // 设置父级角色
        this.setBelongRole(roles);
        // 设置其他属性
        this.setOtherProperty(userId, groupId, roles);
        return roles;
    }

    /**
     * @description: 设置role的父role: belongGrade
     * @param: [roles]
     * @return: void
     * @author wxb
     * @date: 2020/9/22 9:31
     */
    private void setBelongRole(List<RolesPojo> roles) {
        for(RolesPojo role : roles){
            if(role.getPId()!= null){
                RolesPojo rolesPojo = this.getRoleById(role.getPId());
                role.setBelongGrade(rolesPojo);
            }
        }
    }

    /**
     * @description: 设置其他属性
     * @param: [user, roles]
     * @return: void
     * @author wxb
     * @date: 2020/9/22 10:12
     */
    private void setOtherProperty(Integer userId, Integer groupId, List<RolesPojo> roles) {
        if(CollectionUtils.isNotEmpty(roles)){
            for(RolesPojo role : roles){
                if(userRoleService.isSystemAdmin(userId, groupId)){
                    role.setUserCount(userRoleService.getCountByRoleId(role.getId()));
                }else{
                    role.setUserCount(userRoleService.getCountByRoleIdAndGroupId(role.getId(), groupId));
                }
                List<RolePrivileges> rolePrivileges = rolePrivilegesService.getRolePrivilegesByRoleId(role.getId());
                // 设置角色对应的权限
                role.setRolePrivileges(this.getRolePrivileges(rolePrivileges));
            }
        }
    }

    /**
     * @description: 获取角色对应的权限结构
     * @param: [rolePrivileges]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.PrivilegesPojo>
     * @author wxb
     * @date: 2020/10/21 14:30
     */
    private List<MenusPrivilegesVo> getRolePrivileges(List<RolePrivileges> rolePrivileges){
        // 权限id
        List<Integer> privilegeIdList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(rolePrivileges)){
            // 用户权限结构
            for (RolePrivileges rolePrivilege : rolePrivileges){
                // 获取权限id
                if(!privilegeIdList.contains(rolePrivilege.getPrivilegeId())){
                    privilegeIdList.add(rolePrivilege.getPrivilegeId());
                }
            }
            // 获取权限列表
            List<PrivilegesPojo> privilegesPojos = privilegesService.batchGetPrivilegeListByIds(privilegeIdList);
            // 设备菜单所属产品
            return privilegesService.setProduct(privilegesService.getPrivilegesTree(privilegesPojos));
        }
        return null;
    }

    /** 
     * @description: 角色列表
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.Roles>
     * @author wxb
     * @date: 2020/9/21 17:18
     */ 
    @Override
    public List<RolesPojo> getRoleList(Integer userId, Integer groupId) {
        List<RolesPojo> roles = this.getRoleListByUser(userId, groupId);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if(!CollectionUtils.isEmpty(roles)){
            for (RolesPojo role : roles) {
                if(role.getId() <= 2){
                    role.setRoleName(messageSource.getMessage("roles.id."+role.getId(), null, locale));
                }
            }
        }
        return roles;
    }

    /**
     * @description: 通过企业id获取对应的角色列表--用户管理页面
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.RolesPojo>
     * @author wxb
     * @date: 2020/9/24 13:13
     */
    @Override
    public List<RolesPojo> getRoleListByGroup(Integer userId, Integer groupId) {
        List<RolesPojo> roles = new ArrayList<>();
        // 获取用户
        Users user = usersService.getUserById(userId);
        roles = this.getRoleListByUserAndGroupId(user, groupId);
        // 国际化
        Locale locale = LocalStorageUtils.getStorageInfo().getLocale();
        if(!CollectionUtils.isEmpty(roles)){
            for (RolesPojo role : roles) {
                if(role.getId() <= 2){
                    role.setRoleName(messageSource.getMessage("roles.id."+role.getId(), null, locale));
                }
            }
        }
        return roles;
    }

    /**
     * @description: 获取企业角色列表
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.RolesPojo>
     * @author wxb
     * @date: 2020/9/24 13:16
     */
    @Override
    public List<RolesPojo> getRoleListByUserAndGroupId(Users user, Integer groupId) {
        List<RolesPojo> roles = new ArrayList<>();
        if(userRoleService.isSystemAdmin(user.getId(), user.getGroupId()) || userRoleService.isSuperAdmin(user.getId(), groupId) || user.getIsAgency()==1){
            roles  = this.getRoleListByGroupId(user.getGroupId());
        }else{
            roles = this.getRoleListByUserIdAndGroupId(user.getId(), groupId);
        }
        for(RolesPojo role :roles){
            List<RolePrivileges> rolePrivileges = rolePrivilegesService.getRolePrivilegesByRoleId(role.getId());
            // 设置角色对应的权限
            role.setRolePrivileges(this.getRolePrivileges(rolePrivileges));
        }
        return roles;
    }

    /**
     * @description: 根据ID列表获取角色列表
     * @param: [roleIds]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.RolesPojo>
     * @author wxb
     * @date: 2020/9/24 13:48
     */
    @Override
    public List<RolesPojo> getRoleListByIds(List<Integer> roleIds) {
        List<Roles> roles = new ArrayList<>();
        List<RolesPojo> rolesPojos = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(roleIds)){
            LambdaQueryWrapper<Roles> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(Roles::getId,roleIds).eq(Roles::getIsGrades,0);
            roles =  this.baseMapper.selectList(queryWrapper);
        }
        if (CollectionUtils.isNotEmpty(roles)){
            for (Roles role : roles){
                RolesPojo rolesPojo = new RolesPojo();
                BeanUtils.copyProperties(role,rolesPojo);
                rolesPojos.add(rolesPojo);
            }
        }
        return rolesPojos;
    }

    /** 
     * @description: 根据用户id和企业id获取角色列表
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.RolesPojo>
     * @author wxb
     * @date: 2020/9/24 13:49
     */ 
    @Override
    public List<RolesPojo> getRoleListByUserIdAndGroupId(Integer userId, Integer groupId) {
        // 获取角色id列表
        List<Integer> roleIds = userRoleService.getRoleIdsByUserIdAndGroupId(userId, groupId);
        // 根据角色ID列表获取角色列表
        return this.getRoleListByIds(roleIds);
    }

    /**
     * @description: 根据id更新权限
     * @param: [rolesPojo]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 14:02
     */
    @Override
    public void updateRolesById(RolesPojo rolesPojo) {
        if (rolesPojo != null){
            Roles roles = new Roles();
            BeanUtils.copyProperties(rolesPojo,roles);
            this.baseMapper.updateById(roles);
        }
    }

    /**
     * @description: 根据角色名和企业id获取角色列表
     * @param: [roleName, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.pojo.RolesPojo>
     * @author wxb
     * @date: 2020/10/28 9:05
     */
    @Override
    public List<RolesPojo> getRoleListByRoleNameAndGroupId(String roleName, Integer groupId) {
        List<Roles> roles = new ArrayList<>();
        List<RolesPojo> rolesPojos = new ArrayList<>();
        if (roleName != null){
            LambdaQueryWrapper<Roles> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.and(Wrapper -> Wrapper.eq(Roles::getGroupId,groupId).or().isNull(Roles::getGroupId)).eq(Roles::getIsGrades,0).eq(Roles::getRoleName,roleName);
            roles = this.baseMapper.selectList(queryWrapper);
        }
        if (CollectionUtils.isNotEmpty(roles)){
            for (Roles role : roles){
                RolesPojo rolesPojo = new RolesPojo();
                BeanUtils.copyProperties(role,rolesPojo);
                rolesPojos.add(rolesPojo);
            }
        }
        return rolesPojos;
    }

    /**
     * @description: 获取角色id根据角色名称和企业id
     * @param: [roleName, groupId]
     * @return: java.util.List<java.lang.Integer>
     * @author wxb
     * @date: 2020/10/28 10:48
     */
    @Override
    public List<Integer> getRoleIdsByRoleNameAndGroupId(String roleName, Integer groupId) {
        List<Integer> roleIds = new ArrayList<>();
        List<RolesPojo> rolesPojos = this.getRoleListByRoleNameAndGroupId(roleName, groupId);
        if (CollectionUtils.isNotEmpty(rolesPojos)){
            for (RolesPojo rolesPojo : rolesPojos){
                roleIds.add(rolesPojo.getId());
            }
        }
        return roleIds;
    }

    /** 
     * @description: 新增角色并返回
     * @param: [roles]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 9:34
     */ 
    @Override
    public Roles saveRolesAndBack(Roles roles) {
        if (null != roles){
            // 存在
            if (null != roles.getId()) {
                // 更新
                this.baseMapper.updateById(roles);
            }else {
                // 新增
                this.baseMapper.insert(roles);
            }
        }
        return roles;
    }

    /**
     * @description: 保存角色和角色权限
     * @param: [role, privilegeIds]
     * @return: com.ovopark.privilege.server.model.pojo.RolesPojo
     * @author wxb
     * @date: 2020/10/28 9:37
     */
    @Override
    public Roles saveRolesAndRolePrivileges(Roles roles, List<Integer> privilegeIds) {
        // 先保存role数据 并返回
        roles = this.saveRolesAndBack(roles);
        // 再保存RolePrivileges数据
        for(Integer privilegeId : privilegeIds){
            RolePrivileges rolePrivileges  = new RolePrivileges();
            rolePrivileges.setPrivilegeId(privilegeId);
            rolePrivileges.setRoleId(roles.getId());
            rolePrivilegesService.insterRolePrivileges(rolePrivileges);
        }
        return roles;
    }

    /**
     * @description: 更新角色
     * @param: [role]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/28 11:01
     */
    @Override
    public Boolean updateRole(Integer userId, RoleMo role, List<Integer> privilegeIds) {
        if(StringUtils.isNotEmpty(role.getRoleName())) {
            List<Integer> roleIds = this.getRoleIdsByRoleNameAndGroupId(role.getRoleName(), role.getGroupId());
            if (CollectionUtils.isNotEmpty(roleIds) && !roleIds.contains(role.getId())) {
                // 重名
                throw new CommonException(ExceptionEnum.RESULT_DUPLICATE);
            }
        }
        try {
            this.updateRoleAndPrivilege(userId,role,privilegeIds);
            return true;
        } catch (Exception e){
            log.error("更新角色异常", e);
            return false;
        }
    }

    /**
     * @description: 同步更新角色
     * @param: [userId, role, privilegeIds]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/11 15:08
     */
    @Override
    public Boolean synUpdateRole(Integer userId, RoleMo role, List<Integer> privilegeIds) {
        List<Integer> newPrivilegeIds = privilegesService.getNewPrivilegeIds(privilegeIds);
        if(StringUtils.isNotEmpty(role.getRoleName())) {
            List<Integer> roleIds = this.getRoleIdsByRoleNameAndGroupId(role.getRoleName(), role.getGroupId());
            if (CollectionUtils.isNotEmpty(roleIds) && !roleIds.contains(role.getId())) {
                // 重名
                throw new CommonException(ExceptionEnum.RESULT_DUPLICATE);
            }
        }
        try {
            this.updateRoleAndPrivilege(userId,role,newPrivilegeIds);
            return true;
        } catch (Exception e){
            log.error("更新角色异常", e);
            return false;
        }
    }

    /**
     * @description: 更新角色和权限
     * @param: [role, privilegeIds]
     * @return: com.ovopark.privilege.server.model.entity.Roles
     * @author wxb
     * @date: 2020/10/28 11:01
     */
    @Override
    public Roles updateRoleAndPrivilege(Integer userId, RoleMo role, List<Integer> privilegeIds) {
        // 获取要修改的角色
        Roles updateRrole = this.getById(role.getId());
        // 获取修改的角色对应的权限id
        List<Integer> updatePrivilegeIds = rolePrivilegesService.getPrivilegeIdsByRoleId(role.getId());
        // 设置父级
        updateRrole.setPId(role.getPId());
        if(CollectionUtil.compareList(updatePrivilegeIds, privilegeIds)){
            // 权限未变,直接更新并返回
            updateRrole.setRoleName(role.getRoleName());
            updateRrole.setApplyToDept(role.getApplyToDept());
            // 更新并返回
            return this.saveRolesAndBack(updateRrole);
        } else {
            // 先更新角色
            updateRrole.setRoleName(role.getRoleName());
            updateRrole.setApplyToDept(role.getApplyToDept());
            // 返回需要修改的角色对应的权限id集合与参数集合相同的元素(即是求两集合的交集)
            List<Integer> intersections = (List<Integer>) CollectionUtils.intersection(updatePrivilegeIds, privilegeIds);
            // 去除需要修改的角色对应的权限id集合中,在intersections集合中也有的元素,注意intersections != null
            List<Integer> deletePrivilegeIds = (List<Integer>) CollectionUtils.subtract(updatePrivilegeIds, intersections);
            // 去除参数集合中,在intersections集合中也有的元素,注意intersections != null
            List<Integer> addPrivilegeIds = (List<Integer>) CollectionUtils.subtract(privilegeIds, intersections);
            // 删除已取消权限
            if (CollectionUtils.isNotEmpty(deletePrivilegeIds)) {
                // 批量删除角色对应的权限
                rolePrivilegesService.batchDeleteRolePrivilegesByPrivilegeIds(updateRrole.getId(), deletePrivilegeIds);
            }
            // 保存角色并新增权限
            Roles roleNew = this.saveRolesAndRolePrivileges(updateRrole, addPrivilegeIds);
            // 2.获取拥有该权限的所有用户
            List<Users> users = usersService.getUsersByRole(userId,null,role.getId());
            // 3.根据最近角色，同步用户权限
            this.updateUserPrivileges(users, role.getId(), deletePrivilegeIds, addPrivilegeIds, role.getGroupId());
            return roleNew;
        }
    }

    /** 
     * @description: 删除角色
     * @param: [roleId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/28 14:34
     */ 
    @Override
    public Boolean deleteRole(Integer userId, Integer roleId) {
        try {
            // 获取角色
            Roles roles = this.getById(roleId);
            if (roles != null){
                // 获取用户
                List<Users> users = usersService.getUsersByRole(userId,null,roleId);
                userRoleService.deleteUserRolesByRole(roleId, roles.getGroupId());
                // 获取用户权限id
                List<Integer> rolePrivilegeIds = rolePrivilegesService.getPrivilegeIdsByRoleId(roleId);
                rolePrivilegesService.deleteRolePrivilegesByRoleId(roleId);
                this.updateUserPrivileges(users, roleId, rolePrivilegeIds, new ArrayList<>(), roles.getGroupId());
                this.deleteById(roles);
                return true;
            }
        } catch (Exception e){
            log.error("删除角色异常", e);
            return false;
        }
        return false;
    }

    /** 
     * @description: 根据最近角色，同步用户权限
     * @param: [users, roleId, deletedPrivs, addedPrivs, enterpriseId]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 13:38
     */ 
    private void updateUserPrivileges(List<Users> users, Integer roleId, List<Integer> deletePrivilegeIds, List<Integer> addPrivilegeIds, Integer enterpriseId) {
        if (CollectionUtils.isNotEmpty(users)) {
            // 获取该用户企业的权限并和上面结果取交集
            List<Integer> groupPrivilegeIds = enterprisePrivilegesService.getPrivilegeIdsByEnterpriseId(enterpriseId);
            for (Users user : users) {
                // 获取用户现有权限列表
                List<Integer> userPrivilegeIds = userPrivilegesService.getPrvilegeIdsByUser(user.getId(), enterpriseId);
                // 现有用户没有待增加的权限，则增加
                for (Integer addPrivilegeId : addPrivilegeIds) {
                    if (!userPrivilegeIds.contains(addPrivilegeId) && groupPrivilegeIds.contains(addPrivilegeId)) {
                        UserPrivileges userPrivilege = new UserPrivileges();
                        userPrivilege.setUserId(user.getId());
                        userPrivilege.setPrivilegeId(addPrivilegeId);
                        // 添加企业ID
                        userPrivilege.setGroupId(enterpriseId);
                        userPrivilegesService.saveUserPrivileges(userPrivilege);
                    }
                }
                if (CollectionUtils.isNotEmpty(deletePrivilegeIds)) {
                    List<Integer> rolesIds = userRoleService.getRoleIdsByUser(user.getId(), enterpriseId);
                    rolesIds.remove(roleId);
                    if (CollectionUtils.isNotEmpty(rolesIds)) {
                        userPrivilegeIds = rolePrivilegesService.getPrivilegeIdsByRoles(rolesIds);
                        // 只要在用户拥有的其他角色里没有待删除的权限，则删除
                        List<Integer> userDeletePrivilegeIds = (List<Integer>)CollectionUtils.subtract(deletePrivilegeIds, userPrivilegeIds);
                        if (CollectionUtils.isNotEmpty(userDeletePrivilegeIds)) {
                            userPrivilegesService.batchDeleteUserPrivileges(user.getId(), enterpriseId, userDeletePrivilegeIds);
                        }
                    } else {
                        userPrivilegesService.batchDeleteUserPrivileges(user.getId(), user.getGroupId(),deletePrivilegeIds);
                    }
                }
            }
        }
    }
}
