package com.ovopark.privilege.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.common.constant.RoleConstant;
import com.ovopark.privilege.mapper.UserDepartmentMapper;
import com.ovopark.privilege.model.entity.UserDepartment;
import com.ovopark.privilege.service.UserDepartmentService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

/**
 * @author wxb
 * @Title: UserDepartmentServiceImpl
 * @Description: 门店角色人员表
 * @date 2020/10/14 13:55
 **/

@Slf4j
@Service
public class UserDepartmentServiceImpl extends ServiceImpl<UserDepartmentMapper, UserDepartment> implements UserDepartmentService {

    /**
     * @description: 根据用户id和企业id获取门店角色人员
     * @param: [userId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.UserDepartment>
     * @author wxb
     * @date: 2020/10/23 13:27
     */
    @Override
    public List<UserDepartment> getUserDepartmentsByUser(Integer userId, Integer groupId) {
        if (userId != null && groupId != null) {
            LambdaQueryWrapper<UserDepartment> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserDepartment::getUserId, userId).eq(UserDepartment::getGroupId, groupId);
            return this.baseMapper.selectList(queryWrapper);
        }
        return null;
    }

    /** 
     * @description: 根据用户的权限获取门店角色人员
     * @param: [userId, roleId, groupId]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.UserDepartment>
     * @author wxb
     * @date: 2020/10/23 13:41
     */ 
    @Override
    public List<UserDepartment> getUserDepartmentsByUserAndRoleId(Integer userId, Integer groupId, Integer roleId) {
        if (userId != null && roleId != null){
            LambdaQueryWrapper<UserDepartment> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserDepartment::getUserId,userId).eq(UserDepartment::getGroupId,groupId).eq(UserDepartment::getRoleId,roleId);
            return this.baseMapper.selectList(queryWrapper);
        }
        return null;
    }

    /** 
     * @description: 删除用户、角色 关联的门店
     * @param: [userId, roleId, groupId]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 13:26
     */ 
    @Override
    public void deleteByUserAndRoleId(Integer userId, Integer groupId, Integer roleId) {
        if (userId != null && roleId != null && groupId != null){
            LambdaQueryWrapper<UserDepartment> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserDepartment::getUserId,userId).eq(UserDepartment::getRoleId,roleId).eq(UserDepartment::getGroupId,groupId);
            this.baseMapper.delete(queryWrapper);
        }
    }

    /**
     * @description: 删除用户、角色 关联的门店
     * @param: [depId, roleId]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 15:23
     */
    @Override
    public void deleteByDepId(Integer depId, Integer roleId) {
        if (depId != null && roleId != null){
            LambdaQueryWrapper<UserDepartment> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserDepartment::getDepartmentId,depId).eq(UserDepartment::getRoleId,roleId);
            this.baseMapper.delete(queryWrapper);
        }
    }

    /** 
     * @description: 保存用户门店角色
     * @param: [userDepartment]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 13:53
     */ 
    @Override
    public void insertUserDepartment(UserDepartment userDepartment) {
        if (userDepartment != null){
            this.baseMapper.insert(userDepartment);
        }
    }

    /** 
     * @description: 保存用户门店角色关系
     * @param: [userId, groupId, roleId, deptList]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 15:21
     */ 
    @Override
    public void saveUserDepartment(Integer userId, Integer groupId, Integer roleId, List<Integer> deptList) {
        // 删除用户门店角色关系
        this.deleteByUserAndRoleId(userId, groupId, roleId);
        for (Integer depId : deptList) {
            // 一个门店只有一个店长
            if (roleId.equals(RoleConstant.ROLEID_MANAGER)) {
                this.deleteByDepId(depId, roleId);
            }
            UserDepartment userDepartment = new UserDepartment();
            userDepartment.setDepartmentId(depId);
            userDepartment.setUserId(userId);
            userDepartment.setRoleId(roleId);
            userDepartment.setCreateTime(new Date());
            userDepartment.setGroupId(groupId);
            this.insertUserDepartment(userDepartment);
        }
    }
}
