package com.ovopark.privilege.service.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.mapper.UserPrivilegesMapper;
import com.ovopark.privilege.model.entity.Privileges;
import com.ovopark.privilege.model.entity.UserPrivileges;
import com.ovopark.privilege.service.PrivilegesService;
import com.ovopark.privilege.service.UserPrivilegesService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * 用户权限表 服务实现类
 * </p>
 *
 * @author wxb
 * @since 2020-09-01
 */
@Slf4j
@Service
public class UserPrivilegesServiceImpl extends ServiceImpl<UserPrivilegesMapper, UserPrivileges> implements UserPrivilegesService {

    @Resource
    private PrivilegesService privilegesService;

    /** 
     * @description: 删除用户权限
     * @param: [userId, groupId]
     * @return: void
     * @author wxb
     * @date: 2020/9/24 14:54
     */ 
    @Override
    public void deleteUserPrivileges(Integer userId, Integer groupId) {
        if (userId != null && groupId != null){
            LambdaQueryWrapper<UserPrivileges> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserPrivileges::getUserId,userId).eq(UserPrivileges::getGroupId,groupId);
            this.baseMapper.delete(queryWrapper);
        }
    }

    /**
     * @description: 保存用户权限
     * @param: [userPrivileges]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 13:42
     */
    @Override
    public void saveUserPrivileges(UserPrivileges userPrivileges) {
        this.baseMapper.insert(userPrivileges);
        // 查询企业权限
        UserPrivileges userPrivilege = this.getUserPrivilegesByUserIdAndPrivilegeId(userPrivileges.getUserId(),userPrivileges.getPrivilegeId());
        // 不存在
        if (userPrivilege == null){
            // 新增
            this.baseMapper.insert(userPrivilege);
        }
    }

    /**
     * @description: 获取用户的所有权限id
     * @param: [userId, groupId]
     * @return: java.util.List<java.lang.Integer>
     * @author wxb
     * @date: 2020/10/28 13:42
     */
    @Override
    public List<Integer> getPrvilegeIdsByUser(Integer userId, Integer groupId) {
        List<Integer> prvilegeIds = new ArrayList<>();
        List<UserPrivileges> userPrivileges = new ArrayList<>();
        LambdaQueryWrapper<UserPrivileges> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserPrivileges::getUserId,userId).eq(UserPrivileges::getGroupId,groupId);
        userPrivileges = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(userPrivileges)){
            for (UserPrivileges userPrivilege : userPrivileges){
                prvilegeIds.add(userPrivilege.getPrivilegeId());
            }
        }
        return prvilegeIds;
    }

    /** 
     * @description: 批量删除用户的权限
     * @param: [userId, groupId, privilegeIds]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 14:01
     */ 
    @Override
    public void batchDeleteUserPrivileges(Integer userId, Integer groupId, List<Integer> privilegeIds) {
        if (userId != null && groupId != null && CollectionUtils.isNotEmpty(privilegeIds)){
            LambdaQueryWrapper<UserPrivileges> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserPrivileges::getUserId,userId).eq(UserPrivileges::getGroupId,groupId).in(UserPrivileges::getPrivilegeId,privilegeIds);
            this.baseMapper.delete(queryWrapper);
        }
    }

    /**
     * @description: 获取用户权限 权限id
     * @param: [userId, privilegeId]
     * @return: com.ovopark.privilege.server.model.entity.UserPrivileges
     * @author wxb
     * @date: 2020/11/10 20:04
     */
    @Override
    public UserPrivileges getUserPrivilegesByUserIdAndPrivilegeId(Integer userId, Integer privilegeId) {
        LambdaQueryWrapper<UserPrivileges> queryWrapper = new LambdaQueryWrapper();
        queryWrapper.eq(UserPrivileges::getUserId,userId).eq(UserPrivileges::getPrivilegeId,privilegeId);
        return this.baseMapper.selectOne(queryWrapper);
    }

    /**
     * @description: 获取用户权限 权限名称
     * @param: [userId, privilegeName]
     * @return: com.ovopark.privilege.server.model.entity.UserPrivileges
     * @author wxb
     * @date: 2020/11/13 14:04
     */
    @Override
    public UserPrivileges getUserPrivilegesByUserIdAndPrivilegeName(Integer userId, String privilegeName) {
        // 根据权限名获取权限
        Privileges privileges = privilegesService.getPrivilegesByPrivilegeName(privilegeName);
        if (privileges != null){
            LambdaQueryWrapper<UserPrivileges> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserPrivileges::getUserId,userId).eq(UserPrivileges::getPrivilegeId,privileges.getId());
            return this.baseMapper.selectOne(queryWrapper);
        }
        return null;
    }

    /**
     * @description: 新增用户权限
     * @param: [userPrivileges]
     * @return: void
     * @author wxb
     * @date: 2020/11/17 19:31
     */
    @Override
    public void insterUserPrivileges(UserPrivileges userPrivileges) {
        this.baseMapper.insert(userPrivileges);
    }

    /**
     * @description: 删除用户的所有权限
     * @param: [userId]
     * @return: void
     * @author wxb
     * @date: 2020/11/17 19:31
     */
    @Override
    public Boolean deleteUserPrivilegesByUserId(Integer userId) {
        LambdaQueryWrapper<UserPrivileges> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserPrivileges::getUserId,userId);
        return this.baseMapper.delete(queryWrapper) > 0;
    }

    /** 
     * @description: 删除用户指定权限
     * @param: [userId, privilegeId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/17 19:37
     */ 
    @Override
    public Boolean deleteUserPrivilegesByUserIdAndPrivilegeId(Integer userId, Integer privilegeId) {
        // 获取用户权限
        UserPrivileges userPrivileges = this.getUserPrivilegesByUserIdAndPrivilegeId(userId, privilegeId);
        if (userPrivileges != null){
            this.baseMapper.deleteById(userPrivileges);
            return this.baseMapper.deleteById(userPrivileges) > 0;
        }
        return false;
    }

    /**
     * @description: 更新用户权限
     * @param: [userPrivilege]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/11/17 19:38
     */
    @Override
    public Boolean updateUserPrivileges(UserPrivileges userPrivileges) {
        if (userPrivileges != null){
            return this.baseMapper.updateById(userPrivileges) > 0;
        }
        return false;
    }
}
