package com.ovopark.privilege.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.common.constant.RoleConstant;
import com.ovopark.privilege.mapper.UserRoleMapper;
import com.ovopark.privilege.model.entity.UserRole;
import com.ovopark.privilege.model.entity.Users;
import com.ovopark.privilege.service.UserDepartmentService;
import com.ovopark.privilege.service.UserRoleService;
import com.ovopark.privilege.service.UsersService;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author wxb
 */
@Service
public class UserRoleServiceImpl extends ServiceImpl<UserRoleMapper, UserRole> implements UserRoleService {

    @Resource
    private UserDepartmentService userDepartmentService;

    @Resource
    private UsersService usersService;

    /**
     * @description: 是否是超级管理员
     * @param: [userId, groupId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/9/17 15:15
     */
    @Override
    public Boolean isSuperAdmin(Integer userId, Integer groupId) {
        LambdaQueryWrapper<UserRole> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserRole::getRoleId,1).eq(UserRole::getUserId,userId);
        if (groupId != null){
            queryWrapper.eq(UserRole::getGroupId,groupId);
        }
        List<UserRole> userRoles = this.baseMapper.selectList(queryWrapper);
        return CollectionUtils.isNotEmpty(userRoles);
    }

    /** 
     * @description: 是否是系统管理员
     * @param: [userId, groupId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/9/21 16:20
     */ 
    @Override
    public Boolean isSystemAdmin(Integer userId, Integer groupId) {
        if(userId == null){
            return false;
        }
        if(groupId == null && this.isSuperAdmin(userId, null)){
            return true;
        }
        return false;
    }

    /**
     * @description: 根据用户ID和企业ID获取角色ID
     * @param: [userId, groupId]
     * @return: java.util.List<java.lang.Integer>
     * @author wxb
     * @date: 2020/9/17 15:16
     */
    @Override
    public List<Integer> getRoleIdsByUserIdAndGroupId(Integer userId, Integer groupId) {
        LambdaQueryWrapper<UserRole> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserRole::getUserId,userId).eq(UserRole::getGroupId,groupId);
        List<UserRole> userRoles = this.baseMapper.selectList(queryWrapper);
        List<Integer> roleIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(userRoles)){
            for (UserRole userRole:userRoles){
                roleIds.add(userRole.getRoleId());
            }
        }
        return roleIds;
    }

    /** 
     * @description: 
     * @param: [roleId]
     * @return: java.lang.Integer
     * @author wxb
     * @date: 2020/9/22 9:45
     */ 
    @Override
    public Integer getCountByRoleId(Integer roleId) {
        return this.baseMapper.getCountByRoleId(roleId);
    }

    /** 
     * @description:
     * @param: [roleId, groupId]
     * @return: java.lang.Integer
     * @author wxb
     * @date: 2020/9/22 9:45
     */ 
    @Override
    public Integer getCountByRoleIdAndGroupId(Integer roleId, Integer groupId) {
        return this.baseMapper.getCountByRoleIdAndGroupId(roleId, groupId);
    }

    /** 
     * @description: 查询用户设置的角色
     * @param: [userId, groupId]
     * @return: java.util.List<java.lang.Integer>
     * @author wxb
     * @date: 2020/10/23 13:23
     */ 
    @Override
    public List<Integer> getRoleIdsByUser(Integer userId, Integer groupId) {
        List<Integer> roleIds = new ArrayList();
        if (userId != null && groupId != null){
            roleIds = this.baseMapper.getRoleIdsByUser(userId, groupId);
        }
        return roleIds;
    }

    /** 
     * @description: 删除用户的角色
     * @param: [userId, groupId]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 13:46
     */ 
    @Override
    public void deleteUserRolesByUser(Integer userId, Integer groupId) {
        if (userId != null && groupId != null){
            LambdaQueryWrapper<UserRole> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserRole::getUserId,userId).eq(UserRole::getGroupId,groupId);
            this.baseMapper.delete(queryWrapper);
        }
    }

    /**
     * @description: 删除用户的角色
     * @param: [roleId, groupId]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 14:45
     */
    @Override
    public void deleteUserRolesByRole(Integer roleId, Integer groupId) {
        if (roleId != null && groupId != null){
            LambdaQueryWrapper<UserRole> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(UserRole::getRoleId,roleId).eq(UserRole::getGroupId,groupId);
            this.baseMapper.delete(queryWrapper);
        }
    }

    /** 
     * @description: 保存用户角色
     * @param: [userRole]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 13:49
     */ 
    @Override
    public void insterUserRoles(UserRole userRole) {
        if (userRole != null){
            this.baseMapper.insert(userRole);
        }
    }

    /**
     * @description: 保存用户角色关系
     * @param: [roleIds, userId, groupId]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 15:11
     */
    @Override
    public void saveUserRoles(List<Integer> roleIds, Integer userId, Integer groupId) {
        Set<Integer> roleIdSet = new HashSet<>();
        if (CollectionUtil.isNotEmpty(roleIds)){
            roleIdSet.addAll(roleIds);
        }
        // 查询用户设置的角色
        List<Integer> roleIdsByUser = this.getRoleIdsByUser(userId,groupId);
        if (CollectionUtil.isNotEmpty(roleIdsByUser)){
            for (Integer roleId : roleIdsByUser){
                // 删除用户、角色 关联的门店，但不删除 用户店长角色
                if (!roleId.equals(RoleConstant.ROLEID_MANAGER)){
                    userDepartmentService.deleteByUserAndRoleId(userId,groupId,roleId);
                }
            }
        }
        // 删除用户的角色
        this.deleteUserRolesByUser(userId,groupId);
        if(userId != null){
            Iterator<Integer> iterator = roleIdSet.iterator();
            while (iterator.hasNext()){
                Integer roleId = iterator.next();
                UserRole ur = new UserRole();
                ur.setRoleId(roleId);
                ur.setUserId(userId);
                ur.setGroupId(groupId);
                this.insterUserRoles(ur);
            }
        }
    }

    /**
     * @description: 获取拥有该角色的所有用户id
     * @param: [roleId, groupId, isSystemAdmin]
     * @return: java.util.List<java.lang.Integer>
     * @author wxb
     * @date: 2020/10/28 10:00
     */
    @Override
    public List<Integer> getUserIdsByRole(Integer userId, Integer groupId, Integer roleId, boolean isSystemAdmin) {
        // 用户id集
        List<Integer> userIds = new ArrayList();
        if (roleId == null){
            return userIds;
        }
        // 获取当前用户
        Users users = usersService.getUserById(userId);
        // 超管
        if (isSystemAdmin){
            if (groupId != null){
                userIds = this.baseMapper.getUserIdsByRole(roleId, groupId);
            } else {
                userIds = this.baseMapper.getUserIdsByRole(roleId,null);
            }
        } else {
            userIds = this.baseMapper.getUserIdsByRole(roleId,users.getGroupId());
        }
        return userIds;
    }

    /**
     * @description: 获取用户的角色
     * @param: [userId, groupId]
     * @return: com.ovopark.privilege.server.model.entity.UserRole
     * @author wxb
     * @date: 2020/11/12 15:07
     */
    @Override
    public UserRole getUserRoleByUserIdAndGroupId(Integer userId, Integer groupId) {
        LambdaQueryWrapper<UserRole> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserRole::getUserId,userId).eq(UserRole::getGroupId,groupId);
        return this.baseMapper.selectOne(queryWrapper);
    }
}
