package com.ovopark.privilege.service.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ovopark.privilege.common.response.CommonPage;
import com.ovopark.privilege.common.response.PageResultUtil;
import com.ovopark.privilege.common.support.LocalStorageUtils;
import com.ovopark.privilege.common.utils.JsonUtils;
import com.ovopark.privilege.common.utils.SessionUtil;
import com.ovopark.privilege.common.utils.StringUtils;
import com.ovopark.privilege.common.utils.TokenUtils;
import com.ovopark.privilege.mapper.UsersMapper;
import com.ovopark.privilege.model.entity.Logs;
import com.ovopark.privilege.model.entity.UserEnterprise;
import com.ovopark.privilege.model.entity.Users;
import com.ovopark.privilege.service.*;
import com.ovopark.shopweb.utils.LogUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 用户表 服务实现类
 * </p>
 *
 * @author wxb
 * @since 2020-09-10
 */
@Slf4j
@Service
public class UsersServiceImpl extends ServiceImpl<UsersMapper, Users> implements UsersService {

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private UserEnterpriseService userEnterpriseService;

    @Resource
    private LogsService logsService;

    @Resource
    private UserPrivilegesService userPrivilegesService;

    /**
     * @description: 根据用户名和企业ID获取用户
     * @param: [userName, groupId]
     * @return: com.ovopark.privilege.server.model.entity.Users
     * @author wxb
     * @date: 2020/9/17 15:16
     */
    @Override
    public Users getUserByUserNameAndGroupId(String userName, Integer groupId) {
        LambdaQueryWrapper<Users> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Users::getUserName,userName).eq(Users::getGroupId,groupId).eq(Users::getIsDel,0);
        return this.baseMapper.selectOne(queryWrapper);
    }

    /**
     * @description: 根据用户名查询用户
     * @param: [userName]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.Users>
     * @author wxb
     * @date: 2020/9/17 15:16
     */
    @Override
    public List<Users> getUserByUserName(String userName) {
        LambdaQueryWrapper<Users> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Users::getUserName,userName).eq(Users::getIsDel,0);
        return this.baseMapper.selectList(queryWrapper);
    }

    /**
     * @description: 根据邮箱查询用户
     * @param: [email]
     * @return: com.ovopark.privilege.server.model.entity.Users
     * @author wxb
     * @date: 2020/9/17 15:16
     */
    @Override
    public Users getUserByUserEmail(String email) {
        LambdaQueryWrapper<Users> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Users::getMail,email).eq(Users::getIsDel,0);
        List<Users> users = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(users)){
            return users.get(0);
        }
        return null;
    }

    /**
     * @description: 根据手机号查询用户
     * @param: [phone]
     * @return: com.ovopark.privilege.server.model.entity.Users
     * @author wxb
     * @date: 2020/9/17 15:16
     */
    @Override
    public Users getUserByUserPhone(String phone) {
        LambdaQueryWrapper<Users> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Users::getMobilePhone,phone).eq(Users::getIsDel,0);
        List<Users> users = this.baseMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(users)){
            return users.get(0);
        }
        return null;
    }

    /**
     * @description: 根据id获取用户
     * @param: [id]
     * @return: com.ovopark.privilege.server.model.entity.Users
     * @author wxb
     * @date: 2020/9/17 15:16
     */
    @Override
    public Users getUserById(Integer id) {
        // todo 这里需要加缓存
        return this.baseMapper.selectById(id);
    }

    /**
     * @description: 更新用户
     * @param: [users]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 15:39
     */
    @Override
    public void updateUserById(Users users) {
        if (users != null){
            this.baseMapper.updateById(users);
        }
    }

    /**
     * @description: 新增用户
     * @param: [users]
     * @return: void
     * @author wxb
     * @date: 2020/10/23 15:40
     */
    @Override
    public void insterUser(Users users) {
        if (users != null){
            this.baseMapper.insert(users);
        }
    }

    /**
     * @description: 获取拥有该角色的所有用户
     * @param: [userName, roleId]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.Users>
     * @author wxb
     * @date: 2020/10/28 9:54
     */
    @Override
    public List<Users> getUsersByRole(Integer userId, String userName, Integer roleId) {
        Users users = this.getUserById(userId);
        boolean isSystemAdmin =(userRoleService.isSystemAdmin(users.getId(), users.getGroupId()) || users.getIsAgency() == 1);
        List<Integer> userIds = userRoleService.getUserIdsByRole(users.getId(), users.getGroupId(),roleId,isSystemAdmin);
        return  this.getUsersByUserNameAndUserIds(userName,userIds);
    }

    /**
     * @description: 分页查询用户
     * @param: [userName, userIds, pageNum, pageSize]
     * @return: com.baomidou.mybatisplus.core.metadata.IPage<com.ovopark.privilege.server.model.entity.Users>
     * @author wxb
     * @date: 2020/10/28 10:19
     */
    @Override
    public CommonPage<Users> getUsersByUserNameAndUserIdsAsPage(String userName, List<Integer> userIds, Integer pageNum, Integer pageSize) {
        LambdaQueryWrapper<Users> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(Users::getIsDel,0).eq(Users::getIsFrozen,0).in(Users::getId,userIds);
        if (StringUtils.isNotEmpty(userName)){
            queryWrapper.like(Users::getUserName,userName);
        }
        if (pageNum != null && pageSize != null){
            Page<Users> page = new Page<>(pageNum,pageSize);
            IPage<Users> usersIPage = this.baseMapper.selectPage(page, queryWrapper);
            return PageResultUtil.fromIPage(usersIPage);
        }
        return null;
    }

    /**
     * @description: 查询用户
     * @param: [userName, userIds]
     * @return: java.util.List<com.ovopark.privilege.server.model.entity.Users>
     * @author wxb
     * @date: 2020/10/28 10:32
     */
    @Override
    public List<Users> getUsersByUserNameAndUserIds(String userName, List<Integer> userIds) {
        if (CollectionUtils.isNotEmpty(userIds)){
            LambdaQueryWrapper<Users> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(Users::getIsDel,0).eq(Users::getIsFrozen,0).in(Users::getId,userIds);
            if (StringUtils.isNotEmpty(userName)){
                queryWrapper.like(Users::getUserName,userName);
            }
            return this.baseMapper.selectList(queryWrapper);
        }
        return null;
    }

    /** 
     * @description: 删除用户
     * @param: [userId, groupId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/28 15:37
     */ 
    @Override
    public Boolean deleteUser(Integer userId, Integer groupId, String token) {
        try {
            // 填写日志
            Users curUser = this.getUserById(LocalStorageUtils.getStorageInfo().getUserId());
            UserEnterprise userEnterprise = userEnterpriseService.getUserEnterpriseByUserIdAndEnterpriseId(userId, groupId);
            if (userEnterprise != null) {
                userEnterprise.setIsDel(1);
                userEnterpriseService.updateUserEnterpriseById(userEnterprise);
            }
            Users user = this.getUserById(userId);
            Logs logs = new Logs();
            logs.setUserId(curUser.getId());
            // 方法名/action名
            logs.setOperation("deleteUser");
            logs.setUserName(curUser.getUserName());
            logs.setParameters(JsonUtils.beanToJsonString(user));
            logs.setRemoteIp(LogUtils.getRemoteIp(SessionUtil.getRequest()));
            logs.setPlatform(LogUtils.getPlatform(SessionUtil.getRequest()));
            logs.setCreateTime(LocalDateTime.now());
            logs.setLogLevel(0);
            logs.setDescription("删除用户id："+user.getId()+",名称："+user.getUserName());
            if(curUser != null && curUser.getGroupId() != null){
                logs.setEnterpriseId(curUser.getGroupId());
            }
            logsService.insterLogs(logs);
            this.deleteUserByIdAndToken(userId);
            this.userRoleService.deleteUserRolesByUser(userId, groupId);
            userPrivilegesService.deleteUserPrivileges(userId, groupId);
            return true;
        } catch (Exception e) {
            log.error("删除用户异常", e);
            return false;
        }
    }

    /**
     * @description: 删除用户
     * @param: [userId]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 16:49
     */
    private void deleteUserByIdAndToken(Integer userId) {
        Users user = this.getUserById(userId);
        if (user != null) {
            // 登出token
            TokenUtils.logoutByUserId(userId);
            user.setIsDel(1);
            // 登陆名改掉，防止占用
            user.setUserName("____deletedId" + user.getId());
            this.updateUserById(user);
        }
    }

    /** 
     * @description: 冻结用户
     * @param: [userId, groupId, token]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/28 17:21
     */ 
    @Override
    public Boolean userFreeze(Integer userId, Integer groupId, String token) {
        try {
            // 填写日志
            Users curUser = this.getUserById(LocalStorageUtils.getStorageInfo().getUserId());
            UserEnterprise userEnterprise = userEnterpriseService.getUserEnterpriseByUserIdAndEnterpriseId(userId, groupId);
            if (userEnterprise != null) {
                // 冻结
                userEnterprise.setIsFrozen(1);
                userEnterpriseService.updateUserEnterpriseById(userEnterprise);
            }
            Users user = this.getUserById(userId);
            Logs logs = new Logs();
            logs.setUserId(curUser.getId());
            // 方法名/action名
            logs.setOperation("userFreeze");
            logs.setUserName(curUser.getUserName());
            logs.setParameters(JsonUtils.beanToJsonString(user));
            logs.setRemoteIp(LogUtils.getRemoteIp(SessionUtil.getRequest()));
            logs.setPlatform(LogUtils.getPlatform(SessionUtil.getRequest()));
            logs.setCreateTime(LocalDateTime.now());
            logs.setLogLevel(0);
            logs.setDescription("冻结用户id："+user.getId()+",名称："+user.getUserName());
            if(curUser != null && curUser.getGroupId() != null){
                logs.setEnterpriseId(curUser.getGroupId());
            }
            logsService.insterLogs(logs);
            this.freezeUserByIdAndToken(userId);
            return true;
        } catch (Exception e) {
            log.error("冻结用户异常", e);
            return false;
        }
    }

    /** 
     * @description: 冻结用户 登出token
     * @param: [userId, token]
     * @return: void
     * @author wxb
     * @date: 2020/10/28 17:23
     */ 
    private void freezeUserByIdAndToken(Integer userId) {
        Users user = this.getUserById(userId);
        if (user != null) {
            // 登出token
            TokenUtils.logoutByUserId(userId);
            // 冻结
            user.setIsFrozen(1);
            this.updateUserById(user);
        }
    }

    /**
     * @description: 解冻用户
     * @param: [userId, groupId]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/28 17:26
     */
    @Override
    public Boolean userUnfreeze(Integer userId, Integer groupId) {
        try {
            // 获取用户
            Users users = this.getUserById(userId);
            UserEnterprise userEnterprise = userEnterpriseService.getUserEnterpriseByUserIdAndEnterpriseId(userId, groupId);
            if (userEnterprise != null) {
                // 解冻
                userEnterprise.setIsFrozen(0);
                userEnterpriseService.updateUserEnterpriseById(userEnterprise);
            }
            if (users != null){
                // 解冻
                users.setIsFrozen(0);
                this.updateUserById(users);
            }
            return true;
        } catch (Exception e) {
            log.error("解冻用户异常", e);
            return false;
        }
    }

    /**
     * @description: 批量删除用户
     * @param: [userIds, groupId, token]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/29 18:40
     */
    @Override
    public Boolean deleteBatchUser(String userIds, Integer groupId, String token) {
        try {
            // 填写日志
            Users curUser = this.getUserById(LocalStorageUtils.getStorageInfo().getUserId());
            List<Integer> userIdList = StringUtils.stringToList(userIds);
            if (CollectionUtils.isNotEmpty(userIdList)){
                for (Integer userId : userIdList){
                    UserEnterprise userEnterprise = userEnterpriseService.getUserEnterpriseByUserIdAndEnterpriseId(userId, groupId);
                    if (userEnterprise != null) {
                        userEnterprise.setIsDel(1);
                        userEnterpriseService.updateUserEnterpriseById(userEnterprise);
                    }
                    Users user = this.getUserById(userId);
                    Logs logs = new Logs();
                    logs.setUserId(curUser.getId());
                    // 方法名/action名
                    logs.setOperation("deleteUser");
                    logs.setUserName(curUser.getUserName());
                    logs.setParameters(JsonUtils.beanToJsonString(user));
                    logs.setRemoteIp(LogUtils.getRemoteIp(SessionUtil.getRequest()));
                    logs.setPlatform(LogUtils.getPlatform(SessionUtil.getRequest()));
                    logs.setCreateTime(LocalDateTime.now());
                    logs.setLogLevel(0);
                    logs.setDescription("删除用户id："+user.getId()+",名称："+user.getUserName());
                    if(curUser != null && curUser.getGroupId() != null){
                        logs.setEnterpriseId(curUser.getGroupId());
                    }
                    logsService.insterLogs(logs);
                    this.deleteUserByIdAndToken(userId);
                    this.userRoleService.deleteUserRolesByUser(userId, groupId);
                    userPrivilegesService.deleteUserPrivileges(userId, groupId);
                }
                return true;
            }
        } catch (Exception e) {
            log.error("删除用户异常", e);
            return false;
        }
        return false;
    }

    /**
     * @description: 批量冻结用户
     * @param: [userIds, groupId, token]
     * @return: java.lang.Boolean
     * @author wxb
     * @date: 2020/10/29 18:45
     */
    @Override
    public Boolean userBatchFreeze(String userIds, Integer groupId, String token) {
        try {
            // 填写日志
            Users curUser = this.getUserById(LocalStorageUtils.getStorageInfo().getUserId());
            List<Integer> userIdList = StringUtils.stringToList(userIds);
            if (CollectionUtils.isNotEmpty(userIdList)){
                for (Integer userId : userIdList){
                    UserEnterprise userEnterprise = userEnterpriseService.getUserEnterpriseByUserIdAndEnterpriseId(userId, groupId);
                    if (userEnterprise != null) {
                        // 冻结
                        userEnterprise.setIsFrozen(1);
                        userEnterpriseService.updateUserEnterpriseById(userEnterprise);
                    }
                    Users user = this.getUserById(userId);
                    Logs logs = new Logs();
                    logs.setUserId(curUser.getId());
                    // 方法名/action名
                    logs.setOperation("userFreeze");
                    logs.setUserName(curUser.getUserName());
                    logs.setParameters(JsonUtils.beanToJsonString(user));
                    logs.setRemoteIp(LogUtils.getRemoteIp(SessionUtil.getRequest()));
                    logs.setPlatform(LogUtils.getPlatform(SessionUtil.getRequest()));
                    logs.setCreateTime(LocalDateTime.now());
                    logs.setLogLevel(0);
                    logs.setDescription("冻结用户id："+user.getId()+",名称："+user.getUserName());
                    if(curUser != null && curUser.getGroupId() != null){
                        logs.setEnterpriseId(curUser.getGroupId());
                    }
                    logsService.insterLogs(logs);
                    this.freezeUserByIdAndToken(userId);
                }
                return true;
            }
        } catch (Exception e) {
            log.error("冻结用户异常", e);
            return false;
        }
        return false;
    }
}
