package com.ovopark.module.shared.es7x;

import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.module.shared.PageModel;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.admin.indices.alias.get.GetAliasesRequest;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.*;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.GetAliasesResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.core.CountRequest;
import org.elasticsearch.client.core.CountResponse;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.index.engine.VersionConflictEngineException;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.index.reindex.ScrollableHitSource;
import org.elasticsearch.index.reindex.UpdateByQueryRequest;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.script.Script;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import static com.ovopark.kernel.shared.Util.*;

@Slf4j
public class ElasticsearchOperationsImpl implements ElasticsearchOperations{

    @Autowired
    private RestHighLevelClient elasticsearchClient;

    @Resource(name = "ES_JSONAccessor")
    private JSONAccessor jsonAccessor;

    @Override
    public <T extends Serializable> void save(String index, Document<T> model) {
        try {
            boolean idIsNull=model.id()==null;
            IndexResponse indexResponse = elasticsearchClient.index(new IndexRequest(index).source(
                    jsonAccessor.read(jsonAccessor.format(model))
            ).id(idIsNull?null:String.valueOf(model.id())), RequestOptions.DEFAULT);
            DocWriteResponse.Result result = indexResponse.getResult();
            if(result== DocWriteResponse.Result.CREATED) {
                model.id(indexResponse.getId());
                model.docIndexName(indexResponse.getIndex());
                model.ver(indexResponse.getVersion());
                model.seqNo(indexResponse.getSeqNo());
                model.primaryTerm(indexResponse.getPrimaryTerm());
            }
            else if(result== DocWriteResponse.Result.UPDATED || result== DocWriteResponse.Result.NOOP) {
                model.docIndexName(indexResponse.getIndex());
                model.ver(indexResponse.getVersion());
                model.seqNo(indexResponse.getSeqNo());
                model.primaryTerm(indexResponse.getPrimaryTerm());
            }
            else {
                throw new IllegalStateException("create error: "+ JSONAccessor.impl().format(model));
            }
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public <T extends Serializable> BulkSaveResponse bulkSave(String index, List<? extends Document<T>> modelList){

        BulkRequest bulkRequest = new BulkRequest();
        for (Document<T> document : modelList) {
            boolean idIsNull=document.id()==null;
            IndexRequest indexRequest = new IndexRequest(index).source(
                    jsonAccessor.read(jsonAccessor.format(document))
            ).id(idIsNull ? null : String.valueOf(document.id()));
            bulkRequest.add(indexRequest);
        }

        BulkSaveResponse bulkSaveResponse=new BulkSaveResponse();
        try {
            BulkResponse bulkResponse = elasticsearchClient.bulk(bulkRequest, RequestOptions.DEFAULT);
            BulkItemResponse[] items = bulkResponse.getItems();
            for (int i = 0; i < items.length; i++) {
                BulkItemResponse item = items[i];
                if (item.getFailure() != null) {
                    bulkSaveResponse.fail(i);
                }
                else {
                    bulkSaveResponse.success(i,item.getId(),item.getIndex());
                    modelList.get(i).id(item.getId());
                }
            }
            return bulkSaveResponse;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public <T extends Serializable> void update(String index, Document<T> model) {
        update0(index, model,false);
    }

    private <T extends Serializable> void update0(String index, Document<T> model,boolean upsert) {

        if (!upsert && model.id()==null) {
            throw new RuntimeException("id is missing: "+null+" , index: "+ index);
        }

        UpdateRequest updateRequest=new UpdateRequest();
        updateRequest.id(model.id()==null?null:String.valueOf(model.id()));
        updateRequest.index(index);
        if (upsert) {
            updateRequest.doc(jsonAccessor.read(jsonAccessor.format(model)));
            updateRequest.docAsUpsert(true);
        }
        else {
            updateRequest.doc(jsonAccessor.read(jsonAccessor.format(model)));
        }
        updateRequest.retryOnConflict(3);
        if (model.primaryTerm()>0) {
            updateRequest.setIfPrimaryTerm(model.primaryTerm());
        }
        if (model.seqNo()>0) {
            updateRequest.setIfSeqNo(model.seqNo());
        }

        try {
            UpdateResponse response = elasticsearchClient.update(updateRequest, RequestOptions.DEFAULT);
            DocWriteResponse.Result result = response.getResult();
            if (result== DocWriteResponse.Result.NOT_FOUND) {
                throw new RuntimeException("id is missing: "+model.id()+" , index: "+ index);
            }
            else if (result== DocWriteResponse.Result.CREATED) {
                model.docIndexName(response.getIndex());
                model.primaryTerm(response.getPrimaryTerm());
                model.seqNo(response.getSeqNo());
                model.ver(response.getVersion());
            }
        }
        catch (IOException e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
        catch (VersionConflictEngineException e){
            log.error(e.getMessage(),e);
            throw e;
        }
        catch (ElasticsearchException e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public <T extends Serializable> void upsert(String index, Document<T> model) {
        update0(index,model,true);
    }

    @Override
    public void updatePartOfDoc(String index, String id, Map<String, Object> data) {
        if (isEmpty(id)) {
            throw new RuntimeException("id is missing: "+null+" , index: "+ index);
        }

        UpdateRequest updateRequest=new UpdateRequest();
        updateRequest.id(id);
        updateRequest.index(index);
        updateRequest.doc(data);
        updateRequest.retryOnConflict(3);

        try {
            UpdateResponse response = elasticsearchClient.update(updateRequest, RequestOptions.DEFAULT);
            DocWriteResponse.Result result = response.getResult();
            if (result== DocWriteResponse.Result.NOT_FOUND) {
                throw new RuntimeException("id is missing: "+id+" , index: "+ index);
            }
            else if (result== DocWriteResponse.Result.CREATED) {
                return;
            }
        }
        catch (IOException e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
        catch (VersionConflictEngineException e){
            log.error(e.getMessage(),e);
            throw e;
        }
        catch (ElasticsearchException e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public void updatePartOfDoc(String index, String id, Supplier<Script> scriptSupplier) {
        UpdateRequest updateRequest=new UpdateRequest();
        updateRequest.id(id);
        updateRequest.index(index);
        updateRequest.script(scriptSupplier.get());
        updateRequest.retryOnConflict(3);
        try {
            UpdateResponse response = elasticsearchClient.update(updateRequest, RequestOptions.DEFAULT);
            DocWriteResponse.Result result = response.getResult();
            if (result== DocWriteResponse.Result.NOT_FOUND) {
                throw new RuntimeException("id is missing: "+id+" , index: "+index);
            }
        } catch (IOException e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
        catch (ElasticsearchException e) {
            if (e.status() == RestStatus.NOT_FOUND) {
                throw new RuntimeException("id is missing: "+id+" , index: "+index);
            }
        }

    }

    @Override
    public void updatePartOfDocInMultiIndex(String id, Supplier<Script> scriptSupplier, String... alias) {
          /*
        POST my-index-000001/_update_by_query
        {
          "script": {
            "source": "ctx._source.count++",
            "lang": "painless"
          },
          "query": {
            "term": {
              "user.id": "kimchy"
            }
          }
        }
         */
        if (isEmpty(id)) {
            throw new IllegalArgumentException("id is empty: "+id);
        }
        UpdateByQueryRequest updateByQueryRequest=new UpdateByQueryRequest(alias);
        updateByQueryRequest.setQuery(new TermQueryBuilder("_id",id));
        updateByQueryRequest.setScript(scriptSupplier.get());
        updateByQueryRequest.setConflicts("proceed");
        try {
            BulkByScrollResponse response = elasticsearchClient.updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
            if (response.isTimedOut()) {
                throw new RuntimeException("timeout: "+id);
            }

            List<ScrollableHitSource.SearchFailure> searchFailures = response.getSearchFailures();
            List<BulkItemResponse.Failure> bulkFailures = response.getBulkFailures();
            if (isNotEmpty(searchFailures)) {
                ScrollableHitSource.SearchFailure first = searchFailures.getFirst();
                log.error(first.getStatus()+" , searchFailures: "+ first.getNodeId()+":"+first.getShardId()+":"+first.getIndex(),first);
            }
            if (isNotEmpty(bulkFailures)) {
                throw new RuntimeException("update error: "+id);
            }
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public BulkUpdateResponse updatePartOfDocInMultiIndex(List<String> idList, Supplier<Script> scriptSupplier, String... alias) {
        if (isEmpty(idList)) {
            throw new IllegalArgumentException("id is empty");
        }
        UpdateByQueryRequest updateByQueryRequest=new UpdateByQueryRequest(alias);
        updateByQueryRequest.setQuery(QueryBuilders.termsQuery("_id",idList));
        updateByQueryRequest.setScript(scriptSupplier.get());
        updateByQueryRequest.setConflicts("proceed");
        try {
            BulkByScrollResponse response = elasticsearchClient.updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
            if (response.isTimedOut()) {
                throw new RuntimeException("timeout");
            }


            BulkUpdateResponse bulkUpdateResponse=new BulkUpdateResponse();
            List<ScrollableHitSource.SearchFailure> searchFailures = response.getSearchFailures();
            List<BulkItemResponse.Failure> bulkFailures = response.getBulkFailures();
            if (isNotEmpty(searchFailures)) {
                ScrollableHitSource.SearchFailure first = searchFailures.getFirst();
                log.error(first.getStatus()+" , searchFailures: "+ first.getNodeId()+":"+first.getShardId()+":"+first.getIndex(),first);
            }
            for (BulkItemResponse.Failure failure : bulkFailures) {
                bulkUpdateResponse.fail(failure.getId());
            }
            return bulkUpdateResponse;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public BulkUpdateResponse updatePartOfDocInMultiIndex(Supplier<BoolQueryBuilder> querySupplier, Supplier<Script> scriptSupplier, String... alias) {
        UpdateByQueryRequest updateByQueryRequest=new UpdateByQueryRequest(alias);
        updateByQueryRequest.setQuery(querySupplier.get());
        updateByQueryRequest.setScript(scriptSupplier.get());
        updateByQueryRequest.setConflicts("proceed");
        try {
            BulkByScrollResponse response = elasticsearchClient.updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
            if (response.isTimedOut()) {
                throw new RuntimeException("timeout");
            }

            BulkUpdateResponse bulkUpdateResponse=new BulkUpdateResponse();
            List<ScrollableHitSource.SearchFailure> searchFailures = response.getSearchFailures();
            List<BulkItemResponse.Failure> bulkFailures = response.getBulkFailures();
            if (isNotEmpty(searchFailures)) {
                ScrollableHitSource.SearchFailure first = searchFailures.getFirst();
                log.error(first.getStatus()+" , searchFailures: "+ first.getNodeId()+":"+first.getShardId()+":"+first.getIndex(),first);
            }
            for (BulkItemResponse.Failure failure : bulkFailures) {
                bulkUpdateResponse.fail(failure.getId());
            }
            return bulkUpdateResponse;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }


    }

    @Override
    public BulkDeleteResponse deleteDocInMultiIndex(Supplier<BoolQueryBuilder> querySupplier,String... alias) {
        DeleteByQueryRequest deleteByQueryRequest=new DeleteByQueryRequest(alias);
        deleteByQueryRequest.setQuery(querySupplier.get());
        deleteByQueryRequest.setConflicts("proceed");
        try {
            BulkByScrollResponse response = elasticsearchClient.deleteByQuery(deleteByQueryRequest,RequestOptions.DEFAULT);
            if (response.isTimedOut()) {
                throw new RuntimeException("timeout");
            }

            BulkDeleteResponse bulkDeleteResponse=new BulkDeleteResponse();
            List<ScrollableHitSource.SearchFailure> searchFailures = response.getSearchFailures();
            List<BulkItemResponse.Failure> bulkFailures = response.getBulkFailures();
            if (isNotEmpty(searchFailures)) {
                ScrollableHitSource.SearchFailure first = searchFailures.getFirst();
                log.error(first.getStatus()+" , searchFailures: "+ first.getNodeId()+":"+first.getShardId()+":"+first.getIndex(),first);
            }
            for (BulkItemResponse.Failure failure : bulkFailures) {
                bulkDeleteResponse.fail(failure.getId());
            }
            return bulkDeleteResponse;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public BulkUpdateResponse updatePartOfDocInMultiIndex(UpdateByQueryRequest updateByQueryRequest) {
        try {
            updateByQueryRequest.setConflicts("proceed");
            BulkByScrollResponse response = elasticsearchClient.updateByQuery(updateByQueryRequest,RequestOptions.DEFAULT);
            if (response.isTimedOut()) {
                throw new RuntimeException("timeout");
            }
            BulkUpdateResponse bulkUpdateResponse=new BulkUpdateResponse();
            bulkUpdateResponse.setTotal(response.getTotal());
            bulkUpdateResponse.setUpdated(response.getUpdated());
            bulkUpdateResponse.setVersionConflicts(response.getVersionConflicts());

            List<ScrollableHitSource.SearchFailure> searchFailures = response.getSearchFailures();
            List<BulkItemResponse.Failure> bulkFailures = response.getBulkFailures();
            if (isNotEmpty(searchFailures) ) {
                ScrollableHitSource.SearchFailure first = searchFailures.getFirst();
                log.error(first.getStatus()+" , searchFailures: "+ first.getNodeId()+":"+first.getShardId()+":"+first.getIndex(),first);
            }
            for (BulkItemResponse.Failure failure : bulkFailures) {
                bulkUpdateResponse.fail(failure.getId());
            }
            return bulkUpdateResponse;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public BulkDeleteResponse bulkDeleteInMultiIndex(List<String> ids, String... alias) {

        DeleteByQueryRequest deleteByQueryRequest=new DeleteByQueryRequest(alias);
        deleteByQueryRequest.setQuery(QueryBuilders.termsQuery("_id",ids));
        deleteByQueryRequest.setConflicts("proceed");
        try {
            BulkByScrollResponse response = elasticsearchClient.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
            if (response.isTimedOut()) {
                throw new RuntimeException("timeout");
            }
            BulkDeleteResponse bulkDeleteResponse=new BulkDeleteResponse();
            List<ScrollableHitSource.SearchFailure> searchFailures = response.getSearchFailures();
            List<BulkItemResponse.Failure> bulkFailures = response.getBulkFailures();
            if (isNotEmpty(searchFailures) ) {
                ScrollableHitSource.SearchFailure first = searchFailures.getFirst();
                log.error(first.getStatus()+" , searchFailures: "+ first.getNodeId()+":"+first.getShardId()+":"+first.getIndex(),first);
            }
            if (isNotEmpty(bulkFailures)) {
                for (BulkItemResponse.Failure bulkFailure : bulkFailures) {
                    bulkDeleteResponse.fail(bulkFailure.getId());
                }
            }
            return bulkDeleteResponse;
        } catch (IOException e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }

    }

    @Override
    public <M extends Document<?>> M get(String index,String id, Class<M> model) {
        GetRequest getRequest=new GetRequest(index,id);
        try {
            GetResponse response = elasticsearchClient.get(getRequest, RequestOptions.DEFAULT);
            if (response.isExists()) {
                String source = response.getSourceAsString();
                M read = jsonAccessor.read(source, model);
                read.id(response.getId());
                return read;
            }
            return null;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public <M extends Document<?>> M getInMultiIndex(String id, Class<M> model, String... alias) {
        SearchRequest searchRequest=new SearchRequest(alias);
        SearchSourceBuilder searchSourceBuilder=new SearchSourceBuilder();
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.filter(QueryBuilders.termQuery("_id",id));
        searchSourceBuilder.query(boolQueryBuilder);
        searchRequest.source(searchSourceBuilder);
        List<M> list = list(searchRequest, model);
        return isEmpty(list)?null:list.getFirst();
    }

    @Override
    public <M extends Document<?>> List<M> list(SearchRequest searchRequest, Class<M> model) {

        try {
            SearchResponse searchResponse = elasticsearchClient.search(searchRequest, RequestOptions.DEFAULT);
            if (searchResponse.isTimedOut()) {
                throw new RuntimeException("timeout");
            }
            RestStatus status = searchResponse.status();
            if (status != RestStatus.OK) {
                throw new RuntimeException("error: "+status);
            }

            SearchHits hits = searchResponse.getHits();
            List<M> list=new ArrayList<>();
            for (SearchHit hit : hits) {
                String source = hit.getSourceAsString();
                M read = jsonAccessor.read(source, model);
                read.id(hit.getId());
                read.docIndexName(hit.getIndex());
                read.ver(hit.getVersion());
                read.seqNo(hit.getSeqNo());
                read.primaryTerm(hit.getPrimaryTerm());
                list.add(read);
            }
            return list;
        } catch (Exception e) {
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public <M extends Document<?>> PageModel<M> page(SearchRequest searchRequest, Class<M> model) {
        try {
            SearchResponse searchResponse = elasticsearchClient.search(searchRequest, RequestOptions.DEFAULT);
            if (searchResponse.isTimedOut()) {
                throw new RuntimeException("timeout");
            }
            RestStatus status = searchResponse.status();
            if (status != RestStatus.OK) {
                throw new RuntimeException("error: "+status);
            }

            SearchHits hits = searchResponse.getHits();
            List<M> list=new ArrayList<>();
            for (SearchHit hit : hits) {
                String source = hit.getSourceAsString();
                M read = jsonAccessor.read(source, model);
                read.id(hit.getId());
                read.docIndexName(hit.getIndex());
                read.ver(hit.getVersion());
                read.seqNo(hit.getSeqNo());
                read.primaryTerm(hit.getPrimaryTerm());
                list.add(read);
            }
            PageModel<M> pageModel=new PageModel<>();

            pageModel.setTotal(hits.getTotalHits().value);
            pageModel.setData(list);

            return pageModel;
        } catch (Exception e) {
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public long count(CountRequest countRequest) {
        try {
            CountResponse countResponse = elasticsearchClient.count(countRequest, RequestOptions.DEFAULT);
            RestStatus status = countResponse.status();
            if (status != RestStatus.OK) {
                throw new RuntimeException("error: "+status);
            }
            return countResponse.getCount();
        } catch (Exception e) {
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public SearchResponse search(SearchRequest searchRequest) {
        try {
            SearchResponse searchResponse = elasticsearchClient.search(searchRequest, RequestOptions.DEFAULT);
            if (searchResponse.isTimedOut()) {
                throw new RuntimeException("timeout");
            }
            RestStatus status = searchResponse.status();
            if (status != RestStatus.OK) {
                throw new RuntimeException("error: "+status);
            }

            return searchResponse;
        } catch (Exception e) {
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public BulkDeleteResponse bulkDelete(String index, List<String> ids) {

        BulkRequest bulkRequest = new BulkRequest();
        for (String id : ids) {
            bulkRequest.add(new DeleteRequest(index,id));
        }

        try {
            BulkDeleteResponse bulkDeleteResponse=new BulkDeleteResponse();
            BulkResponse bulkResponse = elasticsearchClient.bulk(bulkRequest, RequestOptions.DEFAULT);
            BulkItemResponse[] items = bulkResponse.getItems();
            for (BulkItemResponse item : items) {
                if (item.getFailure() != null) {
                    bulkDeleteResponse.fail(item.getId());
                }
            }
            return bulkDeleteResponse;
        } catch (Exception e) {
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }


    }

    @Override
    public List<AliasIndex> backingIndex(String alias) {
        List<AliasIndex> aliasIndexList=new ArrayList<>();
        try {
            GetAliasesRequest getAliasesRequest=new GetAliasesRequest();
            getAliasesRequest.aliases(alias);
            GetAliasesResponse aliasesResponse = elasticsearchClient.indices().getAlias(getAliasesRequest, RequestOptions.DEFAULT);
            Map<String, Set<AliasMetadata>> aliases = aliasesResponse.getAliases();
            for (Map.Entry<String, Set<AliasMetadata>> entry : aliases.entrySet()) {
                String index = entry.getKey();
                AliasIndex aliasIndex=new AliasIndex();
                aliasIndex.setIndex(index);
                for (AliasMetadata aliasMetadata : entry.getValue()) {
                    if (aliasMetadata.writeIndex()!=null && aliasMetadata.writeIndex()) {
                        aliasIndex.setWrite(true);
                    }
                }
                aliasIndexList.add(aliasIndex);
            }
            return aliasIndexList;
        }
        catch (Exception e){
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public SearchResponse scroll(SearchScrollRequest searchScrollRequest) {
        try {
            return elasticsearchClient.scroll(searchScrollRequest, RequestOptions.DEFAULT);
        }catch (Exception e){
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public ClearScrollResponse clearScroll(ClearScrollRequest clearScrollRequest) {
        try {
            return elasticsearchClient.clearScroll(clearScrollRequest, RequestOptions.DEFAULT);
        }catch (Exception e){
            log.error(e.getMessage(),e);
            throw convert2RuntimeException(e);
        }
    }

    @Override
    public String format(Object object) {
        return jsonAccessor.format(object);
    }

    @Override
    public Map<String, Object> read(String json) {
        return jsonAccessor.read(json);
    }

    @Override
    public Map<String, Object> convert(Object object) {
        return read(format(object));
    }
}
