package com.ovopark.module.shared.es7x;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalTimeSerializer;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.RestHighLevelClient;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import static com.ovopark.kernel.shared.Util.*;

@Configuration
public class ElasticsearchConfig {

    @Bean(value = "ES_JSONAccessor")
    public JSONAccessor esJSONAccessor(){
        ObjectMapper objectMapper=new ObjectMapper();
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addSerializer(LocalDateTime.class, new JsonSerializer<LocalDateTime>() {
            @Override
            public void serialize(LocalDateTime value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
                gen.writeNumber(value.toInstant(GMT_08).toEpochMilli());
            }
        });
        javaTimeModule.addSerializer(LocalDate.class,new LocalDateSerializer(DateTimeFormatter.ofPattern(YYYY_MM_DD)));
        javaTimeModule.addSerializer(LocalTime.class,new LocalTimeSerializer(DateTimeFormatter.ofPattern("HH:mm:ss")));
        javaTimeModule.addDeserializer(LocalDateTime.class, new JsonDeserializer<LocalDateTime>() {
            @Override
            public LocalDateTime deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JacksonException {
                long longValue = p.getLongValue();
                return dateTime(longValue);
            }
        });
        javaTimeModule.addDeserializer(LocalDate.class,new LocalDateDeserializer(DateTimeFormatter.ofPattern(YYYY_MM_DD)));
        javaTimeModule.addDeserializer(LocalTime.class,new LocalTimeDeserializer(DateTimeFormatter.ofPattern("HH:mm:ss")));
        objectMapper.registerModule(javaTimeModule).registerModule(new ParameterNamesModule());
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES,false);

        return new JSONAccessor() {
            @Override
            public String format(Object object) {
                try {
                    return objectMapper.writeValueAsString(object);
                } catch (JsonProcessingException e) {
                    throw Util.convert2RuntimeException(e);
                }
            }

            @Override
            public <T> T read(String text, Class<T> type) {
                try {
                    return objectMapper.readValue(text.getBytes(StandardCharsets.UTF_8),type);
                } catch (IOException e) {
                    throw Util.convert2RuntimeException(e);
                }
            }

            @Override
            public <T> T read(String text, TypeReference<T> typeReference) {
                try {
                    return objectMapper.readValue(text.getBytes(StandardCharsets.UTF_8), new com.fasterxml.jackson.core.type.TypeReference<T>() {
                        @Override
                        public Type getType() {
                            return typeReference.getType();
                        }
                    });
                } catch (IOException e) {
                    throw Util.convert2RuntimeException(e);
                }
            }

            @Override
            public Map read(String text) {
                return read(text, Map.class);
            }
        };
    }


    @Bean
    public RestHighLevelClient elasticsearchClient (ElasticsearchProperties elasticsearchProperties){

        List<String> uris = elasticsearchProperties.getUris();
        HttpHost[] httpHosts = uris.
                stream()
                .map(uri -> new HttpHost(uri.split(":")[0], Integer.parseInt(uri.split(":")[1])))
                .toList()
                .toArray(new HttpHost[]{});

        //创建 low-level client
        RestClientBuilder restClientBuilder = RestClient.builder(httpHosts)
                .setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
                    @Override
                    public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpClientBuilder) {
                        BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
                        credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(elasticsearchProperties.getUsername(), elasticsearchProperties.getPassword()));
                        httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);
                        return httpClientBuilder;
                    }
                });
        RestHighLevelClient restHighLevelClient=new RestHighLevelClient(restClientBuilder);

        return restHighLevelClient;

    }

    @Bean
    public ElasticsearchOperations elasticsearchOperations(){
        return new ElasticsearchOperationsImpl();
    }


    @Primary
    @Bean
    public SpringElasticsearchManager springElasticsearchManager(
            ElasticsearchOperations elasticsearchOperations
            ,List<ElasticsearchManager.AliasIndexRefreshListener> aliasIndexRefreshListenerList
    ){
        SimpleElasticsearchManager simpleElasticsearchManager = new SimpleElasticsearchManager(elasticsearchOperations, Comparator.naturalOrder(), aliasIndexRefreshListenerList);
        return new SpringElasticsearchManager(simpleElasticsearchManager);
    }

    @Bean
    public SpringILMElasticsearchManager springILMElasticsearchManager(
            ElasticsearchOperations elasticsearchOperations
            ,List<ElasticsearchManager.AliasIndexRefreshListener> aliasIndexRefreshListenerList
    ){
        SimpleILMElasticsearchManager simpleILMElasticsearchManager = new SimpleILMElasticsearchManager(elasticsearchOperations,aliasIndexRefreshListenerList);
        return new SpringILMElasticsearchManager(simpleILMElasticsearchManager);
    }


}
