package com.ovopark.module.shared.redis;

import com.ovopark.kernel.shared.kv.CacheService;
import org.redisson.api.RRateLimiter;
import org.redisson.api.RateLimiterConfig;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

@Component
public class RedissonRateLimiterClient implements RateLimiterClient{

    final static private CacheService<String,RateLimiter> RL=new CacheService.MapCacheService<>(true);

    @Autowired
    private RedissonClient redissonClient;

    @Override
    public RateLimiter get(String name) {
        return RL.putIfAbsentAndGet(name, new Supplier<RateLimiter>() {
            @Override
            public RateLimiter get() {
                RRateLimiter rateLimiter = redissonClient.getRateLimiter(name);
                RateLimiterConfig config = rateLimiter.getConfig();
                RateLimiterConf rateLimiterConf = new RateLimiterConf();
                rateLimiterConf.setRate(config.getRate());
                rateLimiterConf.setRateInterval(config.getRateInterval());
                rateLimiterConf.setTimeUnit(TimeUnit.MILLISECONDS);
                RedisRateLimiter redisRateLimiter = new RedisRateLimiter(name, rateLimiter,rateLimiterConf);
                return redisRateLimiter;
            }
        },15,TimeUnit.SECONDS);
    }

    @Override
    public RateLimiter getOrCreate(String name, int ratePerSec) {
        RateLimiterConf rateLimiterConf = new RateLimiterConf();
        rateLimiterConf.setRate(ratePerSec);
        rateLimiterConf.setRateInterval(1);
        rateLimiterConf.setTimeUnit(TimeUnit.SECONDS);
        return getOrCreate(name,rateLimiterConf);
    }


    @Override
    public RateLimiter getOrCreate(String name, RateLimiterConf rateLimiterConf) {
        return RL.putIfAbsentAndGet(name,()->{
            RRateLimiter rateLimiter = redissonClient.getRateLimiter(name);
            RateLimiterConfig config = rateLimiter.getConfig();
            if (config==null) {
                // config is not defined in REDIS, we crate a new
                RedisRateLimiter redisRateLimiter = new RedisRateLimiter(name, rateLimiter, rateLimiterConf);
                redisRateLimiter.setRate(rateLimiterConf.getRate()
                        , rateLimiterConf.getRateInterval()
                        , rateLimiterConf.getTimeUnit()
                );
                return redisRateLimiter;
            }
            else {
                // config is in REDIS , reuse
                RateLimiterConf rlc = new RateLimiterConf();
                rlc.setRate(config.getRate());
                rlc.setRateInterval(config.getRateInterval());
                rlc.setTimeUnit(TimeUnit.MILLISECONDS);
                return new RedisRateLimiter(name, rateLimiter,rlc);
            }
        },15, TimeUnit.SECONDS);
    }

    @Override
    public RateLimiter upset(String name, RateLimiterConf rateLimiterConf) {
        return RL.updateAndGet(name, old -> {
            RRateLimiter rateLimiter = redissonClient.getRateLimiter(name);
            RedisRateLimiter redisRateLimiter = new RedisRateLimiter(name, rateLimiter,rateLimiterConf);
            redisRateLimiter.setRate(rateLimiterConf.getRate()
                    , rateLimiterConf.getRateInterval()
                    , rateLimiterConf.getTimeUnit()
            );
            return redisRateLimiter;
        }, 15, TimeUnit.SECONDS);
    }

    @Override
    public RateLimiter upset(String name, int ratePerSec) {
        RateLimiterConf rateLimiterConf = new RateLimiterConf();
        rateLimiterConf.setRate(ratePerSec);
        rateLimiterConf.setRateInterval(1);
        rateLimiterConf.setTimeUnit(TimeUnit.SECONDS);
        return upset(name,rateLimiterConf);
    }

    @Override
    public RateLimiter setIfAbsent(String name, int ratePerSec) {
        RateLimiterConf rateLimiterConf = new RateLimiterConf();
        rateLimiterConf.setRate(ratePerSec);
        rateLimiterConf.setRateInterval(1);
        rateLimiterConf.setTimeUnit(TimeUnit.SECONDS);
        return RL.updateAndGet(name, old -> {
            RRateLimiter rateLimiter = redissonClient.getRateLimiter(name);
            RedisRateLimiter redisRateLimiter = new RedisRateLimiter(name, rateLimiter,rateLimiterConf);
            boolean f = redisRateLimiter.trySetRate(rateLimiterConf.getRate()
                    , rateLimiterConf.getRateInterval()
                    , rateLimiterConf.getTimeUnit()
            );
            if (!f) {
                return get(name);
            }
            return redisRateLimiter;
        }, 15, TimeUnit.SECONDS);
    }
}
