package com.ovopark.module.shared.spring.rbac;

import com.ovopark.module.shared.Session;
import com.ovopark.module.shared.spring.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.AntPathMatcher;

import static com.ovopark.kernel.shared.Util.convert2RuntimeException;

@Slf4j
public class ResourceFilter<RQ, RS> implements SharedFilter<RQ, RS> {


    final LANResource lanResource;

    final LANAccessHandler lanAccessHandler;

    final SessionClient  sessionClient;

    final TokenHandler tokenHandler;

    public ResourceFilter(LANResource lanResource, LANAccessHandler lanAccessHandler, SessionClient sessionClient,TokenHandler tokenHandler) {
        this.lanResource = lanResource;
        this.lanAccessHandler = lanAccessHandler;
        this.sessionClient = sessionClient;
        this.tokenHandler=tokenHandler;
    }

    @Override
    public void doFilter(SharedRequest<RQ> request, SharedResponse<RS> response, SharedFilterChain<RQ, RS> filterChain) {


        if (lanResource.off()) {
            filterChain.doFilter(request,response);
            return;
        }
        if (ResourceAccess.getOrCreate().isFromWAN(request) && lanResource.lanUrl()!=null) {
            // filter ...
            AntPathMatcher antPathMatcher=new AntPathMatcher();
            for (String url : lanResource.lanUrl()) {
                String servletPath = request.getServletPath();
                if (antPathMatcher.match(url, servletPath)) {
                    //check token
                    Session session ;
                    try {
                        session = sessionClient.session(request);
                    } catch (Exception e) {

                        if (e instanceof TokenInvalidException) {
                            tokenHandler.handleInvalid(request,response);
                            return;
                        }

                        if (e instanceof TokenRefreshException) {
                            tokenHandler.handleRefresh(request,response);
                            return;
                        }

                        if (e instanceof TokenRefreshExpiredException) {
                            tokenHandler.handleRefreshExpired(request,response);
                            return;
                        }

                        throw convert2RuntimeException(e);
                    }
                    if (session ==null) {
                        // token is missing , no privilege
                        log.info("request from wan/internet cannot access lan service: "+servletPath+", pattern: "+url);
                        LANAccessHandler.Result result = lanAccessHandler.handle(request, response);
                        if (result== LANAccessHandler.Result.STOP) {
                            return;
                        }

                        if (result== LANAccessHandler.Result.CONTINUE) {
                            filterChain.doFilter(request,response);
                            return;
                        }
                    }
                    else {
                        break;
                    }
                }
            }
        }
        filterChain.doFilter(request,response);
    }
}
