package com.ovopark.module.shared.spring.rbac;

import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.kernel.shared.Util;
import com.ovopark.module.shared.BaseResult;
import com.ovopark.module.shared.Session;
import com.ovopark.module.shared.spring.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.AntPathMatcher;

import java.io.IOException;

import static com.ovopark.kernel.shared.Util.convert2RuntimeException;
import static com.ovopark.module.shared.spring.rbac.SessionClientImpl.cacheStat;

@Slf4j
public class SessionFilter<RQ, RS> implements SharedFilter<RQ, RS> {

    final SessionService.SessionServiceProvider serviceProvider;

    final JSONAccessor jsonAccessor;

    final TokenHandler tokenHandler;

    final SessionClientResource sessionClientResource;

    public SessionFilter(SessionService.SessionServiceProvider serviceProvider
            , JSONAccessor jsonAccessor,TokenHandler tokenHandler,SessionClientResource sessionClientResource) {
        this.serviceProvider = serviceProvider;
        this.jsonAccessor = jsonAccessor;
        this.tokenHandler=tokenHandler;
        this.sessionClientResource=sessionClientResource;
    }

    @Override
    public void doFilter(SharedRequest<RQ> request, SharedResponse<RS> response, SharedFilterChain<RQ, RS> filterChain) {

        if (sessionClientResource.off()) {
            filterChain.doFilter(request,response);
            return;
        }

        if (sessionClientResource.excludeUrl()!=null) {
            // filter ...
            AntPathMatcher antPathMatcher=new AntPathMatcher();
            for (String url : sessionClientResource.excludeUrl()) {
                String servletPath = request.getServletPath();
                if (antPathMatcher.match(url, servletPath)) {
                    //check url
                    filterChain.doFilter(request,response);
                    return;
                }
            }
        }

        try {
            Session session;
            try {
                session=ResourceAccess.getOrCreate().session(request,jsonAccessor);
                if (session != null) {
                    long l = cacheStat.hit();
                    if (l %100==0) {
                        InfoCollector.getOrCreate().put("SESSION_CLIENT_GATEWAY_TOKEN_HIT_COUNT",String.valueOf(l));
                    }
                }

                if (session==null) {

                    long l = cacheStat.miss();
                    if (l %100==0) {
                        InfoCollector.getOrCreate().put("SESSION_CLIENT_GATEWAY_TOKEN_MISS_COUNT",String.valueOf(l));
                    }

                    try {
                        session = ResourceAccess.getOrCreate().session(request,serviceProvider);
                    } catch (Exception e) {

                        if (e instanceof TokenInvalidException) {
                            tokenHandler.handleInvalid(request,response);
                            return;
                        }

                        if (e instanceof TokenRefreshException) {
                            tokenHandler.handleRefresh(request,response);
                            return;
                        }

                        if (e instanceof TokenRefreshExpiredException) {
                            tokenHandler.handleRefreshExpired(request,response);
                            return;
                        }

                        throw e;
                    }
                }
            } catch (Exception e) {
                response.setContentType("application/json;charset=utf-8");
                try {
                    response.getOutputStream().write(Util.utf8(jsonAccessor.format(BaseResult.invalidToken())));
                } catch (IOException ex) {
                    throw new RuntimeException(ex);
                }
                return;
            }
            Session.getOrCreate().set(session);
            filterChain.doFilter(request,response);
        }
        finally {
            Session.getOrCreate().remove();
        }

    }
}
