package com.ovopark.module.shared.spring;

import com.ovopark.kernel.shared.JSONAccessor;
import com.ovopark.module.shared.Session;
import com.ovopark.module.shared.SharedLocale;
import com.ovopark.module.shared.spring.rbac.*;
import lombok.extern.slf4j.Slf4j;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Locale;
import java.util.TimeZone;

import static com.ovopark.kernel.shared.Util.*;

@Slf4j
final public class ResourceAccess {

    private static final ResourceAccess INSTANCE=new ResourceAccess();

    private ResourceAccess(){}

    public static ResourceAccess getOrCreate(){
        return INSTANCE;
    }

    public boolean isFromWAN(SharedRequest<?> request){
        String header = request.getHeader(Cons.REQUEST_HEAD_GATEWAY_SOURCE_KEY);
        if (log.isDebugEnabled()) {
            log.info("OK , we found gateway route head: "+header);
        }
        return isNotEmpty(header);
    }

    public Session session(SharedRequest<?> request, JSONAccessor jsonAccessor){
        return session(request,jsonAccessor::read);
    }

    public Session session(SharedRequest<?> request, SessionDeserializer sessionDeserializer){

        String str = request.getHeader(Cons.REQUEST_HEAD_GATEWAY_OVO_AUTHORIZATION_KEY_UTF8);
        if (isNotEmpty(str)) {
            if (log.isDebugEnabled()) {
                log.info("OK , we found gateway token, to deserialize value: " + str);
            }
            try {
                str = URLDecoder.decode(str, "utf8");
            } catch (UnsupportedEncodingException e) {
                throw convert2RuntimeException(e);
            }
            return sessionDeserializer.read(str, SessionImpl.class);
        }
        else {
            str = request.getHeader(Cons.REQUEST_HEAD_GATEWAY_OVO_AUTHORIZATION_KEY);
            if (isNotEmpty(str)) {
                if (log.isDebugEnabled()) {
                    log.info("OK , we found gateway token, to deserialize value: " + str);
                }
                return sessionDeserializer.read(str, SessionImpl.class);
            }
        }
        return null;
    }

    public interface SessionDeserializer{

        <T extends Session> Session read(String str, Class<T> clazz);

    }

    /**
     * may throw {@link TokenRefreshException} , {@link TokenInvalidException}, {@link TokenRefreshExpiredException}
     * @param request
     * @param serviceProvider
     * @return
     */
    public Session session(SharedRequest<?> request, SessionService.SessionServiceProvider serviceProvider){

        String newAuthenticator = request.getHeader("Ovo-Authorization");
        String oldAuthenticator = request.getHeader("authenticator");
        String channel = request.getHeader("Ovo-Channel");
        // mcp
        String mcpAuthenticator = request.getHeader("Ovo-Access-Token");

        if (isEmpty(newAuthenticator) && isEmpty(oldAuthenticator) && isEmpty(mcpAuthenticator) ) {
            if (log.isDebugEnabled()) {
                log.info("OK , cannot found any token in the head.");
            }
            return null;
        }

        ClientInfo clientInfo=new ClientInfo();
        clientInfo.setRemoteIp(remoteIp(request));
        clientInfo.setPlatform(platform(request));
        clientInfo.setChannel(channel);

        UserInfo user=new UserInfo();
        SessionImpl session = new SessionImpl();
        session.setClientInfo(clientInfo);
        session.setUserInfo(user);


        SessionService sessionService = serviceProvider.get();

        if (isNotEmpty(newAuthenticator) || isNotEmpty(mcpAuthenticator)) {

            if (log.isDebugEnabled()) {
                log.info("OK , found Ovo-Authorization | Ovo-Access-Token token in the head: "+newAuthenticator+" | "+mcpAuthenticator);
            }
            //parse new token
            SessionToken tokenValue=null;
            String[] auths=null;
            String token=null;
            if (isNotEmpty(newAuthenticator)) {
                auths = newAuthenticator.split(" ");
                token = auths[0];
                tokenValue = sessionService.parseNewToken(token);
            }
            else if(isNotEmpty(mcpAuthenticator)){
                auths = mcpAuthenticator.split(" ");
                token = auths[0];
                tokenValue = sessionService.parseAccessToken(token);
            }
            if (tokenValue==null) {
                throw new IllegalArgumentException("token is missing, mcp("+isNotEmpty(mcpAuthenticator)+") ???: "+token);
            }

            if (auths.length > 1) {
                clientInfo.setClient(auths[1]);
            }
            if (auths.length > 2) {
                clientInfo.setVersion(auths[2]);
            }

            if (auths.length > 3) {
                clientInfo.setLang(auths[3]);
                if (SharedLocale.LANG_ENGLISH.equalsIgnoreCase(auths[3])) {//英文
                    clientInfo.setLocale(Locale.US);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                } else if (SharedLocale.LANG_TRADITIONAL_CHINESE.equalsIgnoreCase(auths[3])) {//繁体中文
                    clientInfo.setLocale(Locale.TRADITIONAL_CHINESE);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                } else if (SharedLocale.LANG_INDONESIA.equalsIgnoreCase(auths[3])) {//印尼
                    clientInfo.setLocale(SharedLocale.IN_ID);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                }else if (SharedLocale.LANG_JAPANESE.equalsIgnoreCase(auths[3])) {//印尼
                    clientInfo.setLocale(Locale.JAPAN);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                }else if (SharedLocale.LANG_ITALIAN.equalsIgnoreCase(auths[3])) {//印尼
                    clientInfo.setLocale(Locale.ITALY);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                }
                else if (SharedLocale.LANG_THAI.equalsIgnoreCase(auths[3])) {//印尼
                    clientInfo.setLocale(SharedLocale.TH_TH);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                }
                else if (SharedLocale.LANG_VIETNAMESE.equalsIgnoreCase(auths[3])) {//印尼
                    clientInfo.setLocale(SharedLocale.VI_VN);
                    clientInfo.setLocaleString(clientInfo.getLocale().toString());
                }

            }
            if (auths.length > 4) {
                try {
                    clientInfo.setTimeZone(TimeZone.getTimeZone(auths[4]));
                } catch (Exception e) {
                    log.info("--------------TimeZone translate faied, timeZone:" + auths[4]);
                }
            }
            if (auths.length > 5) {
                clientInfo.setDeviceName(auths[5]);
            }
            if (auths.length > 6) {
                clientInfo.setDeviceSerialNo(auths[6]);
            }
            user.setUserId(tokenValue.getUserId());
            user.setUserName(tokenValue.getUserName());
            user.setGroupId(tokenValue.getGroupId());
            user.setIsAgency(tokenValue.getIsAgency());
            user.setIsAccountUser(tokenValue.getIsAccountUser());

            session.setBackendToken(newAuthenticator);
        }

        else if(isNotEmpty(oldAuthenticator)) {
            if (log.isDebugEnabled()) {
                log.info("OK , found authenticator token in the head: "+oldAuthenticator);
            }
            String userIdStr = TokenUtils.getUserId(oldAuthenticator);

            if (isEmpty(userIdStr)) {
                throw new TokenInvalidException("token is invalid ???: "+oldAuthenticator);
            }

            if (!isEmpty(userIdStr)) {
                SessionUsersPojo users = sessionService.getUsersById(Integer.parseInt(userIdStr));
                if (users != null) {
                    user.setUserId(users.getId());
                    user.setUserName(users.getUserName());
                    user.setGroupId(users.getGroupId());
                    user.setIsAgency(users.getIsAgency());
                    user.setIsAccountUser(0);
                }
            }
            session.setBackendToken(oldAuthenticator);
        }
        return session;

    }

    public String platform(SharedRequest<?> request) {
        String userAgent = request.getHeader("User-Agent");
//		System.out.println("UA:===========" + userAgent);
        String platform = null;
        if (userAgent != null) {
            // 安卓使用okhttp框架
            if (userAgent.contains("Android") || userAgent.contains("okhttp")) {
                platform = "Android";
            } else if (userAgent.contains("iOS") || userAgent.contains("iPhone")) {
                platform = "iOS";
            } else {
                // 获取客户端浏览器
                platform = "web";
				/*StringTokenizer st = new StringTokenizer(userAgent, ";");
				st.nextToken();
				platform = st.nextToken();*/
            }
        } else {
            //  没有默认UA，是自定义客户端传来的，需定义各个客户端的标记
            platform = "unknown";
        }
        return platform;
    }


    public String remoteIp(SharedRequest<?> request) {
        String realIp = request.getHeader("X-Real-IP");
        String remoteIp = request.getHeader("x-forwarded-for"); // nginx反向代理
        if (isNotEmpty(realIp)) {
//            log.info("X-Real-IP:" + realIp);
//                        return realIp;
        }
        if (isNotEmpty(remoteIp)) {
//            log.info("x-forwarded-for:" + remoteIp);
            // 截取第一个
            // TODO 这里无法防止伪造x-forwarded-for，要防止伪造需要配置第一台nginx的ip。取ip前面的一个地址
            if (remoteIp.contains(",")) {
                remoteIp = remoteIp.split(",")[0].trim();
            }
            return remoteIp;
        } else {
            remoteIp = request.getHeader("x-forwarded-for");// apache反射代理
            if (isNotEmpty(remoteIp)) {
                String[] ips = remoteIp.split(",");
                for (String ip : ips) {
                    if (!"null".equalsIgnoreCase(ip)) {
                        return ip;
                    }
                }
            }
            return request.getRemoteAddr();
        }
    }
}
