package com.ovopark.module.shared.spring;

import com.ovopark.kernel.shared.Model;
import com.ovopark.kernel.shared.Ver;
import com.ovopark.kernel.shared.kv.KVEngine;
import com.ovopark.kernel.shared.vfile.LemonEngine;
import com.ovopark.module.shared.BaseResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.*;
import java.util.function.Function;
import java.util.regex.Pattern;

import static com.ovopark.kernel.shared.Util.*;

@Slf4j
@RestController("com.ovopark.module.shared.spring.EngineEndpoint")
@RequestMapping("/module-shared/engine")
public class EngineEndpoint {


    final static KVEngine.TtlFunc<String> ttlFunc=KVEngine.newTtl(EngineEndpoint.class.getName());

    @Data
    static class Context{

        final String contextId;

        public Context(String contextId) {
            this.contextId = contextId;
        }

        String nextStartKey;

    }


    final ExecutorService executorService= Executors.newFixedThreadPool(16,newThreadFactory("lemon-engine-seek"));

    private <T> T submitAndGet0(Callable<T> callable, Function<Exception,T> function, int timeout, TimeUnit timeUnit){
        try {
            Future<T> future = executorService.submit(callable);
            return future.get(timeout,timeUnit);
        } catch (Exception e) {
            return function.apply(e);
        }
    }

    @GetMapping("/next")
    @ResponseBody
    public BaseResult<ListRes> next(
            final @RequestParam(value = "contextId") String contextId
            , final @RequestParam(value = "engine") String engine
            , final @RequestParam(value = "initStartKey",required = false) String initStartKey
            , final @RequestParam(value = "num",required = false,defaultValue = "10") int num
            , final @RequestParam(value = "reversed",required = false) boolean reversed
            , final @RequestParam(value = "regex",required = false) String regex
            , final @RequestParam(value = "getSumIf",required = false) boolean getSumIf
    ){
//        Session session = Session.getOrCreate().get();
//        if (session==null) {
//            return BaseResult.noPrivilege();
//        }

        return submitAndGet0(new Callable<BaseResult<ListRes>>() {
            @Override
            public BaseResult<ListRes> call() throws Exception {
                Pattern pattern;
                if (isNotEmpty(regex)) {
                    pattern = Pattern.compile(regex);
                } else {
                    pattern = null;
                }

                LemonEngine lemonEngine = LemonEngine.mgr().get(engine);

                if (lemonEngine == null) {
                    return BaseResult.error(null, "cannot find engine: " + engine);
                }

                KVEngine.PutResult<String, Context> putResult = ttlFunc.ttlAndGet(contextId, new KVEngine.Upset<String, Context, Context>() {
                    @Override
                    public Context apply(KVEngine.GetResult<String, Context> getResult) {
                        if (getResult.exists()) {
                            return getResult.value();
                        }
                        Context c = new Context(contextId);
                        c.setNextStartKey(initStartKey);
                        return c;
                    }
                }, 60, TimeUnit.SECONDS);

                Context context = putResult.value();

                ListRes res = lock(EngineEndpoint.class.getName() + ":contextId:" + contextId, new Callable<ListRes>() {
                    @Override
                    public ListRes call() throws Exception {

                        ListRes listRes = new ListRes();
                        listRes.setContextCreated(putResult.created());
                        String nextStartKey = context.nextStartKey;
                        listRes.setStartKey(nextStartKey);

                        final int n = Math.min(num, 1000);
                        List<LemonEngine.GetResult> resultList;
                        if (reversed) {
                            resultList = lemonEngine.searchBefore(nextStartKey, false, n, k -> pattern == null || pattern.matcher(k).matches());
                        } else {
                            resultList = lemonEngine.searchAfter(nextStartKey, false, n, k -> pattern == null || pattern.matcher(k).matches());
                        }

                        if (getSumIf) {
                            listRes.setSum(lemonEngine.count());
                        } else {
                            listRes.setSum(-1);
                        }

                        if (isEmpty(resultList)) {
                            context.setNextStartKey(null);
                            return listRes;
                        }
                        String next = null;
                        List<Map<String, Map<String, Object>>> list = new ArrayList<>();
                        for (LemonEngine.GetResult getResult : resultList) {
                            Map<String, Map<String, Object>> data = new HashMap<>();
                            Map map = getResult.value();
                            data.put(getResult.key(), map);
                            list.add(data);
                            next = getResult.key();
                        }
                        context.setNextStartKey(next);
                        listRes.setList(list);
                        return listRes;
                    }
                });

                return BaseResult.success(res);
            }
        },t->BaseResult.error(null,t.getMessage()),15,TimeUnit.SECONDS);
    }


    @RequestMapping("/nextForUI")
    @ResponseBody
    public BaseResult<ListRow> nextForUI(
            final @RequestParam(value = "engine") String engine
            , final @RequestParam(value = "initStartKey",required = false) String initStartKey
            , final @RequestParam(value = "num",required = false,defaultValue = "10") int num
            , final @RequestParam(value = "reversed",required = false) boolean reversed
            , final @RequestParam(value = "regex",required = false) String regex
            , final @RequestParam(value = "getSumIf",required = false) boolean getSumIf
    ){
//        Session session = Session.getOrCreate().get();
//        if (session==null) {
//            return BaseResult.noPrivilege();
//        }

        return submitAndGet0(new Callable<BaseResult<ListRow>>() {
            @Override
            public BaseResult<ListRow> call() throws Exception {
                Pattern pattern;
                if (isNotEmpty(regex)) {
                    pattern = Pattern.compile(regex);
                } else {
                    pattern = null;
                }

                LemonEngine lemonEngine = LemonEngine.mgr().get(engine);

                if (lemonEngine == null) {
                    return BaseResult.error(null, "cannot find engine: " + engine);
                }

                ListRow listRes = new ListRow();
                listRes.setInitStartKey(initStartKey);

                final int n = Math.min(num, 1000);
                List<LemonEngine.GetResult> resultList;
                if (reversed) {
                    resultList = lemonEngine.searchBefore(initStartKey, false, n, k -> pattern == null || pattern.matcher(k).matches());
                } else {
                    resultList = lemonEngine.searchAfter(initStartKey, false, n, k -> pattern == null || pattern.matcher(k).matches());
                }

                if (getSumIf) {
                    listRes.setSum(lemonEngine.count());
                } else {
                    listRes.setSum(-1);
                }

                if (isEmpty(resultList)) {
                    return BaseResult.success(null);
                }
                String next = null;
                List<Row> rowList=new ArrayList<>();
                for (LemonEngine.GetResult getResult : resultList) {
                    Map map = getResult.value();
                    Row row=new Row();
                    row.setKey(getResult.key());
                    row.setValue(map);
                    rowList.add(row);
                    next = getResult.key();
                }
                listRes.setList(rowList);
                listRes.setInitStartKey(next);
                return BaseResult.success(listRes);
            }
        },t->BaseResult.error(null,t.getMessage()),15,TimeUnit.SECONDS);
    }

    @GetMapping("/get")
    @ResponseBody
    public BaseResult<GetRes> get(
            final @RequestParam(value = "engine") String engine
            , final @RequestParam(value = "key") String key){
//        Session session = Session.getOrCreate().get();
//        if (session==null) {
//            return BaseResult.noPrivilege();
//        }
        return submitAndGet0(new Callable<BaseResult<GetRes>>() {
            @Override
            public BaseResult<GetRes> call() throws Exception {
                LemonEngine lemonEngine = LemonEngine.mgr().get(engine);
                if (lemonEngine==null) {
                    return BaseResult.error(null,"cannot find engine: "+engine);
                }
                GetRes getRes =new GetRes();
                LemonEngine.GetResult getResult = lemonEngine.get(key);
                if (!getResult.exists()) {
                    getRes.setExists(false);
                    return BaseResult.success(getRes);
                }
                getRes.setExists(true);
                getRes.setData(getResult.value());
                return BaseResult.success(getRes);
            }
        },t->BaseResult.error(null,t.getMessage()),15,TimeUnit.SECONDS);

    }

    @GetMapping("/listEngine")
    @ResponseBody
    public BaseResult<List<String>> listEngine(){
//        Session session = Session.getOrCreate().get();
//        if (session==null) {
//            return BaseResult.noPrivilege();
//        }
        return BaseResult.success(LemonEngine.mgr().list());
    }


    @GetMapping("/disk")
    public String disk(final @RequestParam(value = "engine") String engine) throws Exception{
        LemonEngine lemonEngine = LemonEngine.mgr().get(engine);
        if (lemonEngine==null) {
            return "cannot find engine: "+engine;
        }
        String result= formatTime(LocalDateTime.now())+" , ver: "+ Ver.main+"."+Ver.min;
        result+= "\r\n"+lemonEngine.disk();
        return result;
    }

    @Data
    public static class ListRes implements Model {

        boolean contextCreated;

        private String startKey;

        int sum;

        List<Map<String, Map<String, Object>>> list;


    }

    @Data
    public static class GetRes implements Model {

        private boolean exists;

        Map<String, Object> data;


    }

    @Data
    public static class Row implements Model{

        private String key;

        private Object value;


    }

    @Data
    public static class ListRow implements Model {

        private String initStartKey;

        int sum;

        List<Row> list;


    }

    /**
     * 简单的冒泡排序：对传入列表做原地排序，使用短路避免多余遍历。
     */
    public static <T extends Comparable<? super T>> void bubbleSort(List<T> list) {
        if (list == null || list.size() < 2) {
            return;
        }
        int n = list.size();
        for (int i = 0; i < n - 1; i++) {
            boolean swapped = false;
            for (int j = 0; j < n - i - 1; j++) {
                if (list.get(j).compareTo(list.get(j + 1)) > 0) {
                    Collections.swap(list, j, j + 1);
                    swapped = true;
                }
            }
            if (!swapped) {
                break;
            }
        }
    }

}
