# 多数据源配置

## 背景

在实际项目中,我们经常需要访问多个数据源:

- 主从数据库
- 多个业务库
- 不同类型的数据库

如何优雅地管理多个数据源成为一个重要问题。

## 实现方案

### 1. 动态数据源

- 基于 AbstractRoutingDataSource 实现数据源路由
- 通过 ThreadLocal 保存当前数据源
- 使用注解切换数据源

### 2. 多数据源配置

- 独立配置每个数据源

## 示例代码

```java
public interface DataSourceConst {
    // 主库
    String Master = "master";
    // 从库
    String Slave = "slave";
    // 统计库
    String Stat = "stat";
}
```

```java
@Slf4j
@Configuration
public class DataSourceConfiguration {

    @Bean(DataSourceConst.Master)
    @ConfigurationProperties("spring.datasource.training.training-master")
    public DataSource master() {
        return DruidDataSourceBuilder.create().build();
    }

    @Bean(DataSourceConst.Slave)
    @ConfigurationProperties("spring.datasource.training.training-slave")
    public DataSource slave() {
        return DruidDataSourceBuilder.create().build();
    }

    @Bean(DataSourceConst.Stat)
    @ConfigurationProperties("spring.datasource.training.training-stat")
    public DataSource stat() {
        return DruidDataSourceBuilder.create().build();
    }
}
```

```properties

# 路由数据源是否启用 默认启用
routing-datasource.enable=true
# 默认数据源 不指定默认为'master'
routing-datasource.default=master

spring.datasource.training.training-master.url=jdbc:mysql://localhost:3306/training-master
spring.datasource.training.training-master.username=root
spring.datasource.training.training-master.password=root

spring.datasource.training.training-slave.url=jdbc:mysql://localhost:3306/training-slave
spring.datasource.training.training-slave.username=root
spring.datasource.training.training-slave.password=root

spring.datasource.training.training-stat.url=jdbc:mysql://localhost:3306/training-stat
spring.datasource.training.training-stat.username=root
spring.datasource.training.training-stat.password=root
```

```java
@Repository
public interface TrainingUserViewTimeMaterialMapper extends BaseMapper<TrainingUserViewTimeMaterial> {

    @DataSourceRouting(DataSourceConst.Stat)
    List<TrainingUserViewTimeMaterial> getMaterialStudyByPage(
            @Param("page") Page<MaterialStudyVo> page, @Param("enterpriseId") Integer enterpriseId,
            @Param("materialId") Integer materialId, @Param("userIdList") List<Integer> userIdList,
            @Param("status") Integer status
    );

    @DataSourceRouting(DataSourceConst.Master)
    int insert(TrainingUserViewTimeMaterial record);

    @DataSourceRouting(DataSourceConst.Slave)
    int updateByPrimaryKey(TrainingUserViewTimeMaterial record);
}
```


默认情况下,所有mapper都为slave, 如果特定写入方法需要切换到master, 则需要使用@DataSourceRouting(DataSourceConst.Master)注解标记该方法。
```java
@Repository
@DataSourceRouting(DataSourceConst.Slave)
public class TrainingUserViewTimeMaterialService {

    @DataSourceRouting(DataSourceConst.Master)
    int insert(TrainingUserViewTimeMaterial record);

    /**
     * 无需注解 默认跟随类注解
     */
    List<TrainingUserViewTimeMaterial> getList(int enterpriseId);
}
```

本示例演示了:

- 动态数据源的实现
- 多数据源配置方法
