/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.uri;

import io.helidon.common.uri.UriValidationException;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class UriValidator {
    private static final Pattern IP_V4_PATTERN;
    private static final boolean[] HEXDIGIT;
    private static final boolean[] UNRESERVED;
    private static final boolean[] SUB_DELIMS;
    private static final boolean[] PRINTABLE;

    private UriValidator() {
    }

    public static void validateScheme(String scheme) {
        if ("http".equals(scheme)) {
            return;
        }
        if ("https".equals(scheme)) {
            return;
        }
        char[] chars = scheme.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            char c = chars[i];
            UriValidator.validateAscii(UriValidationException.Segment.SCHEME, chars, i, c);
            if (Character.isLetterOrDigit(c) || c == '+' || c == '-' || c == '.') continue;
            UriValidator.failInvalidChar(UriValidationException.Segment.SCHEME, chars, i, c);
        }
    }

    public static void validateQuery(String rawQuery) {
        Objects.requireNonNull(rawQuery);
        if (rawQuery.isEmpty()) {
            return;
        }
        char[] chars = rawQuery.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            char c = chars[i];
            UriValidator.validateAscii(UriValidationException.Segment.QUERY, chars, i, c);
            if (UNRESERVED[c] || SUB_DELIMS[c] || c == '@' || c == '/' || c == '?') continue;
            if (c == '%') {
                UriValidator.validatePercentEncoding(UriValidationException.Segment.QUERY, rawQuery, chars, i);
                i += 2;
                continue;
            }
            UriValidator.failInvalidChar(UriValidationException.Segment.QUERY, chars, i, c);
        }
    }

    public static void validateHost(String host) {
        Objects.requireNonNull(host);
        if (host.indexOf(91) == 0 && host.indexOf(93) == host.length() - 1) {
            UriValidator.validateIpLiteral(host);
        } else {
            UriValidator.validateNonIpLiteral(host);
        }
    }

    public static void validateIpLiteral(String ipLiteral) {
        Objects.requireNonNull(ipLiteral);
        UriValidator.checkNotBlank(UriValidationException.Segment.HOST, "IP Literal", ipLiteral, ipLiteral);
        if (ipLiteral.charAt(0) != '[') {
            throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), "Invalid IP literal, missing square bracket(s)", 0, ipLiteral.charAt(0));
        }
        int lastIndex = ipLiteral.length() - 1;
        if (ipLiteral.charAt(lastIndex) != ']') {
            throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), "Invalid IP literal, missing square bracket(s)", lastIndex, ipLiteral.charAt(lastIndex));
        }
        String host = ipLiteral.substring(1, ipLiteral.length() - 1);
        UriValidator.checkNotBlank(UriValidationException.Segment.HOST, "Host", ipLiteral, host);
        if (host.charAt(0) == 'v') {
            UriValidator.validateIpFuture(ipLiteral, host);
            return;
        }
        if (host.equals("::")) {
            return;
        }
        if (host.equals("::1")) {
            return;
        }
        boolean skipped = false;
        int segments = 0;
        String inProgress = host;
        while (!inProgress.isEmpty()) {
            if (inProgress.length() == 1) {
                ++segments;
                UriValidator.validateH16(ipLiteral, inProgress);
                break;
            }
            if (inProgress.charAt(0) == ':' && inProgress.charAt(1) == ':') {
                if (skipped) {
                    throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), "Host IPv6 contains more than one skipped segment");
                }
                skipped = true;
                ++segments;
                inProgress = inProgress.substring(2);
                continue;
            }
            if (inProgress.charAt(0) == ':') {
                throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), inProgress.toCharArray(), "Host IPv6 contains excessive colon");
            }
            int nextColon = inProgress.indexOf(58);
            if (nextColon == -1) {
                if (inProgress.indexOf(46) == -1) {
                    ++segments;
                    UriValidator.validateH16(ipLiteral, inProgress);
                    break;
                }
                Matcher matcher = IP_V4_PATTERN.matcher(inProgress);
                if (matcher.matches()) {
                    UriValidator.validateIpOctet("Host IPv6 dual address contains invalid IPv4 address", ipLiteral, matcher.group(1));
                    UriValidator.validateIpOctet("Host IPv6 dual address contains invalid IPv4 address", ipLiteral, matcher.group(2));
                    UriValidator.validateIpOctet("Host IPv6 dual address contains invalid IPv4 address", ipLiteral, matcher.group(3));
                    UriValidator.validateIpOctet("Host IPv6 dual address contains invalid IPv4 address", ipLiteral, matcher.group(4));
                    break;
                }
                throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), "Host IPv6 dual address contains invalid IPv4 address");
            }
            UriValidator.validateH16(ipLiteral, inProgress.substring(0, nextColon));
            ++segments;
            if (inProgress.length() >= nextColon + 2 && inProgress.charAt(nextColon + 1) == ':') {
                inProgress = inProgress.substring(nextColon);
                continue;
            }
            if (!(inProgress = inProgress.substring(nextColon + 1)).isBlank()) continue;
            UriValidator.validateH16(ipLiteral, inProgress);
        }
        if (segments > 8) {
            throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), "Host IPv6 address contains too many segments");
        }
    }

    public static void validateNonIpLiteral(String host) {
        Objects.requireNonNull(host);
        UriValidator.checkNotBlank(UriValidationException.Segment.HOST, "Host", host, host);
        Matcher matcher = IP_V4_PATTERN.matcher(host);
        if (matcher.matches()) {
            return;
        }
        char[] chars = host.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            char c = chars[i];
            UriValidator.validateAscii(UriValidationException.Segment.HOST, chars, i, c);
            if (UNRESERVED[c] || SUB_DELIMS[c]) continue;
            if (c == '%') {
                UriValidator.validatePercentEncoding(UriValidationException.Segment.HOST, host, chars, i);
                i += 2;
                continue;
            }
            UriValidator.failInvalidChar(UriValidationException.Segment.HOST, chars, i, c);
        }
    }

    public static void validateFragment(String rawFragment) {
        Objects.requireNonNull(rawFragment);
        if (rawFragment.isEmpty()) {
            return;
        }
        char[] chars = rawFragment.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            char c = chars[i];
            UriValidator.validateAscii(UriValidationException.Segment.FRAGMENT, chars, i, c);
            if (UNRESERVED[c] || SUB_DELIMS[c] || c == '@' || c == ':') continue;
            if (c == '%') {
                UriValidator.validatePercentEncoding(UriValidationException.Segment.FRAGMENT, rawFragment, chars, i);
                i += 2;
                continue;
            }
            UriValidator.failInvalidChar(UriValidationException.Segment.FRAGMENT, chars, i, c);
        }
    }

    static String print(char c) {
        if (UriValidator.printable(c)) {
            return "'" + c + "'";
        }
        return "0x" + UriValidator.hex(c);
    }

    static String encode(char[] chars) {
        StringBuilder result = new StringBuilder(chars.length);
        for (char aChar : chars) {
            if (aChar > '\u00fe') {
                result.append('?');
                continue;
            }
            if (UriValidator.printable(aChar)) {
                result.append(aChar);
                continue;
            }
            result.append('?');
        }
        return result.toString();
    }

    private static void failInvalidChar(UriValidationException.Segment segment, char[] chars, int i, char c) {
        throw new UriValidationException(segment, chars, segment.text() + " contains invalid char", i, c);
    }

    private static void validateAscii(UriValidationException.Segment segment, char[] chars, int i, char c) {
        if (c > '\u00fe') {
            throw new UriValidationException(segment, chars, segment.text() + " contains invalid char (non-ASCII)", i, c);
        }
    }

    private static void validatePercentEncoding(UriValidationException.Segment segment, String value, char[] chars, int i) {
        if (i + 2 >= chars.length) {
            throw new UriValidationException(segment, chars, segment.text() + " contains invalid % encoding, not enough chars left at index " + i);
        }
        char p1 = chars[i + 1];
        char p2 = chars[i + 2];
        UriValidator.validateHex(segment, value, chars, p1, segment.text(), i + 1, true);
        UriValidator.validateHex(segment, value, chars, p2, segment.text(), i + 2, true);
    }

    private static void validateHex(UriValidationException.Segment segment, String fullValue, char[] chars, char c, String type, int index, boolean isPercentEncoding) {
        if (c > '\u00ff' || !HEXDIGIT[c]) {
            if (fullValue.length() == chars.length) {
                if (isPercentEncoding) {
                    throw new UriValidationException(segment, chars, type + " has non hexadecimal char in % encoding", index, c);
                }
                throw new UriValidationException(segment, chars, type + " has non hexadecimal char", index, c);
            }
            if (isPercentEncoding) {
                throw new UriValidationException(segment, fullValue.toCharArray(), chars, type + " has non hexadecimal char in % encoding", index, c);
            }
            throw new UriValidationException(segment, fullValue.toCharArray(), chars, type + " has non hexadecimal char", index, c);
        }
    }

    private static String hex(char c) {
        String hexString = Integer.toHexString(c);
        if (hexString.length() == 1) {
            return "0" + hexString;
        }
        return hexString;
    }

    private static void validateH16(String host, String inProgress) {
        if (inProgress.isBlank()) {
            throw new UriValidationException(UriValidationException.Segment.HOST, host.toCharArray(), "IPv6 segment is empty");
        }
        if (inProgress.length() > 4) {
            throw new UriValidationException(UriValidationException.Segment.HOST, host.toCharArray(), inProgress.toCharArray(), "IPv6 segment has more than 4 chars");
        }
        UriValidator.validateHexDigits(UriValidationException.Segment.HOST, "IPv6 segment", host, inProgress);
    }

    private static void validateHexDigits(UriValidationException.Segment segment, String description, String host, String section) {
        char[] chars = section.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            char c = chars[i];
            UriValidator.validateHex(segment, host, chars, c, description, i, false);
        }
    }

    private static void validateIpOctet(String message, String host, String octet) {
        int octetInt = Integer.parseInt(octet);
        if (octetInt > 255) {
            throw new UriValidationException(UriValidationException.Segment.HOST, host.toCharArray(), message);
        }
    }

    private static void validateIpFuture(String ipLiteral, String host) {
        int dot = host.indexOf(46);
        if (dot == -1) {
            throw new UriValidationException(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), "IP Future must contain 'v<version>.'");
        }
        String version = host.substring(1, dot);
        UriValidator.checkNotBlank(UriValidationException.Segment.HOST, "Version", ipLiteral, version);
        UriValidator.validateHexDigits(UriValidationException.Segment.HOST, "Future version", ipLiteral, version);
        String address = host.substring(dot + 1);
        UriValidator.checkNotBlank(UriValidationException.Segment.HOST, "IP Future", ipLiteral, address);
        char[] chars = address.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            char c = chars[i];
            UriValidator.validateAscii(UriValidationException.Segment.HOST, chars, i, c);
            if (UNRESERVED[c] || SUB_DELIMS[c] || c == ':') continue;
            UriValidator.failInvalidChar(UriValidationException.Segment.HOST, ipLiteral.toCharArray(), i + dot + 1, c);
        }
    }

    private static void checkNotBlank(UriValidationException.Segment segment, String message, String ipLiteral, String toValidate) {
        if (toValidate.isBlank()) {
            if (ipLiteral.equals(toValidate)) {
                throw new UriValidationException(segment, ipLiteral.toCharArray(), message + " cannot be blank");
            }
            throw new UriValidationException(segment, ipLiteral.toCharArray(), toValidate.toCharArray(), message + " cannot be blank");
        }
    }

    private static boolean printable(char c) {
        if (c > '\u00fe') {
            return false;
        }
        if (UNRESERVED[c]) {
            return true;
        }
        if (SUB_DELIMS[c]) {
            return true;
        }
        return PRINTABLE[c];
    }

    static {
        int i;
        IP_V4_PATTERN = Pattern.compile("^([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})\\.([0-9]{1,3})$");
        HEXDIGIT = new boolean[256];
        UNRESERVED = new boolean[256];
        SUB_DELIMS = new boolean[256];
        PRINTABLE = new boolean[256];
        for (i = 48; i <= 57; ++i) {
            UriValidator.UNRESERVED[i] = true;
        }
        for (i = 97; i <= 122; ++i) {
            UriValidator.UNRESERVED[i] = true;
        }
        for (i = 65; i <= 90; ++i) {
            UriValidator.UNRESERVED[i] = true;
        }
        UriValidator.UNRESERVED[45] = true;
        UriValidator.UNRESERVED[46] = true;
        UriValidator.UNRESERVED[95] = true;
        UriValidator.UNRESERVED[126] = true;
        for (i = 48; i <= 57; ++i) {
            UriValidator.HEXDIGIT[i] = true;
        }
        for (i = 97; i <= 102; ++i) {
            UriValidator.HEXDIGIT[i] = true;
        }
        for (i = 65; i <= 70; ++i) {
            UriValidator.HEXDIGIT[i] = true;
        }
        UriValidator.SUB_DELIMS[33] = true;
        UriValidator.SUB_DELIMS[36] = true;
        UriValidator.SUB_DELIMS[38] = true;
        UriValidator.SUB_DELIMS[39] = true;
        UriValidator.SUB_DELIMS[40] = true;
        UriValidator.SUB_DELIMS[41] = true;
        UriValidator.SUB_DELIMS[42] = true;
        UriValidator.SUB_DELIMS[43] = true;
        UriValidator.SUB_DELIMS[44] = true;
        UriValidator.SUB_DELIMS[59] = true;
        UriValidator.SUB_DELIMS[61] = true;
        UriValidator.PRINTABLE[58] = true;
        UriValidator.PRINTABLE[47] = true;
        UriValidator.PRINTABLE[63] = true;
        UriValidator.PRINTABLE[64] = true;
        UriValidator.PRINTABLE[37] = true;
        UriValidator.PRINTABLE[35] = true;
        UriValidator.PRINTABLE[91] = true;
        UriValidator.PRINTABLE[93] = true;
    }
}

