package com.ovopark.boot.cache.redis.service.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.core.type.TypeReference;
import com.ovopark.boot.cache.redis.RedisFactory;
import com.ovopark.boot.cache.redis.service.BootRedisService;
import com.ovopark.boot.cache.utils.SerializeUtil;
import com.ovopark.boot.kit.json.JacksonKit;
//import com.google.gson.Gson;
//import com.google.gson.reflect.TypeToken;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.ScanParams;
import redis.clients.jedis.ScanResult;
import redis.clients.jedis.ShardedJedis;
import redis.clients.jedis.ShardedJedisPool;

@Component
public class BootRedisServiceImpl implements BootRedisService {
    @Autowired
    private JedisPool jedisPool;
    //分布式
    private static ShardedJedisPool shardedJedisPool = null;

    public static Logger log = LoggerFactory.getLogger(RedisFactory.class);
    //默认数据库
    private static int default_db = 0;

    /**
     * @param @return 参数
     * @return Jedis    返回类型
     * @throws
     * @Title: getJedis
     * @Description: TODO(获取Jedis实例)
     */
    public synchronized Jedis getJedis() {
        try {
            if (jedisPool != null) {
                Jedis resource = jedisPool.getResource();
                return resource;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis缓存获取Jedis实例 出错！", e);
            return null;
        }
    }

    /**
     * @param @return 参数
     * @return ShardedJedis    返回类型
     * @throws
     * @Title: getShardedJedis
     * @Description: TODO(获取shardedJedis实例)
     */
    public static ShardedJedis getShardedJedis() {
        try {
            if (shardedJedisPool != null) {
                ShardedJedis resource = shardedJedisPool.getResource();
                return resource;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis缓存获取shardedJedis实例 出错！", e);
            return null;
        }
    }

    /**
     * @param @param jedis    参数
     * @return void    返回类型
     * @throws
     * @Title: closeResource
     * @Description: TODO(释放jedis资源)
     */
    public void closeResource(final Jedis jedis) {
        if (jedis != null) {
            jedis.close();
//            jedisPool.returnResource(jedis);
        }
    }

    /**
     * @param @param shardedJedis    参数
     * @return void    返回类型
     * @throws
     * @Title: closeResource
     * @Description: TODO(释放shardedJedis资源)
     */
    public void closeResource(final ShardedJedis shardedJedis) {
        if (shardedJedis != null) {
            shardedJedis.close();
//        	shardedJedisPool.returnResource(shardedJedis);
        }
    }

    @Override
    public boolean set(Object key, String value) {
        return set(key, value, default_db);
    }

    @Override
    public boolean set(Object key, String value, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                jedis.set((String) key, value);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public boolean set(Object key, String value, String nxxx, String expx, long time) {
        return set(key, value, nxxx, expx, time, default_db);
    }

    @Override
    public boolean set(Object key, String value, String nxxx, String expx, long time, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                jedis.set((String) key, value, nxxx, expx, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> boolean setPoJo(Object key, Object pojo) {
        return setPoJo(key, pojo, default_db);
    }

    @Override
    public <T> boolean setPoJo(Object key, Object pojo, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//            	String json=new Gson().toJson(pojo);
                String json = JacksonKit.toJSon(pojo);
                jedis.set((String) key, json);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> boolean setPoJo(Object key, Object pojo, String nxxx, String expx, long time) {
        return setPoJo(key, pojo, nxxx, expx, time, default_db);
    }

    @Override
    public <T> boolean setPoJo(Object key, Object pojo, String nxxx, String expx, long time, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//            	String json=new Gson().toJson(pojo);
                String json = JacksonKit.toJSon(pojo);
                jedis.set((String) key, json, nxxx, expx, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> boolean setlist(Object key, List<T> list) {
        return setlist(key, list, default_db);
    }

    @Override
    public <T> boolean setlist(Object key, List<T> list, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//        		String json=new Gson().toJson(list);
                String json = JacksonKit.toJSon(list);
                jedis.set((String) key, json);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】 出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> boolean setlist(Object key, List<T> list, String nxxx, String expx, long time) {
        return setlist(key, list, nxxx, expx, time, default_db);
    }

    @Override
    public <T> boolean setlist(Object key, List<T> list, String nxxx, String expx, long time, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//        		String json=new Gson().toJson(list);
                String json = JacksonKit.toJSon(list);
                jedis.set((String) key, json, nxxx, expx, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】 出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Object get(Object key) {
        return get(key, default_db);
    }

    @Override
    public Object get(Object key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.get((String) key);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存获取key值【" + key + "】 出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> T get(Object key, Class<T> clazz) {
        return get(key, clazz, default_db);
    }

    @Override
    public <T> T get(Object key, Class<T> clazz, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//            	 return (T) new Gson().fromJson(jedis.get((String)key), clazz);
                return JacksonKit.readValue(JacksonKit.toJSon(jedis.get((String) key)), new TypeReference<T>() {
                });
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存获取key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> List<T> getList(Object key) {
        return getList(key, default_db);
    }

    @Override
    public <T> List<T> getList(Object key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//                return (List<T>)new Gson().fromJson(jedis.get((String)key), new TypeToken<List<T>>() {}.getType());
                return JacksonKit.readValue(JacksonKit.toJSon(jedis.get((String) key)), new TypeReference<List<T>>() {
                });

            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存获取key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long del(Object key) {
        return del(key, default_db);
    }

    @Override
    public Long del(Object key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.del((String) key);
            } else {
                return 0L;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存删除key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Set<String> keys(String pattern) {
        return keys(pattern, default_db);
    }

    @Override
    public Set<String> keys(String pattern, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.keys(pattern);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存获取pattern值【" + pattern + "】 出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public boolean setExpire(Object key, String value, int time) {
        return setExpire(key, value, time, default_db);
    }

    @Override
    public boolean setExpire(Object key, String value, int time, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                jedis.set((String) key, value);
                jedis.expire((String) key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> boolean setPoJoExpire(Object key, Object pojo, int time) {
        return setPoJoExpire(key, pojo, time, default_db);
    }

    @Override
    public <T> boolean setPoJoExpire(Object key, Object pojo, int time, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//            	String json=new Gson().toJson(pojo);
                String json = JacksonKit.toJSon(pojo);
                jedis.set((String) key, json);
                jedis.expire((String) key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public <T> boolean setlistExpire(Object key, List<T> list, int time) {
        return setPoJoExpire(key, list, time, default_db);
    }

    @Override
    public <T> boolean setlistExpire(Object key, List<T> list, int time, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
//        		String json=new Gson().toJson(list);
                String json = JacksonKit.toJSon(list);
                jedis.set((String) key, json);
                jedis.expire((String) key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db" + index + "缓存设置key值 【" + key + "】 出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long ttl(Object key) {
        return ttl(key, default_db);
    }

    @Override
    public Long ttl(Object key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.ttl((String) key);
            } else {
                return 0L;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "获取key值 【" + key + "】剩余生存时间出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long zadd(String key, double score, String member) {
        return zadd(key, score, member, default_db);
    }

    @Override
    public Long zadd(String key, double score, String member, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.zadd(key, score, member);
            } else {
                return 0L;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "获取key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
            closeResource(jedis);
        }

    }

    @Override
    public Set<String> zrevrange(String key, int start, int end) {
        return zrevrange(key, start, end, default_db);
    }

    @Override
    public Set<String> zrevrange(String key, int start, int end, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.zrevrange(key, start, end);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "获取key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Set<String> zrange(String key, int start, int end) {
        return zrange(key, start, end, default_db);
    }

    @Override
    public Set<String> zrange(String key, int start, int end, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.zrange(key, start, end);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "获取key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Double zscore(String key, String member) {
        return zscore(key, member, default_db);
    }

    @Override
    public Double zscore(String key, String member, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.zscore(key, member);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "获取key值 【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long zrem(String key, String member) {
        return zrem(key, member, default_db);
    }

    @Override
    public Long zrem(String key, String member, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.zrem(key, member);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "删除key值 【" + key + "】,member值【" + member + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long zcard(String key) {
        return zcard(key, default_db);
    }

    @Override
    public Long zcard(String key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.zcard(key);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "scard【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long hset(String key, String field, String value) {
        return hset(key, field, value, default_db);
    }

    @Override
    public Long hset(String key, String field, String value, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.hset(key, field, value);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "hset【" + key + "】,field【" + field + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public String hget(String key, String field) {
        return hget(key, field, default_db);
    }

    @Override
    public String hget(String key, String field, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.hget(key, field);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "hget【" + key + "】,field【" + field + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long hdel(String key, String field) {
        return hdel(key, field, default_db);
    }

    @Override
    public Long hdel(String key, String field, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.hdel(key, field);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "hdel【" + key + "】,field【" + field + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Map<String, String> hgetall(String key) {
        return hgetall(key, default_db);
    }

    @Override
    public Map<String, String> hgetall(String key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.hgetAll(key);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "hgetall【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }


    @Override
    public byte[] getObject(String key, int index) {
        Jedis jedis = getJedis();
        Object obj = null;
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);

                byte[] data = jedis.get(key.getBytes());
//	       		if (data != null && data.length > 0) {
//	       			obj = SerializeUtil.unserialize(data);
//	       		}
//            	return  obj;
                return data;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "getObject【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public boolean setObject(String key, Object obj, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                jedis.set(key.getBytes(), SerializeUtil.serialize(obj));
                return true;
            } else {
                return false;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "setObject【" + key + "】出错！", e);
            e.printStackTrace();
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public Long dbsize(int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.dbSize();
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "dbsize出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    @Override
    public List<String> scan(String key) {
        return scan(key, default_db);
    }

    @Override
    public List<String> scan(String key, int index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                ScanParams scanParams = new ScanParams();
                scanParams.match(key + "*");
                scanParams.count(1000);
                ScanResult<String> result = jedis.scan("0", scanParams);
                return result.getResult();
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "scan【" + key + "】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
            closeResource(jedis);
        }
    }

    /**
     * Redis SetNX（SET if Not exists）
     * 命令在指定的key不存在时，为key设置指定的值。
     * 设置成功，返回1。设置失败，返回0。
     *
     * @param key   key
     * @param value value
     * @return
     */
    public Long setNX(String key, String value) {
        return setNX(key, value, default_db);
    }

    /**
     * Redis SetNX（SET if Not exists）
     * 命令在指定的key不存在时，为key设置指定的值。
     * 设置成功，返回1。设置失败，返回0。
     *
     * @param key   key
     * @param value value
     * @param index redis库
     * @return
     */
    public Long setNX(String key, String value, Integer index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.setnx(key, value);
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "setNX【" + key + "】,field【" + value + "】出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
            closeResource(jedis);
        }
    }

    /***
     * 将值value关联到key，并将key的生存时间设为seconds(以秒为单位)。
     * 如果key已经存在，setEX命令将覆写旧值。
     * 这个命令类似于以下两个命令：
     * SET key value
     * EXPIRE key seconds
     * 不同之处是，setEX是一个原子性(atomic)操作，关联值和设置生存时间两个动作会在同一时间内完成，该命令在Redis用作缓存时，非常实用。
     *
     * @param key key
     * @param value value
     * @param seconds 过期时长 单位：秒
     * @return
     */
    public Boolean setEX(String key, String value, Integer seconds) {
        return setEX(key, value, seconds, default_db);
    }

    /***
     * 将值value关联到key，并将key的生存时间设为seconds(以秒为单位)。
     * 如果key已经存在，setEX命令将覆写旧值。
     * 这个命令类似于以下两个命令：
     * SET key value
     * EXPIRE key seconds
     * 不同之处是，setEX是一个原子性(atomic)操作，关联值和设置生存时间两个动作会在同一时间内完成，该命令在Redis用作缓存时，非常实用。
     *
     * @param key key
     * @param value value
     * @param seconds 过期时长 单位：秒
     * @param index redis库
     * @return
     */
    public Boolean setEX(String key, String value, Integer seconds, Integer index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                String result = jedis.setex(key, seconds, value);
                return result.equals("OK");
            } else {
                return false;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "setEX【" + key + "】,field【" + value + "】出错！", e);
            e.printStackTrace();
            return false;
        } finally {
            closeResource(jedis);
        }
    }

    /**
     * key 是否存在
     *
     * @param key key
     * @return
     */
    public Boolean exists(String key) {
        return exists(key, default_db);
    }

    /**
     * key 是否存在
     *
     * @param key   key
     * @param index redis库
     * @return
     */
    public Boolean exists(String key, Integer index) {
        Jedis jedis = getJedis();
        try {
            if (jedis != null) {
                if (index != default_db)
                    jedis.select(index);
                return jedis.exists(key);
            } else {
                return false;
            }
        } catch (Exception e) {
            log.error("Redis db" + index + "exists【" + key + "】出错！", e);
            e.printStackTrace();
            return false;
        } finally {
            closeResource(jedis);
        }
    }
}
