package com.ovopark.boot.utils.validate;

import org.hibernate.validator.HibernateValidator;
import org.springframework.util.StringUtils;

import javax.validation.ConstraintViolation;
import javax.validation.Valid;
import javax.validation.Validation;
import javax.validation.Validator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * Validation校验工具类.
 *
 * @author Turbo Klaus
 * @author <a href="mailto:guyue375@outlook.com">klaus</a>
 **/
public class ConditionalParamValidateUtils {
    
    private static Validator validatorFast = Validation.byProvider(HibernateValidator.class).configure().failFast(true)
            .buildValidatorFactory().getValidator();
    
    private static Validator validatorAll = Validation.byProvider(HibernateValidator.class).configure().failFast(false)
            .buildValidatorFactory().getValidator();
    
    public static <T> Set<ConstraintViolation<T>> validateFast(@Valid T domain) throws Exception {
        return validateFast(domain, false);
    }
    
    /**
     * 校验遇到第一个不合法的字段直接返回不合法字段，后续字段不再校验.
     *
     * @param showValidField 是否同时输出是哪个字段校验失败.
     */
    public static <T> Set<ConstraintViolation<T>> validateFast(@Valid T domain, boolean showValidField)
            throws Exception {
        Set<ConstraintViolation<T>> validateResult = validatorFast.validate(domain);
        if (validateResult.size() > 0) {
            String validResultStr = validateResult.iterator().next().getMessage();
            if (showValidField) {
                validResultStr = "[ " + validateResult.iterator().next().getPropertyPath() + " ] " + validResultStr;
            }
            throw new IllegalArgumentException(validResultStr);
        }
        return validateResult;
    }
    
    /**
     * 校验所有字段并返回不合法字段.
     */
    public static <T> Set<ConstraintViolation<T>> validateAll(@Valid T domain) throws Exception {
        return validateAll(domain,false);
    }
    
    /**
     *
     * @param domain
     * @param showValidField 是否展示校验字段.
     */
    public static <T> Set<ConstraintViolation<T>> validateAll(@Valid T domain,boolean showValidField) throws Exception {
        Set<ConstraintViolation<T>> validateResult = validatorAll.validate(domain);
        if (validateResult.size() > 0) {
            Iterator<ConstraintViolation<T>> it = validateResult.iterator();
            StringBuilder stringBuilder = new StringBuilder();
            while (it.hasNext()) {
                ConstraintViolation<T> cv = it.next();
                String res = cv.getMessage();
                if (showValidField) {
                    res = "[ "+cv.getPropertyPath()+" ] "+cv.getMessage();
                }
                if (StringUtils.hasText(stringBuilder)) {
                    stringBuilder.append(", "+res);
                } else {
                    stringBuilder.append(res);
                }
            }
            throw new IllegalArgumentException(stringBuilder.toString());
        }
        return validateResult;
    }
}
