package com.ovopark.boot.bus.redis.service.base;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Remiel_Mercy xuefei_fly@126.com
 * @ClassName: Redis
 * @Description: TODO(Redis方法接口)
 * @date 2017年8月28日 下午1:59:47
 * <p>
 * 命令参考：http://redisdoc.com/
 */
public interface Redis {
  /**
   * @Title: set
   * @Description: TODO(SET key value [EX seconds] [PX milliseconds] [NX | XX])
   * 将字符串值 value 关联到 key 。
   * 如果 key 已经持有其他值， SET 就覆写旧值，无视类型。
   * 对于某个原本带有生存时间（TTL）的键来说， 当 SET 命令成功在这个键上执行时， 这个键原有的 TTL 将被清除。
   * <p>
   * 可选参数
   * 从 Redis 2.6.12 版本开始， SET 命令的行为可以通过一系列参数来修改：
   * EX second ：设置键的过期时间为 second 秒。 SET key value EX second 效果等同于 SETEX key second value 。
   * PX millisecond ：设置键的过期时间为 millisecond 毫秒。 SET key value PX millisecond 效果等同于 PSETEX key millisecond value 。
   * NX ：只在键不存在时，才对键进行设置操作。 SET key value NX 效果等同于 SETNX key value 。
   * XX ：只在键已经存在时，才对键进行设置操作。
   * <p>
   * 在 Redis 2.6.12 版本以前， SET 命令总是返回 OK 。
   * 从 Redis 2.6.12 版本开始， SET 在设置操作成功完成时，才返回 OK 。
   * 如果设置了 NX 或者 XX ，但因为条件没达到而造成设置操作未执行，那么命令返回空批量回复（NULL Bulk Reply）。
   */
  /**
   * 数据库索引(配置文件database 16 #默认为16个数据库，可以自己更改)
   **/
  boolean set(Object key, String value, int index);

  String set(Object key, String value, String nxxx, String expx, long time, int index);

  /**
   * 为给定 key 设置生存时间，当 key 过期时(生存时间为 0 )，它会被自动删除   time:秒
   **/
  boolean setExpire(Object key, String value, int time, int index);

  /**
   * @param <T>
   * @param @param  key
   * @param @param  obj
   * @param @return 参数
   * @return String    返回类型
   * @throws
   * @Title: setPoJo
   * @Description: TODO(将pojo类对象序列化成redis支持的内容)
   */
  <T> boolean setPoJo(Object key, Object pojo, int index);

  <T> boolean setPoJo(Object key, Object pojo, String nxxx, String expx, long time, int index);

  /**
   * @param @param  key  	redis key
   * @param @param  pojo	pojo对象
   * @param @param  time  过期时间
   * @param @return 参数
   * @return boolean    返回类型
   * @throws
   * @Title: setPoJoExpire
   * @Description: TODO(设置一个会过期的key)
   */
  <T> boolean setPoJoExpire(Object key, Object pojo, int time, int index);

  /**
   * @param <T>
   * @Title: setlist
   * @Description: TODO(设置list < T > 集合)
   */
  <T> boolean setlist(Object key, List<T> list, int index);

  <T> boolean setlist(Object key, List<T> list, String nxxx, String expx, long time, int index);

  <T> boolean setlistExpire(Object key, List<T> list, int time, int index);

  /**
   * @Title: get
   * 返回 key 所关联的字符串值。
   * 如果 key 不存在那么返回特殊值 nil 。
   * 假如 key 储存的值不是字符串类型，返回一个错误，因为 GET 只能用于处理字符串值。
   * <p>
   * 返回值：
   * 当 key 不存在时，返回 nil ，否则，返回 key 的值。
   * 如果 key 不是字符串类型，那么返回一个错误。（eg:(error) ERR Operation against a key holding the wrong kind of value）
   */
  Object get(Object key, int index);

  /**
   * @Title: get
   * @Description: TODO(获取pojo对象)
   */
  <T> T get(Object key, Class<T> clazz, int index);

  /**
   * @Title: getList
   * @Description: TODO(获取list < T > 集合)
   */
  <T> List<T> getList(Object key, int index);

  /**
   * @Title: del
   * @Description: TODO(删除给定的一个或多个 key 。 不存在的 key 会被忽略)
   */
  Long del(Object key, int index);

  /**
   * @Title: keys
   * @Description: TODO(查找所有符合给定模式 pattern 的 key)
   * KEYS * 匹配数据库中所有 key 。
   * KEYS h?llo 匹配 hello ， hallo 和 hxllo 等。
   * KEYS h*llo 匹配 hllo 和 heeeeello 等。
   * KEYS h[ae]llo 匹配 hello 和 hallo ，但不匹配 hillo 。
   * KEYS 的速度非常快，但在一个大的数据库中使用它仍然可能造成性能问题，如果你需要从一个数据集中查找特定的 key ，你最好还是用 Redis 的集合结构(set)来代替
   */
  Set<String> keys(String pattern, int index);

  /**
   * @Title: ttl
   * @Description: TODO(以秒为单位 ， 返回给定 key 的剩余生存时间 ( TTL, time to live)
   * 当 key 不存在时，返回 -2 。
   * 当 key 存在但没有设置剩余生存时间时，返回 -1 。
   * 否则，以秒为单位，返回 key 的剩余生存时间。
   */
  Long ttl(Object key, int index);

  /**
   * @param @param  key    集合名称
   * @param @param  score  分值
   * @param @param  member 成员
   * @param @return 参数
   * @return Long    返回类型
   * @throws
   * @Title: zadd
   * @Description: TODO(有序集合添加)
   * 将一个或多个 member 元素及其 score 值加入到有序集 key 当中。
   * 如果某个 member 已经是有序集的成员，那么更新这个 member 的 score 值，并通过重新插入这个 member 元素，来保证该 member 在正确的位置上。
   * score 值可以是整数值或双精度浮点数。
   * 如果 key 不存在，则创建一个空的有序集并执行 ZADD 操作。
   */
  Long zadd(String key, double score, String member, int index);

  /**
   * zset  key = key ,score、member = value
   * @param values
   * @param index
   * @return
   */
  Integer zadd(Map<String,Long> values, int index);
  /**
   * @param @param  key
   * @param @param  start
   * @param @param  end
   * @param @return 参数
   * @return Set<String>    返回类型
   * @throws
   * @Title: zrange
   * @Description: TODO(sort set集合从高到低排序)
   */
  Set<String> zrange(String key, int start, int end, int index);

  Map<String, Double> zrangeWithScores(String key, int start, int end, int index);

  /**
   * @param @param  key
   * @param @param  start
   * @param @param  end
   * @param @return 参数
   * @return Set<String>    返回类型
   * @throws
   * @Title: zrevrange
   * @Description: TODO(sort set集合从低到高排序)
   */
  Set<String> zrevrange(String key, int start, int end, int index);

  /**
   * @param @param  key
   * @param @param  member
   * @param @return 参数
   * @return Double    返回类型
   * @throws
   * @Title: zscore
   * @Description: TODO(获得sort set集合中元素的分数)
   */
  Double zscore(String key, String member, int index);

  /**
   * @param @param  key
   * @param @param  member
   * @param @return 参数
   * @return Long    返回类型
   * @throws
   * @Title: zrem
   * @Description: TODO(删除一个或多个sort set集合中元素)
   */
  Long zrem(String key, String member, int index);

  /**
   * 根据分数区间删除
   *
   * @param key
   * @param min
   * @param max
   * @return
   */
  Long zremRangeByScore(String key, double min, double max, int index);

  /**
   * @param @param  key
   * @param @return 参数
   * @return Long    返回类型
   * @throws
   * @Title: zcard
   * @Description: TODO(返回sort set集合的数量)
   * 返回set 集合数量的方法是scard
   */
  Long zcard(String key, int index);

  /**
   * 按照分数返回数据
   *
   * @param key
   * @param min
   * @param max
   * @return
   */
  Map<String, Double> zrangeByScore(String key, double min, double max, int index);

  /**
   * @throws
   * @Title: hset
   * @Description: TODO(向名称为key的hash中添加元素field < — > value)
   * @param: @param key
   * @param: @param field
   * @param: @param value
   * @param: @return
   * @return: boolean
   */
  Long hset(String key, String field, String value, int index);

  String hmset(String key, Map<String, String> hash, int index);

  /**
   * @throws
   * @Title: hget
   * @Description: TODO(返回名称为key的hash中field对应的value)
   * @param: @param key
   * @param: @param field
   * @param: @return
   * @return: Object
   */
  String hget(String key, String field, int index);

  /**
   * @throws
   * @Title: hdel
   * @Description: TODO(删除名称为key的hash中键为field的域)
   * @param: @param key
   * @param: @param field
   * @param: @return
   * @return: Long
   */
  Long hdel(String key, String field, int index);

  /**
   * @throws
   * @Title: hgetall
   * @Description: TODO(返回名称为key的hash中所有键对应的value)
   * @param: @param key
   * @param: @return
   * @return: Map<String   ,       String>
   */
  Map<String, String> hgetall(String key, int index);

  /**
   * key 类似keys 模糊查询
   *
   * @param key
   * @return
   */
  Map<String, String> hscan(String key, String field, int index);

  /**
   * 取出并反序列化对象
   *
   * @param key
   * @param obj
   * @param index
   * @return
   */
  public boolean setObject(String key, Object obj, int index);

  /**
   * 序列化对象直接存储
   *
   * @param key
   * @param index
   * @return
   */
  public Object getObject(String key, int index);

  /**
   * 数据set 类型 添加
   *
   * @param key
   * @param index
   * @param members
   * @return
   */
  Long sadd(String key, int index, String... members);

  /**
   * 获取全部数据
   *
   * @param key
   * @param index
   * @return
   */
  Set<String> smembers(String key, int index);

  /**
   * 删除数据
   *
   * @param key
   * @param index
   * @param members
   * @return
   */
  Long srem(String key, int index, String... members);

  boolean eval(String script, List<String> keys, List<String> args, int index);

  List<String> scan(String key, int count, int index);

  /**
   * key 加一
   * @param key
   * @param index
   * @return
   */
  Long incr(String key, int index);

  /**
   * 减一
   * @param key
   * @param index
   * @return
   */
  Long decr(String key, int index);
}
