package com.ovopark.cloud.cache.redis;

import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.ovopark.cloud.cache.Redis;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.ScanParams;
import redis.clients.jedis.ScanResult;
import redis.clients.jedis.ShardedJedis;
import redis.clients.jedis.ShardedJedisPool;
/**
 * 
    * @ClassName: RedisFactory
    * @Description: TODO(Redis实现)
    * @author Remiel_Mercy xuefei_fly@126.com
    * @date 2017年8月28日 下午2:59:39
 */
public class RedisFactory implements Redis{
	private enum CoCoRedis{
		instance;
		private RedisFactory factory;
		CoCoRedis(){
			factory=new RedisFactory();
		}
		private RedisFactory init(){
	        return factory;
	    }
	}
	public static RedisFactory createRedis() {
		return CoCoRedis.instance.init();
	}
	
	public static Logger log = LoggerFactory.getLogger(RedisFactory.class);
	private static JedisPool jedisPool = null;
	private static int default_db=0;
	//分布式
	private static ShardedJedisPool shardedJedisPool = null;
	static {
		loadConfig();
	}
	
	private static void loadConfig(){
		try {
            // 加载redis配置文件
            ResourceBundle bundle = ResourceBundle.getBundle("config/properties/app");
            if (bundle == null) {
                throw new IllegalArgumentException( "[redis.properties] is not found!");
            }
            // 创建jedis池配置实例
            JedisPoolConfig config = new JedisPoolConfig();
            //最大连接数
            config.setMaxTotal(Integer.valueOf(bundle.getString("redis.maxTotal")));
            //最大空闲连接数
            config.setMaxIdle(Integer.valueOf(bundle.getString("redis.maxIdle")));
            //等待可用连接的最大时间
            config.setMaxWaitMillis(Long.valueOf(bundle.getString("redis.maxWaitMillis")));
            //逐出连接的最小空闲时间 默认1800000毫秒(30分钟)
            config.setMinEvictableIdleTimeMillis(Long.valueOf(bundle.getString("redis.minEvictableIdleTimeMillis")));
            //最小空闲连接数
            config.setMaxIdle(Integer.valueOf(bundle.getString("redis.minIdle")));
            //每次逐出检查时 逐出的最大数目 如果为负数就是 : 1/abs(n), 默认3
            config.setNumTestsPerEvictionRun(Integer.valueOf(bundle.getString("redis.numTestsPerEvictionRun")));
            //在获取连接的时候检查有效性, 默认false
            config.setTestOnBorrow(Boolean.valueOf(bundle.getString("redis.testOnBorrow")));
            //return给pool时，是否提前进行validate操作；
            config.setTestOnReturn(Boolean.valueOf(bundle.getString("redis.testOnReturn")));
            //在空闲时检查有效性, 默认false
            config.setTestWhileIdle(Boolean.valueOf(bundle.getString("redis.testWhileIdle")));
            //逐出扫描的时间间隔(毫秒) 如果为负数,则不运行逐出线程, 默认-1
            config.setTimeBetweenEvictionRunsMillis(Integer.valueOf(bundle.getString("redis.timeBetweenEvictionRunsMillis")));
            //连接耗尽时是否阻塞, false报异常,ture阻塞直到超时, 默认true
            config.setBlockWhenExhausted(Boolean.valueOf(bundle.getString("redis.blockWhenExhausted")));
            
            jedisPool = new JedisPool(config, bundle.getString("redis.host"),
                    Integer.valueOf(bundle.getString("redis.port")), 10000,
                    bundle.getString("redis.password"));
            // slave链接
//            List<JedisShardInfo> shards = new ArrayList<JedisShardInfo>();
//            shards.add(new JedisShardInfo(bundle.getString("redis.ip"), Integer
//                    .valueOf(bundle.getString("redis.port1"))));
//            shardedJedisPool = new ShardedJedisPool(config, shards);
            log.info("初始化Redis连接池success");
        } catch (Exception e) {
            log.error("初始化Redis连接池 出错！", e);
        }
	}
	/**
	 * 
	    * @Title: getJedis
	    * @Description: TODO(获取Jedis实例)
	    * @param @return    参数
	    * @return Jedis    返回类型
	    * @throws
	 */
    public synchronized static Jedis getJedis() {
        try {
            if (jedisPool != null) {
                Jedis resource = jedisPool.getResource();
                return resource;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis缓存获取Jedis实例 出错！", e);
            return null;
        }
    }
    /**
     * 
        * @Title: getShardedJedis
        * @Description: TODO(获取shardedJedis实例)
        * @param @return    参数
        * @return ShardedJedis    返回类型
        * @throws
     */
    public static ShardedJedis getShardedJedis() {
        try {
            if (shardedJedisPool != null) {
                ShardedJedis resource = shardedJedisPool.getResource();
                return resource;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Redis缓存获取shardedJedis实例 出错！", e);
            return null;
        }
    }
    
    /**
        * @Title: closeResource
        * @Description: TODO(释放jedis资源)
        * @param @param jedis    参数
        * @return void    返回类型
        * @throws
     */
    public static void closeResource(final Jedis jedis) {
        if (jedis != null) {
            jedisPool.returnResource(jedis);
        }
    }

   /**
       * @Title: closeResource
       * @Description: TODO( 释放shardedJedis资源)
       * @param @param shardedJedis    参数
       * @return void    返回类型
       * @throws
    */
    public static void closeResource(final ShardedJedis shardedJedis) {
        if (shardedJedis != null) {
        	shardedJedisPool.returnResource(shardedJedis);
        }
    }
    
    @Override
	public boolean  set(Object key, String value) {
    	 return set(key,value,default_db);
	}
	@Override
	public boolean set(Object key, String value, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
            	if(index!=default_db)
            		jedis.select(index);
                jedis.set((String)key, value);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	
	@Override
	public boolean set(Object key, String value,String nxxx, String expx,  long time) {
		return set(key,value,nxxx,expx,time,default_db);
	}
	@Override
	public boolean set(Object key, String value, String nxxx, String expx, long time, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
               	if(index!=default_db)
               		jedis.select(index);
                jedis.set((String)key, value,nxxx,expx,time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	
	@Override
	public <T> boolean  setPoJo(Object key,  Object pojo) {
		return setPoJo(key,pojo,default_db);
	}
	@Override
	public <T> boolean setPoJo(Object key, Object pojo, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
             	if(index!=default_db)
               		jedis.select(index);
            	String json=new Gson().toJson(pojo);
                jedis.set((String)key, json);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	
	@Override
	public <T> boolean setPoJo(Object key,Object pojo,String nxxx, String expx,  long time) {
		return setPoJo(key,pojo,nxxx,expx,time,default_db);
	}
	@Override
	public <T> boolean setPoJo(Object key, Object pojo, String nxxx, String expx, long time, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
            	if(index!=default_db)
               		jedis.select(index);
            	String json=new Gson().toJson(pojo);
                jedis.set((String)key, json,nxxx,expx,time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	
	@Override
	public <T> boolean  setlist(Object key, List<T> list) {
		return setlist(key,list,default_db);
	}
	@Override
	public <T> boolean setlist(Object key, List<T> list, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
            	if(index!=default_db)
               		jedis.select(index);
        		String json=new Gson().toJson(list);
                jedis.set((String)key, json);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】 出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	
	@Override
	public <T> boolean setlist(Object key, List<T> list,String nxxx, String expx,  long time) {
		return setlist(key,list,nxxx,expx,time,default_db);
	}
	@Override
	public <T> boolean setlist(Object key, List<T> list, String nxxx, String expx, long time, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
             	if(index!=default_db)
               		jedis.select(index);
        		String json=new Gson().toJson(list);
                jedis.set((String)key, json,nxxx,expx,time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】 出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	
	@Override
	public Object get(Object key) {
		return get(key,default_db);
	}
	@Override
	public Object get(Object key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
            	if(index!=default_db)
               		jedis.select(index);
            	 return jedis.get((String)key);
            } else {
                return null;
            }
        } catch (Exception e) {
        	log.error("Redis db"+index+"缓存获取key值【"+key+"】 出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	
	@Override
	public <T> T get(Object key, Class<T> clazz) {
		return get(key,clazz,default_db);
	}
	@Override
	public <T> T get(Object key, Class<T> clazz, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
               	if(index!=default_db)
               		jedis.select(index);
            	 return (T) new Gson().fromJson(jedis.get((String)key), clazz);
            } else {
                return null;
            }
        } catch (Exception e) {
        	log.error("Redis db"+index+"缓存获取key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	
	@Override
	public  <T> List<T> getList(Object key) {
		return getList(key,default_db);
	}
	@SuppressWarnings("unchecked")
	@Override
	public <T> List<T> getList(Object key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
               	if(index!=default_db)
               		jedis.select(index);
            	 return (List<T>)new Gson().fromJson(jedis.get((String)key), new TypeToken<List<T>>() {}.getType());
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"缓存获取key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	
	@Override
	public Long del(Object key) {
		return del(key,default_db);
	}
	@Override
	public Long del(Object key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return jedis.del((String)key) ; 
            } else {
                return 0L;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"缓存删除key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Set<String> keys(String pattern) {
		return keys(pattern,default_db);
	}
	@Override
	public Set<String> keys(String pattern, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
            	if(index!=default_db)
               		jedis.select(index);
            	 return  jedis.keys(pattern);
            } else {
                return null;
            }
        } catch (Exception e) {
        	log.error("Redis db"+index+"缓存获取pattern值【"+pattern+"】 出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	
	@Override
	public boolean setExpire(Object key, String value, int time) {
		return setExpire(key,value,time,default_db);
	}
	@Override
	public boolean setExpire(Object key, String value, int time, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
            	if(index!=default_db)
            		jedis.select(index);
                jedis.set((String)key, value);
                jedis.expire((String)key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	@Override
	public <T> boolean setPoJoExpire(Object key, Object pojo, int time) {
		return setPoJoExpire(key,pojo,time,default_db);
	}
	@Override
	public <T> boolean setPoJoExpire(Object key, Object pojo, int time, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
             	if(index!=default_db)
               		jedis.select(index);
            	String json=new Gson().toJson(pojo);
                jedis.set((String)key, json);
                jedis.expire((String)key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	@Override
	public <T> boolean setlistExpire(Object key, List<T> list, int time) {
		return  setPoJoExpire(key,list,time,default_db);
	}
	@Override
	public <T> boolean setlistExpire(Object key, List<T> list, int time, int index) {
		Jedis jedis = getJedis();
		try {
            if(jedis != null){
            	if(index!=default_db)
               		jedis.select(index);
        		String json=new Gson().toJson(list);
                jedis.set((String)key, json);
                jedis.expire((String)key, time);
            }
            return true;
        } catch (Exception e) {
            log.error("Redis db"+index+"缓存设置key值 【"+key+"】 出错！", e);
            return false;
        } finally {
            closeResource(jedis);
        }
	}
	@Override
	public Long ttl(Object key) {
		return ttl(key,default_db);
	}
	@Override
	public Long ttl(Object key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return jedis.ttl((String)key) ; 
            } else {
                return 0L;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"获取key值 【"+key+"】剩余生存时间出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Long zadd(String key, double score, String member) {
		return zadd(key,score,member,default_db);
	}
	@Override
	public Long zadd(String key, double score, String member, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.zadd(key, score, member); 
            } else {
                return 0L;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"获取key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return 0L;
        } finally {
        	closeResource(jedis);
        }
		
	}
	@Override
	public Set<String> zrevrange(String key, int start, int end) {
		return zrevrange(key,start,end,default_db);
	}
	@Override
	public Set<String> zrevrange(String key, int start, int end, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.zrevrange(key, start, end); 
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"获取key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Set<String> zrange(String key, int start, int end) {
		return zrange(key,start,end,default_db);
	}
	@Override
	public Set<String> zrange(String key, int start, int end, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.zrange(key, start, end); 
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"获取key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Double zscore(String key, String member) {
		return zscore(key,member,default_db);
	}
	@Override
	public Double zscore(String key, String member, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.zscore(key, member); 
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"获取key值 【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Long zrem(String key, String member) {
		return zrem(key,member,default_db);
	}
	@Override
	public Long zrem(String key, String member, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.zrem(key, member); 
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"删除key值 【"+key+"】,member值【"+member+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Long zcard(String key) {
		return zcard(key,default_db);
	}
	@Override
	public Long zcard(String key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.zcard(key);
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"scard【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}

	@Override
	public Long incr(String key) {
		return incr(key,default_db);
	}
	
	@Override
	public Long incr(String key, int index) {
		Jedis jedis = getJedis();
		try {
			if (jedis != null) {
				if(index!=default_db)
					jedis.select(index);
				return jedis.incr(key);
			} else {
				return null;
			}
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		} finally {
			closeResource(jedis);
		}
	}
	@Override
	public Long hset(String key, String field, String value) {
		return hset(key,field,value,default_db);
	}
	@Override
	public Long hset(String key, String field, String value, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.hset(key, field, value);
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"hset【"+key+"】,field【"+field+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public String hget(String key, String field) {
		return hget(key,field,default_db);
	}
	@Override
	public String hget(String key, String field, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.hget(key, field);
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"hget【"+key+"】,field【"+field+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Long hdel(String key, String field) {
		return hdel(key,field,default_db);
	}
	@Override
	public Long hdel(String key, String field, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.hdel(key, field);
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"hdel【"+key+"】,field【"+field+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	@Override
	public Map<String, String> hgetall(String key) {
		return hgetall(key,default_db);
	}
	@Override
	public Map<String, String> hgetall(String key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
            	return  jedis.hgetAll(key);
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"hgetall【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}
	
	
	@Override
	public List<String> scan(String key) {
		return scan(key,default_db);
	}
	@Override
	public List<String> scan(String key, int index) {
		Jedis jedis = getJedis();
        try {
            if (jedis != null) {
              	if(index!=default_db)
               		jedis.select(index);
              	ScanParams scanParams = new ScanParams();
                scanParams.match(key+"*");
                scanParams.count(1000);
              	ScanResult<String> result=jedis.scan("0", scanParams);
            	return result.getResult();
            } else {
                return null;
            }
        } catch (Exception e) {
         	log.error("Redis db"+index+"scan【"+key+"】出错！", e);
            e.printStackTrace();
            return null;
        } finally {
        	closeResource(jedis);
        }
	}


}
