package com.ovopark.dc.dynamicroute;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.cloud.endpoint.event.RefreshEvent;
import org.springframework.cloud.netflix.zuul.RoutesRefreshedEvent;
import org.springframework.cloud.netflix.zuul.filters.RouteLocator;
import org.springframework.cloud.netflix.zuul.filters.ZuulProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationContextException;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.lang.NonNull;

/**
 * Dynamic Route Listener .
 *
 * @author <a href="mailto:guyue375@outlook.com">Klaus.turbo</a>
 * @program dc-avengers-ingress
 **/
@Slf4j
public class DynamicRouteListener implements ApplicationListener<RefreshEvent>, ApplicationContextAware {
    
    private final RouteLocator routeLocator;
    
    private ApplicationContext context;
    
    /**
     * Constructor with parameter instance of {@link RouteLocator}
     *
     * @param routeLocator A map of route path (pattern) to location (e.g. service id or URL).
     */
    public DynamicRouteListener(RouteLocator routeLocator) {
        this.routeLocator = routeLocator;
    }
    
    /**
     * Set the ApplicationContext that this object runs in. Normally this call will be used to initialize the object.
     *
     * <p>Invoked after population of normal bean properties but before an init callback such as
     * {@link InitializingBean#afterPropertiesSet()} or a custom init-method. Invoked after {@link
     * ResourceLoaderAware#setResourceLoader}, {@link ApplicationEventPublisherAware#setApplicationEventPublisher} and
     * {@link MessageSourceAware}, if applicable.
     *
     * @param context the ApplicationContext object to be used by this object
     * @throws {@link         ApplicationContextException} in case of context initialization errors
     * @throws BeansException if thrown by application context methods
     */
    @Override
    public void setApplicationContext(ApplicationContext context) throws BeansException {
        this.context = context;
    }
    
    @Override
    public void onApplicationEvent(@NonNull RefreshEvent refreshEvent) {
        log.warn("[RefreshEvent] Listens to the refresh event and prepares to refresh the application context ...");
        this.context.publishEvent(new RoutesRefreshedEvent(routeLocator));
    }
}
