package com.ovopark.training.enhancer.filter;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;
import com.ovopark.training.enhancer.utils.EhContextUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.core.annotation.Order;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Map;

@Slf4j
@Order(110)
public class EhLogFilter extends OncePerRequestFilter {

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        String url = request.getRequestURL().toString();
        String method = request.getMethod();
        String uri = request.getRequestURI();
        long startTime = System.currentTimeMillis();
        String params = MDC.get("params");

        StringBuilder sb = new StringBuilder();
        buildOne(sb, "url", url);
        buildOne(sb, "method", method);
        buildOne(sb, "uri", uri);
        buildOne(sb, "param", params);
        buildOne(sb, "userAgent", request.getHeader("User-Agent"));
        buildOne(sb, "enterpriseId", EhContextUtil.getEnterpriseId());
        buildOne(sb, "userId", EhContextUtil.getUserId());
        String invokeInfo = sb.toString();
        log.info(invokeInfo);
        Map<String, String> headerMap = extractHeader(request);
        log.info("header:{}", JSON.toJSONString(headerMap));
        Map<String, String> cookieMap = buildCookieMap(request);
        log.info("cookie:{}", JSON.toJSONString(cookieMap));

        filterChain.doFilter(request, response);

        long cost = System.currentTimeMillis() - startTime;
        log.info("接口耗时 cost {}", cost);
    }


    private void buildOne(StringBuilder sb, String key, Object value) {
        sb.append("\n[").append(key).append(":").append(value).append("]");
    }

    private static Map<String, String> buildCookieMap(HttpServletRequest httpServletRequest) {
        Map<String, String> cookieMap = Maps.newHashMap();
        Cookie[] cookies = httpServletRequest.getCookies();
        if (null != cookies) {
            int length = cookies.length;

            for (int i = 0; i < length; ++i) {
                Cookie cookie = cookies[i];
                if (!StringUtils.isEmpty(cookie.getValue())) {
                    cookieMap.put(cookie.getName(), cookie.getValue());
                } else {
                    cookieMap.put(cookie.getName(), "");
                }
            }
        }
        return cookieMap;
    }

    public static Map<String, String> extractHeader(HttpServletRequest request) {
        Map<String, String> headerMap = Maps.newHashMap();
        Enumeration headerNames = request.getHeaderNames();

        while (headerNames.hasMoreElements()) {
            String headerName = (String) headerNames.nextElement();
            String headerValue = request.getHeader(headerName);
            if (!StringUtils.isEmpty(headerValue)) {
                headerMap.put(headerName, headerValue);
            }
        }

        return headerMap;
    }
}
