package com.ovopark.training.enhancer.subject.gracefulshutdown;

import com.ovopark.training.enhancer.subject.gracefulshutdown.shutdown.AutoServiceRegistrationShutdown;
import com.ovopark.training.enhancer.subject.gracefulshutdown.shutdown.KafkaShutdown;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.client.actuator.HasFeatures;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

@Configuration
@ConditionalOnProperty(
    prefix = "endpoints.shutdown.graceful",
    name = {"enabled"},
    havingValue = "true",
    matchIfMissing = true
)
@Slf4j
@EnableConfigurationProperties({GracefulShutdownProperties.class})
public class GracefulShutdownAutoConfiguration {

    @Autowired
    private GracefulShutdownProperties gracefulShutdownProperties;

    public GracefulShutdownAutoConfiguration() {
    }

    @Bean
    public HasFeatures gracefulShutdownFeatures() {
        return HasFeatures.namedFeature("graceful shutdown", GracefulShutdownEndpoint.class);
    }

    @Bean
    public GracefulHealth gracefulHealth() {
        return new GracefulHealth();
    }

    @Bean
    @ConditionalOnMissingBean
    protected GracefulShutdownEndpoint gracefulShutdownEndpoint() {
        return new GracefulShutdownEndpoint();
    }

    @Bean
    @ConditionalOnClass(
        name = {"org.springframework.cloud.client.serviceregistry.AutoServiceRegistration"}
    )
    public AutoServiceRegistrationShutdown autoServiceRegistrationShutdown() {
        return new AutoServiceRegistrationShutdown();
    }

    @Bean
    @ConditionalOnClass(
        name = {"org.springframework.kafka.config.KafkaListenerEndpointRegistry"}
    )
    public KafkaShutdown kafkaShutdown() {
        return new KafkaShutdown();
    }

    @Bean
    public ShutdownManager shutdownManager(List<Shutdown> shutdowns) {
        shutdowns = shutdowns.stream()
                .sorted(Comparator.comparing(Ordered::getOrder))
                .peek(shutdown -> log.info("load shutdown: {}", shutdown.getClass()))
                .collect(Collectors.toList());
        return new ShutdownManager(shutdowns);
    }

}
