package com.ovopark.eventhub.sdk;

import com.ovopark.eventhub.sdk.model.OpeLogPut;
import com.ovopark.eventhub.sdk.model.PlayLogPut;
import com.ovopark.kernel.shared.delay.DelayTask;
import com.ovopark.kernel.shared.delay.TimingWheel;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;

import static com.ovopark.kernel.shared.Util.catchRunnable;
import static com.ovopark.kernel.shared.Util.defaultExecutorService;

public class SimpleDeviceLogEventAsyncService implements DeviceLogEventService{


    @Autowired
    private DeviceLogApi deviceLogApi;

    static final TimingWheel timingWheel=new TimingWheel(SimpleDeviceLogEventAsyncService.class.getName());

    static final ExecutorService executorService= defaultExecutorService("device-log-push"
            ,16,10_0000,new ThreadPoolExecutor.DiscardPolicy());

    @Override
    public void playLogPut(PlayLogPut playLogPut) {
        executorService.execute(catchRunnable(() -> {
            try {
                deviceLogApi.playLogPut(playLogPut);
            } catch (Exception e) {
                //retry later ...
                long next=System.currentTimeMillis()+3*60*1000;
                timingWheel.delay(new DelayTask() {
                    @Override
                    public long triggerTimeMs() {
                        return next;
                    }

                    @Override
                    public Comparable taskId() {
                        return playLogPut.getTime();
                    }

                    @Override
                    public Runnable task() {
                        return () -> executorService.execute(catchRunnable(() -> deviceLogApi.playLogPut(playLogPut)));
                    }
                });
            }
        }));
    }

    @Override
    public void opeLogPut(OpeLogPut opeLogPut) {
        executorService.execute(catchRunnable(() -> {
            try {
                deviceLogApi.opeLogPut(opeLogPut);
            } catch (Exception e) {
                //retry later ...
                long next=System.currentTimeMillis()+3*60*1000;
                timingWheel.delay(new DelayTask() {
                    @Override
                    public long triggerTimeMs() {
                        return next;
                    }

                    @Override
                    public Comparable taskId() {
                        return opeLogPut.getTime();
                    }

                    @Override
                    public Runnable task() {
                        return () -> executorService.execute(catchRunnable(() -> deviceLogApi.opeLogPut(opeLogPut)));
                    }
                });
            }
        }));
    }
}
